package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.utility.constants.UtilityAgentProperties;
import org.lsst.ccs.subsystem.utility.data.UtilityState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Implementation of auxiliary test system
 *
 * @author The LSST CCS Team
 */
public class AuxTest implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private FanControl fanControl;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    private static final Logger LOG = Logger.getLogger(AuxTest.class.getName());
    private int ctrlState = UtilityState.OPS_CTRL_OFF;
    private double coldTemp;
    private double dutyCycle;
    

    /**
     *  Post initialization
     */
    @Override
    public void postInit() {
        subsys.setAgentProperty(UtilityAgentProperties.AUXTEST_TYPE, AuxTest.class.getCanonicalName());
        if (fanControl != null) {
            fanControl.initialize(LOG);
        }
        else {
            LOG.warn("No fan control available");
        }
    }


    /**
     *  Post start
     */
    @Override
    public void postStart() {
        LOG.info("AuxTest subsystem started");
    }


    /**
     *  Gets the state of the AuxTest module.
     *
     *  @return  The full Raft state
     */
    public UtilityState getState() {
        return new UtilityState(getTickPeriod(), ctrlState, coldTemp, dutyCycle,
                                fanControl == null ? 0.0 : fanControl.getGain(),
                                fanControl == null ? 0.0 : fanControl.getTimeConstant());
    }    


    /**
     *  Gets the full state of the AuxTest module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The full Raft state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the AuxTest system state")
    public UtilityState getSystemState()
    {
        return getState();
    }    


    /**
     *  Sets the update period.
     *
     *  @param  value  The update period (milliseconds) to set.
     */
    @Command(type=Command.CommandType.ACTION, description="Set the update interval")
    public void setUpdatePeriod(@Argument(description="The tick period (ms)") int value)
    {
        setTickPeriod(value);
        publishState();
    }


   /**
    *  Sets the fan control state.
    *
    *  @param  state  The state to set
    */
    @Command(type=Command.CommandType.ACTION, description="Set the fan control state")
    public void setControlState(@Argument(description="The control state") int state)
    {
        if (state == ctrlState) return;
        if (state == UtilityState.OPS_CTRL_AUTO) {
            if (fanControl != null) {
                fanControl.setTemperature(coldTemp);
                fanControl.startLoop(dutyCycle);
            }
        }
        else {
            if (fanControl != null) {
                fanControl.stopLoop();
                if (state == UtilityState.OPS_CTRL_MANUAL) {
                    fanControl.setDutyCycle(dutyCycle);
                }
            }
        }
        ctrlState = state;
        publishState();
    }


   /**
    *  Sets the manual fan "speed".
    *
    *  @param  duty  The PWM duty cycle
    */
    @Command(type=Command.CommandType.ACTION, description="Set the manual fan speed")
    public void setFanSpeed(@Argument(description="The duty cycle") double duty)
    {
        dutyCycle = duty;
        if (fanControl != null && ctrlState == UtilityState.OPS_CTRL_MANUAL) {
            fanControl.setDutyCycle(dutyCycle);
        }
        publishState();
    }


   /**
    *  Sets the cold temperature.
    *
    *  @param  temp  The temperature to set
    */
    @Command(type=Command.CommandType.ACTION, description="Set the cold temperature")
    public void setColdTemp(@Argument(description="The cold temperature") double temp)
    {
        coldTemp = temp;
        if (fanControl != null && ctrlState == UtilityState.OPS_CTRL_AUTO) {
            fanControl.setTemperature(coldTemp);
        }
        publishState();
    }


   /**
    *  Sets the control loop gain.
    *
    *  @param  gain  The gain to set
    */
    @Command(type=Command.CommandType.ACTION, description="Set the control loop gain")
    public void setGain(@Argument(description="The loop gain") double gain)
    {
        if (fanControl != null) {
            fanControl.setGain(gain);
        }
        publishState();
    }


   /**
    *  Sets the control loop time constant.
    *
    *  @param  time  The time constant to set
    */
    @Command(type=Command.CommandType.ACTION, description="Set the control loop time constant")
    public void setTimeConstant(@Argument(description="The loop time constant") double time)
    {
        if (fanControl != null) {
            fanControl.setTimeConstant(time);
        }
        publishState();
    }


    /**
     *  Publishes the state of the AuxTest module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(UtilityState.KEY, getState()));
    }


    /**
     *  Sets the monitoring publishing period
     */
    private void setTickPeriod(long period)
    {
        pts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period
     */
    private int getTickPeriod()
    {
        return (int)pts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
