package org.lsst.ccs.subsystem.utility;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.common.devices.Maq20PWMControl;
import org.lsst.ccs.subsystem.common.PIController;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Implements a fan speed loop controller for the utility trunk system.
 *
 *  @author Owen Saxton
 */
public class FanControl implements HasLifecycle {

    private Double  gain;              // loop gain
    private Double  timeConst;         // integration time constant (secs)
    private Double  smoothTime;        // input smoothing time (secs)   
    private Double  maxOutput = 10.0;  // maximum PID output (inverse duty cycle)
    private Double  awGain;            // anti-windup gain
    private Double  baseDuty;          // base duty cycle input
    private Double  tolerance;         // maximum on-target error (%)
    private double  minOutput = 1.0;   // minimum PID output (inverse duty cycle)
    private double  maxInput = 100.0;  // maximum input
    private double  minInput = -200.0; // minimum input
    private Integer updateTime;        // The update time interval (msec)
    private String[] tempChans;        // Temperature channels to use
    private String[] fans;             // The fan controllers to use

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String,Channel> allChannels = new HashMap<>();
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private Map<String,Maq20PWMControl> allFans = new HashMap<>();
    
    private final List<Channel> tempChansL = new ArrayList<>();  // Temperature channels to use
    private final List<Maq20PWMControl> fansL = new ArrayList<>();     // Maq20 fan controllers to use

    private PIController pic;          // The PI controller
    private double lastDuty;           // The last duty cycle value set
    private boolean active;            // True if loop is active
    private double setTemp;            // Temperature set point
    private Logger log;                // The logger


    /**
     *  Sets up the fan speed control loop timer task.
     */
    @Override
    public void build() {
        pts.scheduleAgentPeriodicTask(new AgentPeriodicTask("fan-loop-timer", () -> {
            iterateLoop();
        }).withPeriod(Duration.ofMillis(updateTime)));
    }


    /**
     *  Initializes the parameters.
     * 
     *  @param  log  The subsystem logger
     */
    public void initialize(Logger log)
    {
        if (gain == null) {
            MonitorLogUtils.reportConfigError(log, name, "gain", "is missing");
        }
        if (timeConst == null) {
            MonitorLogUtils.reportConfigError(log, name, "timeConst", "is missing");
        }
        if (smoothTime == null) {
            MonitorLogUtils.reportConfigError(log, name, "smoothTime", "is missing");
        }
        if (maxOutput == null) {
            MonitorLogUtils.reportConfigError(log, name, "maxOutput", "is missing");
        }
        if (awGain == null) {
            MonitorLogUtils.reportConfigError(log, name, "awGain", "is missing");
        }
        if (baseDuty == null) {
            MonitorLogUtils.reportConfigError(log, name, "baseDuty", "is missing");
        }
        if (tolerance == null) {
            MonitorLogUtils.reportConfigError(log, name, "tolerance", "is missing");
        }
        if (updateTime == null) {
            MonitorLogUtils.reportConfigError(log, name, "updateTime", "is missing");
        }
        if (tempChans == null) {
            MonitorLogUtils.reportConfigError(log, name, "tempChans", "is missing");
        }
        else {
            for (String cName : tempChans) {
                Channel cmpt = allChannels.get(cName);
                if (cmpt != null) {
                    tempChansL.add(cmpt);
                }
                else {
                    MonitorLogUtils.reportConfigError(log, name, "tempChans", "contains non-Channel item");
                }
            }
        }
        if (fans == null) {
            MonitorLogUtils.reportConfigError(log, name, "fans", "is missing");
        }
        else {
            for (String cName : fans) {
                Maq20PWMControl cmpt = allFans.get(cName);
                if (cmpt != null) {
                    fansL.add(cmpt);
                }
                else {
                    MonitorLogUtils.reportConfigError(log, name, "rebs", "contains non-Maq20PWMControl item");
                }
            }
        }
        this.log = log;
        pic = new PIController(gain, timeConst);
        pic.setSmoothTime(smoothTime);
        pic.setAwGain(awGain);
        pic.setBaseOutput(baseDuty);
        pic.setInputRange(minInput, maxInput);
        pic.setOutputRange(minOutput, maxOutput);
        pic.setTolerance(tolerance);
    }


    /**
     *  Sets the fan speed (PWM duty cycle).
     *
     *  @param  duty  The duty cycle to set
     */
    void setDutyCycle(double duty)
    {
        for (Maq20PWMControl fan : fansL) {
            try {
                fan.setDutyCycle1(duty);
            }
            catch (Exception e) {
                log.error("Error setting " + fan.getName() + " fan speed: " + e.getMessage());
            }
        }
    }


    /**
     *  Sets the target temperature.
     *
     *  @param  temp  The temperature to set
     */
    public void setTemperature(double temp)
    {
        setTemp = temp;
        pic.setSetpoint(setTemp);
    }


    /**
     *  Gets the target temperature.
     *
     *  @return  The set temperature
     */
    public double getTemperature()
    {
        return setTemp;
   }


    /**
     *  Sets the gain.
     *
     *  @param  gain  The gain to set
     */
    public void setGain(double gain)
    {
        this.gain = gain;
        if (pic != null) {  // Can get called during startup before pic exists
            pic.setPID(gain, timeConst);
        }
    }


    /**
     *  Gets the gain.
     *
     *  @return  The gain
     */
    public double getGain()
    {
        return gain;
   }


    /**
     *  Sets the time constant.
     *
     *  @param  time  The time constant to set
     */
    public void setTimeConstant(double time)
    {
        timeConst = time;
        pic.setPID(gain, timeConst);
    }


    /**
     *  Gets the time constant.
     *
     *  @return  The time constant
     */
    public double getTimeConstant()
    {
        return timeConst;
   }


    /**
     *  Starts the control loop.
     *
     *  @param  duty  The initial duty cycle value
     */
    public void startLoop(double duty)
    {
        if (active) return;
        lastDuty = 1.0 / duty;
        startLoop();
    }


    /**
     *  Starts the control loop.
     */
    public void startLoop()
    {
        if (active) return;
        pic.reset();
        pic.setIntegral(lastDuty - baseDuty);
        active = true;
    }


    /**
     *  Stops the control loop.
     */
    public void stopLoop()
    {
        if (!active) return;
        active = false;
        setDutyCycle(1.0);  // Maximum fan speed
    }


    /**
     *  Gets the control loop state.
     *
     *  @return  Whether the control loop is active
     */
    public boolean isLoopActive()
    {
        return active;
    }


    /**
     *  Resets the controller.
     */
    public void reset()
    {
        pic.reset();
    }


    /**
     *  Timer routine for control loop iteration.
     */
    private void iterateLoop()
    {
        if (!active) return;
        int count = 0;
        double temp = 0.0;
        for (Channel tempChan : tempChansL) {
            double value = tempChan.getValue();
            if (!Double.isNaN(value)) {
                temp += value;
                count++;
            }
        }
        if (count > 0) {
            double tod = (double)System.currentTimeMillis() / 1000;
            lastDuty = pic.performPI(new double[]{temp / count}, tod);
            setDutyCycle(1.0 / lastDuty);
            //System.out.format("Temp: %s, Inv. duty factor: %s, Set point: %s, Error: %s, Integral: %s\n",
            //                  temp / count, lastDuty, pic.getSetpoint(), pic.getError(), pic.getIntegral());
        }
        else {
            log.error("Control loop iteration failed: no valid temperature values available");
        }
    }

}
