package org.lsst.ccs.subsystem.utility;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.abb.Cms;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Lambda CPFE1000 power supply system.
 *
 *  @author Owen Saxton
 */
public class CmsDevice extends Device {

   /**
    *  Constants.
    */
    public static final int
        TYPE_RMS = 0,
        TYPE_AC  = 1,
        TYPE_DC  = 2;

   /**
    *  Private lookup maps.
    */
    private static final Map<String, Cms.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", Cms.ConnType.SERIAL);
        cTypeMap.put("FTDI", Cms.ConnType.FTDI);
    }

    private static final Map<String, Integer> sTypeMap = new HashMap<>();
    static {
        sTypeMap.put("RMS", TYPE_RMS);
        sTypeMap.put("AC", TYPE_AC);
        sTypeMap.put("DC", TYPE_DC);
    }

   /**
    *  Data fields.
    */
    private String connType;     // Connection type string
    private String devcId;       // Device identification
    private int baudRate = 0;    // Baud rate (default)
    private int address = 1;     // Modbus address

    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final Cms cms = new Cms();  // Associated CMS object
    private Cms.ConnType cType;


   /**
    *  Performs basic initialization.
    */
    @Override
    protected void initDevice()
    {
        if (connType == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "connType", "is missing");
        }
        cType = cTypeMap.get(connType.toUpperCase());
        if (cType == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "connType", "is invalid");
        }
        if (devcId == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "devcId", "is missing");
        }
        fullName = "CMS100 (" + devcId + ":" + address + ")";
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        try {
            cms.open(cType, devcId, baudRate);
            setOnline(true);
            initSensors();
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        try {
            cms.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan <= 0 || hwChan > Cms.NUM_SENSORS) {
            MonitorLogUtils.reportError(sLog, name, "hwchan", hwChan);
        }
        Integer sType = sTypeMap.get((type == null || type.isEmpty()) ? "RMS" : type.toUpperCase());
        if (sType == null) {
            MonitorLogUtils.reportError(sLog, name, "type", type);
        }
        return new int[]{sType, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        try {
            switch (type) {
                
            case TYPE_RMS:
                value = cms.readRMS(address, hwChan); break;
                
            case TYPE_AC:
                value = cms.readAC(address, hwChan); break;
                
            case TYPE_DC:
                value = cms.readDC(address, hwChan); break;
            }
        }
        catch (DriverException e) {
            sLog.error("Error reading CMS100 data: " + e);
            setOnline(false);
        }

        return value;
    }

}
