package org.lsst.ccs.subsystem.utility;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pluto.Pluto;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Pluto PLC system.
 *
 *  @author Owen Saxton
 */
public class PlutoDevice extends Device {

   /**
    *  Constants.
    */
    private static final int
        TYPE_GLOBAL  = 0,
        TYPE_ADD_BIT = 1,
        TYPE_ADD_REG = 2;

   /**
    *  Private lookup maps.
    */
    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("GLOBAL", TYPE_GLOBAL);
        typeMap.put("ADDBIT", TYPE_ADD_BIT);
        typeMap.put("ADDREG", TYPE_ADD_REG);
    }

   /**
    *  Data fields.
    */
    private String node;          // Network node name
    private List<Integer> areas;  // Area configuration data

    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private int nArea;                // Area count
    private final Pluto plu = new Pluto();  // Associated Pluto object
    private final int[] globData = new int[Pluto.NUM_MODULES];
    private final int[] addData = new int[Pluto.NUM_ADD_AREAS];
    private final Set<Integer> modsUsed = new HashSet<>();
    private final Set<Integer> areasUsed = new HashSet<>();


   /**
    *  Performs basic initialization.
    */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "node", "is missing");
        }
        if (areas == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "areas", "is missing");
        }
        if ((areas.size() & 1) != 0) {
            MonitorLogUtils.reportConfigError(sLog, name, "areas", "has odd number of elements");
        }
        nArea = areas.size() / 2;
        fullName = "Pluto PLC system (" + node + ")";
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize()
    {
        try {
            plu.open(node);
            plu.configStart(-1, 15, 0, 100);
            for (int j = 0; j < areas.size(); j += 2) {
                plu.configDataArea(j / 2, areas.get(j), areas.get(j + 1));
            }
            plu.configWrite();
            setOnline(true);
            initSensors();
            setOutputLines();
            log.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close()
    {
        try {
            plu.close();
        }
        catch (DriverException e) {
        }
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype)
        throws Exception
    {
        Integer iType = null;
        int id;
        String[] typeFields = type.split(":", -1);
        if (typeFields.length == 2) {
            iType = typeMap.get(typeFields[0].toUpperCase());
        }
        try {
            id = Integer.decode(typeFields[1]);
        }
        catch (NumberFormatException e) {
            id = -1;
        }
        int maxId = 0;
        if (iType != null) {
            maxId = iType == TYPE_GLOBAL ? Pluto.NUM_MODULES : nArea;
            int maxChan = iType == TYPE_GLOBAL ? 32 : iType == TYPE_ADD_BIT ? 16 : 2;
            if (hwChan < 0 || hwChan >= maxChan) {
                MonitorLogUtils.reportError(sLog, name, "hwchan", hwChan);
            }
        }
        if (iType == null || id < 0 || id >= maxId) {
            MonitorLogUtils.reportError(sLog, name, "type", type);
        }
        Set used = iType == TYPE_GLOBAL ? modsUsed : areasUsed;
        used.add(id);

        return new int[]{(iType << 8) | id, 0};
    }


   /**
    *  Reads all referenced channels.
    */
    @Override
    protected void readChannelGroup()
    {
        if (!online) return;
        try {
            for (int id : modsUsed) {
                globData[id] = plu.readGlobalData(id);
            }
            for (int id : areasUsed) {
                addData[id] = plu.readAdditionalData(id);
            }
        }
        catch (DriverException e) {
            log.error("Error reading " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        if (online) {
            int index = type & 0xff;
            switch (type >> 8) {

            case TYPE_GLOBAL:
                value = (globData[index] >> hwChan) & 1;
                break;

            case TYPE_ADD_BIT:
                value = (addData[index] >> hwChan) & 1;
                break;

            case TYPE_ADD_REG:
                value = (addData[index] << (16 * (1 - hwChan))) >> 16;  // Sign extend
                break;
            }
        }

        return value;
    }


   /**
    *  Checks an output line number.
    *
    *  @param  name  The name of the output line.
    *  @param  line  The hardware line number of the output line.
    *  @throws  Exception  If line number is invalid
    */
    @Override
    protected void checkHwLine(String name, int line) throws Exception
    {
        if (line < 0 || line >= 16 * Pluto.NUM_DTP_AREAS) {
            MonitorLogUtils.reportError(sLog, name, "line", line);
        }
        addLine(line);
    }


   /**
    *  Sets an output line on or off.
    *
    *  @param  line  The output line number.
    *  @param  on    The on state to set: true or false
    */
    @Override
    protected void setHwLine(int line, boolean on)
    {
        writeBit(line / 16, line & 0x0f, on ? 1 : 0);
    }


   /**
    *  Gets the set state of an output line.
    *
    *  @param  line  The output line number.
    *  @return  Whether the line is on
    */
    @Override
    protected Boolean isHwLineSet(int line)
    {
        return readBit(line / 16, line & 0x0f) != 0;
    }


   /**
    *  Writes a bit value.
    *
    *  @param  area   The area number
    *  @param  bit    The bit number
    *  @param  value  The bit value
    */
    public void writeBit(int area, int bit, int value)
    {
        try {
            plu.writeAreaBit(area, bit, value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }


   /**
    *  Reads a bit value.
    *
    *  @param  area   The area number
    *  @param  bit    The bit number
    *  @return  The bit value
    */
    public int readBit(int area, int bit)
    {
        try {
            return plu.readAreaBit(area, bit);
        }
        catch (DriverException e) {
            log.error("Error reading from " + fullName + ": " + e);
            setOnline(false);
            return 0;
        }
    }


   /**
    *  Writes a register value.
    *
    *  @param  area   The area number
    *  @param  reg    The register number
    *  @param  value  The register value
    */
    public void writeRegister(int area, int reg, int value)
    {
        try {
            plu.writeAreaRegister(area, reg, value);
        }
        catch (DriverException e) {
            log.error("Error writing to " + fullName + ": " + e);
            setOnline(false);
        }
    }

}
