package org.lsst.ccs.subsystem.utility;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.pluto.Pluto;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Pluto PLC system control channel.
 *
 *  @author Owen Saxton
 */
public class PlutoControl extends Control {

   /**
    *  Constants.
    */
    private final static int
        TYPE_BIT  = 0,
        TYPE_REG  = 1;

   /**
    *  Private lookup maps.
    */
    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("BIT", TYPE_BIT);
        typeMap.put("REG", TYPE_REG);
    }

   /**
    *  Data fields.
    */
    private String type;
    private double offset = 0.0;
    private double scale = 1.0;

    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final PlutoDevice pDevc = (PlutoDevice)devc;
    private int iType, area;


   /**
    *  Performs configuration.
    *
    *  @param  mon   The associated monitor
    *  @param  devc  The associated device
    */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        if (type == null) {
            MonitorLogUtils.reportConfigError(sLog, getName(), "type", "not specified");
        }
        Integer iTyp = null;
        String[] typeFields = type.split(":", -1);
        if (typeFields.length == 2) {
            iTyp = typeMap.get(typeFields[0].toUpperCase());
        }
        try {
            area = Integer.decode(typeFields[1]);
        }
        catch (NumberFormatException e) {
            area = -1;
        }
        if (iTyp == null || area < 0 || area >= Pluto.NUM_DTP_AREAS) {
            MonitorLogUtils.reportConfigError(sLog, getName(), "type", "has invalid value");
        }
        iType = iTyp;
        int maxChan = iType == TYPE_BIT ? 16 : 2;
        if (hwChan >= maxChan) {
            MonitorLogUtils.reportConfigError(sLog, getName(), "hwchan", "has invalid value");
        }
    }


   /**
    *  Writes a value.
    *
    *  @param  value  The value
    */
    public void writeValue(double value)
    {
        int ival = (int)((value - offset) / scale);
        if (iType == TYPE_BIT) {
            pDevc.writeBit(area, hwChan, ival);
        }
        else {
            pDevc.writeRegister(area, hwChan, ival);
        }
    }

}
