package org.lsst.ccs.subsystem.utility;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20DiscretePulse;
import org.lsst.ccs.monitor.Control;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Handles a Dataforth Maq20 system pulse control channel.
 *
 *  @author Owen Saxton
 */
public class Maq20PulseControl extends Control implements Maq20Device.Maq20DiscControl {

    /**
     *  Data fields.
     */
    private String type;

    @ConfigurationParameter
    private int pulsesPerRevn = 1;

    private final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private Maq20Device.ModuleData modData;
    private Maq20DiscretePulse pulse;
    private int funcNum;
    private boolean inited = false;


    /**
     *  Performs configuration.
     *
     *  @param  mon   The associated monitor
     *  @param  devc  The associated device
     */
    @Override
    protected void configure(Monitor mon, Device devc)
    {
        super.configure(mon, devc);
        if (type == null) {
            MonitorLogUtils.reportConfigError(sLog, getName(), "type", "not specified");
        }
        String[] words = type.split(":");
        if (words.length != 2) {
            MonitorLogUtils.reportConfigError(sLog, getName(), "type", "doesn't have exactly two elements");
        }
        try {
            int modIx = Integer.decode(words[0]);
            modData = ((Maq20Device)devc).getModuleData(modIx);
            if (modData == null) {
                MonitorLogUtils.reportConfigError(sLog, getName(), "module", "doesn't exist");
            }
        }
        catch (NumberFormatException ex) {
            MonitorLogUtils.reportConfigError(sLog, getName(), "module index", "is not numeric");
        }
        if (!words[1].equals("0") && !words[1].equals("1")) {
            MonitorLogUtils.reportConfigError(sLog, getName(), "function number", "is not 0 or 1");
        }
        funcNum = Integer.decode(words[1]);
    }


    /**
     *  Performs initialization.
     *
     *  @throws DriverException
     */
    @Override
    public void initialize() throws DriverException
    {
        pulse = modData.maqDiscPulse;
        inited = true;
        checkPulse();
        pulse.enable(funcNum);
        setPulsesPerRevn(pulsesPerRevn);
    }


    /**
     *  Setter for the pulses/revolution.
     *
     *  @param  pulses  The number of pulses per revolution
     *  @throws DriverException
     */
    @ConfigurationParameterChanger
    public void setPulsesPerRevn(int pulses) throws DriverException
    {
        pulsesPerRevn = pulses;
        if (inited) {
            checkPulse();
            pulse.setPulsesPerRevn(funcNum, pulsesPerRevn);
            pulse.armFunction(funcNum);
        }
    }


    /**
     *  Getter for pulses/revolution.
     *
     *  @return  The number of pulses/revn
     */
    @Command(description = "Get number of pulses per revolution")
    public int getPulsesPerRevn()
    {
        return pulsesPerRevn;
    }


    /**
     *  Checks whether pulse operations are accessible.
     *
     *  @throws DriverException
     */
    private void checkPulse() throws DriverException
    {
        if (pulse == null) {
            throw new DriverException("Cannot access pulse function");
        }
    }

}
