package org.lsst.ccs.subsystem.utility.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.utility.constants.ConditionState;
import org.lsst.ccs.subsystem.utility.constants.LatchState;
import org.lsst.ccs.subsystem.utility.constants.MonitorControl;
import org.lsst.ccs.subsystem.utility.constants.MpmConditions;
import org.lsst.ccs.subsystem.utility.constants.MpmLatches;
import org.lsst.ccs.subsystem.utility.constants.MpmLimits;
import org.lsst.ccs.subsystem.utility.constants.MpmPlcs;
import org.lsst.ccs.subsystem.utility.constants.MpmSwitches;
import org.lsst.ccs.subsystem.utility.constants.PLCState;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.data.MpmSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class MpmControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final String CMND_GET_STATE = "getSystemState";
    private static final String[] plcNames = new String[MpmPlcs.NUM_PLCS];
    static {
        plcNames[MpmPlcs.PLC_POWER] = "Power";
        plcNames[MpmPlcs.PLC_COLD]  = "Cold";
        plcNames[MpmPlcs.PLC_CRYO]  = "Cryo";
    }
    private static final String[] switchNames = new String[MpmSwitches.NUM_SWITCHES];
    static {
        switchNames[MpmSwitches.SW_BLOCK_COLD_HEAT] = "Block Cold Heater";
        switchNames[MpmSwitches.SW_BLOCK_COLD_REFG] = "Block Cold Refrig";
        switchNames[MpmSwitches.SW_BLOCK_CRYO_HEAT] = "Block Cryo Heater";
        switchNames[MpmSwitches.SW_BLOCK_CRYO_REFG] = "Block Cryo Refrig";
        switchNames[MpmSwitches.SW_BLOCK_COOLANT]   = "Block UT Coolant";
        switchNames[MpmSwitches.SW_BLOCK_REB_POWER] = "Block REB Power";
        switchNames[MpmSwitches.SW_BLOCK_UT_POWER]  = "Block UT Power";
    }
    private static final String[] latchNames = new String[MpmLatches.NUM_LATCHES];
    static {
        latchNames[MpmLatches.LATCH_COLD_TEMP_HIGH] = "Cold Temp > 99C";
        latchNames[MpmLatches.LATCH_COLD_TEMP_LOW]  = "Cold Temp < -99C";
        latchNames[MpmLatches.LATCH_CRYO_TEMP_HIGH] = "Cryo Temp > 99C";
        latchNames[MpmLatches.LATCH_CRYO_TEMP_LOW]  = "Cryo Temp < -999C";
        latchNames[MpmLatches.LATCH_CRYO_VACUUM]    = "Cryo Vacuum Bad";
        latchNames[MpmLatches.LATCH_HEX_VACUUM]     = "Hex Vacuum Bad";
        latchNames[MpmLatches.LATCH_UT_LEAK]        = "UT Leak";
        latchNames[MpmLatches.LATCH_UT_LEAK_FAULT]  = "UT Leak Fault";
        latchNames[MpmLatches.LATCH_UT_SMOKE]       = "UT Smoke";
        latchNames[MpmLatches.LATCH_UT_SMOKE_FAULT] = "UT Smoke Fault";
        latchNames[MpmLatches.LATCH_UT_TEMP]        = "UT Temp High";
    }
    private static final String[] conditionNames = new String[MpmConditions.NUM_CONDITIONS];
    static {
        conditionNames[MpmConditions.COND_UT_POWER]  = "UT Power Permit";
        conditionNames[MpmConditions.COND_REB_POWER] = "REB Power Permit";
        conditionNames[MpmConditions.COND_COOLANT]   = "UT Coolant Permit";
        conditionNames[MpmConditions.COND_COLD_REFG] = "Cold Refrig Permit";
        conditionNames[MpmConditions.COND_COLD_HEAT] = "Cold Heater Permit";
        conditionNames[MpmConditions.COND_CRYO_REFG] = "Cryo Refrig Permit";
        conditionNames[MpmConditions.COND_CRYO_HEAT] = "Cryo Heater Permit";
    }

    private SystemStatusPanel systemPanel;

    private JPanel plcPanel;
    private final JLabel[] plcStatus = new JLabel[MpmPlcs.NUM_PLCS];

    private JPanel switchPanel;
    private final JLabel[] switchStatus = new JLabel[MpmSwitches.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[MpmSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[MpmSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[MpmSwitches.NUM_SWITCHES];

    private JPanel latchPanel;
    private final JLabel[] latchLabel = new JLabel[MpmLatches.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[MpmLatches.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[MpmLatches.NUM_LATCHES];

    private JPanel conditionPanel;
    private final JLabel[] conditionStatus = new JLabel[MpmConditions.NUM_CONDITIONS];

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private boolean gotTempLimits = false;

    public MpmControlPanel(String subsys) {
        sender = new CommandSender(subsys, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((MpmSysState)reply);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    public void updatePanel(MpmSysState rs) {
        SwingUtilities.invokeLater(new UpdateMpmStatus(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // System status
        systemPanel = new SystemStatusPanel(sender, MonitorControl.NODE_NAME, true);

        // PLC states
        for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
            plcStatus[plc] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(PLCState.class));
        }

        // Switch states
        for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
            switchStatus[sw] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(SwitchState.class));
            switchOffRB[sw] = uiUtils.newRadioButton("Off", "F" + sw);
            switchOnRB[sw] = uiUtils.newRadioButton("On", "T" + sw);
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(switchOffRB[sw]);
            bg.add(switchOnRB[sw]);
        }

        // Latched condition states
        for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
            latchLabel[cond] = UiUtilities.newLabel(latchNames[cond] + ":", 0);
            latchStatus[cond] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(LatchState.class));
            latchResetBtn[cond] = uiUtils.newButton("Reset", String.valueOf(cond), true);
        }

        // Condition states
        for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
            conditionStatus[cond] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(ConditionState.class));
        }

        // Layout the PLC panel
        plcPanel = UiUtilities.newBorderedPanel("PLC States");
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.anchor = GridBagConstraints.WEST;
        gbp.gridx = 0;
        gbp.gridy = 0;
        gbp.insets = new Insets(4, 4, 4, 4);
        for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
            plcPanel.add(UiUtilities.newLabel(plcNames[plc] + ":", 0), gbp);
            gbp.gridx++;
            gbp.insets.left = 4;
            plcPanel.add(plcStatus[plc], gbp);
            gbp.gridx++;
            gbp.insets.left = 35;
        }

        // Layout the switch panel
        switchPanel = UiUtilities.newBorderedPanel("Switches");
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, 4, 4);
        gbs.gridy = -1;
        int numColm = 2;
        int colm = 0;
        for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = gbs.gridy == 0 ? 4 : 0;
            gbs.insets.left = colm == 0 ? 4 : 40;
            switchPanel.add(UiUtilities.newLabel(switchNames[sw] + ":", 0), gbs);
            gbs.insets.left = 4;
            gbs.gridx++;
            switchPanel.add(switchStatus[sw], gbs);
            gbs.gridx++;
            switchPanel.add(switchOffRB[sw], gbs);
            gbs.gridx++;
            switchPanel.add(switchOnRB[sw], gbs);
            gbs.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Layout the latched conditions panel
        latchPanel = UiUtilities.newBorderedPanel("PLC Error Conditions");
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.WEST;
        gbl.insets = new Insets(0, 0, 4, 4);
        gbl.insets.bottom = 4;
        gbl.gridy = -1;
        numColm = 2;
        colm = 0;
        for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
            if (colm == 0) {
                gbl.gridx = 0;
                gbl.gridy++;
            }
            gbl.insets.top = gbl.gridy == 0 ? 4 : 0;
            gbl.insets.left = colm == 0 ? 4 : 40;
            latchPanel.add(latchLabel[cond], gbl);
            gbl.insets.left = 4;
            gbl.gridx++;
            latchPanel.add(latchStatus[cond], gbl);
            gbl.gridx++;
            latchPanel.add(latchResetBtn[cond], gbl);
            gbl.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Layout the conditions panel
        conditionPanel = UiUtilities.newBorderedPanel("PLC Running Conditions");
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, 4, 4);
        gbc.gridy = -1;
        numColm = 2;
        colm = 0;
        for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
            if (colm == 0) {
                gbc.gridx = 0;
                gbc.gridy++;
            }
            gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
            gbc.insets.left = colm == 0 ? 4 : 35;
            conditionPanel.add( UiUtilities.newLabel(conditionNames[cond] + ":", 0), gbc);
            gbc.insets.left = 4;
            gbc.gridx++;
            conditionPanel.add(conditionStatus[cond], gbc);
            gbc.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.insets = new Insets(6, 0, 6, 0);
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(systemPanel, gbm);
        gbm.gridy++;
        add(plcPanel, gbm);
        gbm.gridy++;
        add(switchPanel, gbm);
        gbm.gridy++;
        add(latchPanel, gbm);
        gbm.gridy++;
        add(conditionPanel, gbm);

    }

    @Override
    public void handleRadioButton(String name) {
        char action = name.charAt(0);
        int sw = Integer.valueOf(name.substring(1));
        switchOffRB[sw].setEnabled(false);
        switchOnRB[sw].setEnabled(false);
        sender.sendCommand(null, "setNamedSwitchOn", MpmSwitches.getName(sw), action == 'T');
    }

    @Override
    public void handleButton(String name) {
        int latch = Integer.valueOf(name);
        latchResetBtn[latch].setEnabled(false);
        sender.sendCommand(null, "clearLatch", MpmLatches.getName(latch));
    }

    class UpdateMpmStatus implements Runnable {

        private final MpmSysState ps;

        UpdateMpmStatus(MpmSysState ps) {
            this.ps = ps;
        }

        @Override
        public void run() {
            systemPanel.updatePanel(ps.getTickMillis());

            for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
                PLCState state = ps.getPlcState(plc);
                plcStatus[plc].setText(state.name());
                plcStatus[plc].setForeground(state == PLCState.OFFLINE ? UiConstants.BLUE :
                                             state == PLCState.DEAD ? UiConstants.RED : UiConstants.GREEN);
                plcStatus[plc].setEnabled(true);
            }
            for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
                SwitchState state = ps.getSwitchState(sw);
                String text;
                Color color;
                text = state.name();
                color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? UiConstants.GREEN :
                        state == SwitchState.ALARM ? UiConstants.RED : UiConstants.BLUE;
                switchStatus[sw].setText(text);
                switchStatus[sw].setForeground(color);

                JRadioButton selButton = state == SwitchState.ON || state == SwitchState.ALARM ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
                LatchState state = ps.getLatch(cond);
                latchStatus[cond].setText(state.name());
                latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
                                                state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
                latchStatus[cond].setEnabled(true);
                latchResetBtn[cond].setEnabled(true);
            }
            for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
                ConditionState state = ps.getCondition(cond);
                conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
            }
            if (!gotTempLimits && ps.getLimit(0) != Integer.MAX_VALUE) {
                latchLabel[MpmLatches.LATCH_COLD_TEMP_HIGH].setText("Cold Temp > " + ps.getLimit(MpmLimits.LIMIT_COLD_HIGH) + "C:");
                latchLabel[MpmLatches.LATCH_COLD_TEMP_LOW].setText("Cold Temp < " + ps.getLimit(MpmLimits.LIMIT_COLD_LOW) + "C:");
                latchLabel[MpmLatches.LATCH_CRYO_TEMP_HIGH].setText("Cryo Temp > " + ps.getLimit(MpmLimits.LIMIT_CRYO_HIGH) + "C:");
                latchLabel[MpmLatches.LATCH_CRYO_TEMP_LOW].setText("Cryo Temp < " + ps.getLimit(MpmLimits.LIMIT_CRYO_LOW) + "C:");
                gotTempLimits = true;
            }
            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            systemPanel.disablePanel();
            for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
                plcStatus[plc].setEnabled(false);
            }
            for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
                latchStatus[cond].setEnabled(false);
                latchResetBtn[cond].setEnabled(false);
            }
            for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
                conditionStatus[cond].setEnabled(false);
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
