package org.lsst.ccs.subsystem.utility.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.utility.constants.FanState;
import org.lsst.ccs.subsystem.utility.constants.HeaterState;
import org.lsst.ccs.subsystem.utility.constants.MonitorControl;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkFans;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkHeaters;
import org.lsst.ccs.subsystem.utility.constants.UtilTrunkValves;
import org.lsst.ccs.subsystem.utility.constants.ValveState;
import org.lsst.ccs.subsystem.utility.constants.VpcControlState;
import org.lsst.ccs.subsystem.utility.data.UtilTrunkState;

/**
 *  Implements the utility trunk control panel.
 *
 *  @author Owen Saxton
 */
public class UtilTrunkControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final String CMND_GET_STATE = "getFullState";
    private static final String[] heaterNames = new String[UtilTrunkHeaters.NUM_HEATERS];
    static {
        heaterNames[UtilTrunkHeaters.HEATER_VPC1_ID] = "Heater1";
        heaterNames[UtilTrunkHeaters.HEATER_VPC2_ID] = "Heater2";
    }

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private SystemStatusPanel statusPanel;
    private JPanel mpcPanel, vpcPanel, utPanel;
    private final JLabel[] fanState = new JLabel[UtilTrunkFans.NUM_FANS];
    private final ButtonGroup[] fanBG = new ButtonGroup[UtilTrunkFans.NUM_FANS];
    private final JRadioButton[] fanOffRB = new JRadioButton[UtilTrunkFans.NUM_FANS];
    private final JRadioButton[] fanSpeedRB = new JRadioButton[UtilTrunkFans.NUM_FANS];
    private final TextFieldX[] fanSpeedTF = new TextFieldX[UtilTrunkFans.NUM_FANS];
    private final JRadioButton[] fanTempRB = new JRadioButton[UtilTrunkFans.NUM_FANS];
    private final TextFieldX[] fanTempTF = new TextFieldX[UtilTrunkFans.NUM_FANS];
    private final JLabel[] heaterState = new JLabel[UtilTrunkHeaters.NUM_HEATERS];
    private final ButtonGroup[] heaterBG = new ButtonGroup[UtilTrunkHeaters.NUM_HEATERS];
    private final JRadioButton[] heaterOffRB = new JRadioButton[UtilTrunkHeaters.NUM_HEATERS];
    private final JRadioButton[] heaterOnRB = new JRadioButton[UtilTrunkHeaters.NUM_HEATERS];
    private final TextFieldX[] heaterValueTF = new TextFieldX[UtilTrunkHeaters.NUM_HEATERS];
    private final JLabel[] valveState = new JLabel[UtilTrunkValves.NUM_VALVES];
    private final ButtonGroup[] valveBG = new ButtonGroup[UtilTrunkValves.NUM_VALVES];
    private final JRadioButton[] valveShutRB = new JRadioButton[UtilTrunkValves.NUM_VALVES];
    private final JRadioButton[] valveOpenRB = new JRadioButton[UtilTrunkValves.NUM_VALVES];
    private final TextFieldX[] valvePosnTF = new TextFieldX[UtilTrunkValves.NUM_VALVES];
    private JLabel vpcState;
    private ButtonGroup vpcBG;
    private JRadioButton vpcManualRB;
    private JRadioButton vpcTempRB;

    public UtilTrunkControlPanel(String subsys) {
        sender = new CommandSender(subsys, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, CMND_GET_STATE);
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((UtilTrunkState)reply);
    }

    @Override
    public void onCommandReject(String path, String command, Object[] args) {
        if (!command.equals(CMND_GET_STATE)) {
            sender.sendCommand(true, null, CMND_GET_STATE);
        }
    }

    public void updatePanel(UtilTrunkState rs) {
        SwingUtilities.invokeLater(new UpdateUtilState(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {

        // VPC state elements
        vpcState = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(VpcControlState.class));
        vpcManualRB = uiUtils.newRadioButton("Manual", "MF0");
        vpcTempRB = uiUtils.newRadioButton("Temp", "MT0");
        vpcBG = new ButtonGroup();
        vpcBG.add(vpcManualRB);
        vpcBG.add(vpcTempRB);

        // Fan control elements
        for (int fanId = 0; fanId < UtilTrunkFans.NUM_FANS; fanId++) {
            fanState[fanId] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(FanState.class));
            fanOffRB[fanId] = uiUtils.newRadioButton("Off", "FF" + fanId);
            fanSpeedRB[fanId] = uiUtils.newRadioButton("Speed", "FS" + fanId);
            fanTempRB[fanId] = uiUtils.newRadioButton("Temp", "FT" + fanId);
            ButtonGroup bg = fanBG[fanId] = new ButtonGroup();
            bg.add(fanOffRB[fanId]);
            bg.add(fanSpeedRB[fanId]);
            bg.add(fanTempRB[fanId]);
            fanSpeedTF[fanId] = uiUtils.newTextFieldX("0000.00", "S" + fanId, TextFieldX.TYPE_DOUBLE, true);
            fanTempTF[fanId] = uiUtils.newTextFieldX("0000.00", "T" + fanId, TextFieldX.TYPE_DOUBLE, true);
        }

        // Heater control elements
        for (int htrId = 0; htrId < UtilTrunkHeaters.NUM_HEATERS; htrId++) {
            heaterState[htrId] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(HeaterState.class));
            heaterOffRB[htrId] = uiUtils.newRadioButton("Off", "HF" + htrId);
            heaterOnRB[htrId] = uiUtils.newRadioButton("On", "HT" + htrId);
            ButtonGroup bg = heaterBG[htrId] = new ButtonGroup();
            bg.add(heaterOffRB[htrId]);
            bg.add(heaterOnRB[htrId]);
            heaterValueTF[htrId] = uiUtils.newTextFieldX("0000.00", "H" + htrId, TextFieldX.TYPE_DOUBLE, true);
        }

        // Valve control elements
        for (int vlvId = 0; vlvId < UtilTrunkValves.NUM_VALVES; vlvId++) {
            valveState[vlvId] = UiUtilities.newLabel("XXX", UiUtilities.maxEnumLabelWidth(ValveState.class));
            valveShutRB[vlvId] = uiUtils.newRadioButton("Shut", "VF" + vlvId);
            valveOpenRB[vlvId] = uiUtils.newRadioButton("Open", "VT" + vlvId);
            ButtonGroup bg = valveBG[vlvId] = new ButtonGroup();
            bg.add(valveShutRB[vlvId]);
            bg.add(valveOpenRB[vlvId]);
            valvePosnTF[vlvId] = uiUtils.newTextFieldX("0000.00", "V" + vlvId, TextFieldX.TYPE_DOUBLE, true);
        }

        //** Construct panels

        // Status line panel
        statusPanel = new SystemStatusPanel(sender, MonitorControl.NODE_NAME);

        // UT cooling control panel
        utPanel = UiUtilities.newBorderedPanel("UT Cooling Control");
        GridBagConstraints gbu = new GridBagConstraints();
        gbu.anchor = GridBagConstraints.WEST;
        gbu.insets = new Insets(4, 4, 0, 4);
        gbu.gridx = 0;
        gbu.gridy = 0;
        addFanLine(UtilTrunkFans.FAN_UT_ID, utPanel, gbu);
        gbu.gridy++;
        gbu.insets.top = 0;
        gbu.gridx = 0;
        gbu.insets.bottom = 4;
        addValveLine(UtilTrunkValves.VALVE_UT_ID, utPanel, gbu);
        utPanel.setMinimumSize(utPanel.getPreferredSize());

        // MPC control panel
        mpcPanel = UiUtilities.newBorderedPanel("MPC Control");
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.anchor = GridBagConstraints.WEST;
        gbm.insets = new Insets(4, 4, 0, 4);
        gbm.gridx = 0;
        gbm.gridy = 0;
        addFanLine(UtilTrunkFans.FAN_MPC_ID, mpcPanel, gbm);
        gbm.gridy++;
        gbm.insets.top = 0;
        gbm.gridx = 0;
        gbm.insets.bottom = 4;
        addValveLine(UtilTrunkValves.VALVE_MPC_ID, mpcPanel, gbm);
        mpcPanel.setMinimumSize(mpcPanel.getPreferredSize());

        // VPC control panel
        vpcPanel = UiUtilities.newBorderedPanel("VPC Control");
        GridBagConstraints gbv = new GridBagConstraints();
        gbv.anchor = GridBagConstraints.WEST;
        gbv.insets = new Insets(4, 4, 0, 4);
        gbv.gridx = 0;
        gbv.gridy = 0;
        addModeLine(vpcPanel, gbv);
        gbv.gridy++;
        gbv.insets.top = 0;
        gbv.gridx = 0;
        addHeaterLine(UtilTrunkHeaters.HEATER_VPC1_ID, vpcPanel, gbv);
        gbv.gridy++;
        gbv.gridx = 0;
        addHeaterLine(UtilTrunkHeaters.HEATER_VPC2_ID, vpcPanel, gbv);
        gbv.gridy++;
        gbv.gridx = 0;
        addFanLine(UtilTrunkFans.FAN_VPC_ID, vpcPanel, gbv);
        gbv.gridy++;
        gbv.gridx = 0;
        gbv.insets.bottom = 4;
        addValveLine(UtilTrunkValves.VALVE_VPC_ID, vpcPanel, gbv);
        vpcPanel.setMinimumSize(vpcPanel.getPreferredSize());

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(statusPanel, gbc);
        gbc.gridy++;
        add(utPanel, gbc);
        gbc.gridy++;
        add(mpcPanel, gbc);
        gbc.gridy++;
        add(vpcPanel, gbc);
    }

    private void addFanLine(int fanId, JPanel panel, GridBagConstraints gbc) {
        panel.add(UiUtilities.newLabel("Fan:", 0), gbc);
        gbc.gridx++;
        gbc.insets.left = 4;
        panel.add(fanState[fanId], gbc);
        gbc.gridx++;
        panel.add(fanOffRB[fanId], gbc);
        gbc.gridx++;
        panel.add(fanSpeedRB[fanId], gbc);
        gbc.gridx++;
        panel.add(fanSpeedTF[fanId], gbc);
        gbc.gridx++;
        gbc.insets.left = 0;
        panel.add(UiUtilities.newLabel("D/C", 0), gbc);
        gbc.insets.left = 4;
        gbc.gridx++;
        if (UtilTrunkFans.hasTempState(fanId)) {
            panel.add(fanTempRB[fanId], gbc);
            gbc.gridx++;
            panel.add(fanTempTF[fanId], gbc);
            gbc.gridx++;
            gbc.insets.left = 0;
            panel.add(UiUtilities.newLabel("\u00b0C", 0), gbc);
            gbc.insets.left = 4;
            gbc.gridx++;
        }
    }

    private void addModeLine(JPanel panel, GridBagConstraints gbc) {
        panel.add(UiUtilities.newLabel("Mode:", 0), gbc);
        gbc.gridx++;
        gbc.insets.left = 4;
        panel.add(vpcState, gbc);
        gbc.gridx++;
        panel.add(vpcManualRB, gbc);
        gbc.gridx++;
        panel.add(vpcTempRB, gbc);
        gbc.gridx++;
    }

    private void addValveLine(int valveId, JPanel panel, GridBagConstraints gbc) {
        panel.add(UiUtilities.newLabel("Valve:", 0), gbc);
        gbc.gridx++;
        gbc.insets.left = 4;
        panel.add(valveState[valveId], gbc);
        gbc.gridx++;
        panel.add(valveShutRB[valveId], gbc);
        gbc.gridx++;
        panel.add(valveOpenRB[valveId], gbc);
        gbc.gridx++;
        panel.add(valvePosnTF[valveId], gbc);
        gbc.gridx++;
        panel.add(UiUtilities.newLabel("%", 0), gbc);
        gbc.gridx++;
    }

    private void addHeaterLine(int htrId, JPanel panel, GridBagConstraints gbc) {
        panel.add(UiUtilities.newLabel(heaterNames[htrId] + ":", 0), gbc);
        gbc.gridx++;
        gbc.insets.left = 4;
        panel.add(heaterState[htrId], gbc);
        gbc.gridx++;
        panel.add(heaterOffRB[htrId], gbc);
        gbc.gridx++;
        panel.add(heaterOnRB[htrId], gbc);
        gbc.gridx++;
        panel.add(heaterValueTF[htrId], gbc);
        gbc.gridx++;
        gbc.insets.left = 0;
        panel.add(UiUtilities.newLabel("\u00b0C", 0), gbc);
        gbc.insets.left = 4;
        gbc.gridx++;
    }

    @Override
    public void handleRadioButton(String name) {
        char type = name.charAt(0);
        char action = name.charAt(1);
        int id = Integer.valueOf(name.substring(2));
        switch (type) {
        case 'M':
            VpcControlState mState = action == 'T' ? VpcControlState.TEMP : VpcControlState.MANUAL;
            sender.sendCommand(null, "setVpcState", mState);
            break;
        case 'F':
            FanState fState = action == 'T' ? FanState.TEMP : action == 'S' ? FanState.SPEED : FanState.OFF;
            sender.sendCommand(null, "setFanState", UtilTrunkFans.getName(id), fState);
            break;
        case 'V':
            ValveState vState = action == 'T' ? ValveState.POSN : ValveState.SHUT;
            sender.sendCommand(null, "setValveState", UtilTrunkValves.getName(id), vState);
            break;
        case 'H':
            HeaterState hState = action == 'T' ? HeaterState.ON : HeaterState.OFF;
            sender.sendCommand(null, "setHeaterState", UtilTrunkHeaters.getName(id), hState);
            break;
        }
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        char type = name.charAt(0);
        int id = Integer.valueOf(name.substring(1));
        switch (type) {
        case 'S':
            sender.sendCommand(null, "setFanSpeed", UtilTrunkFans.getName(id), (Double)value);
            break;
        case 'T':
            sender.sendCommand(null, "setFanDeltaTemp", UtilTrunkFans.getName(id), (Double)value);
            break;
        case 'V':
            sender.sendCommand(null, "setValvePosition", UtilTrunkValves.getName(id), (Double)value / 100.0);
            break;
        case 'H':
            sender.sendCommand(null, "setHeaterValue", UtilTrunkHeaters.getName(id), (Double)value);
            break;
        }
    }

    class UpdateUtilState implements Runnable {

        private final UtilTrunkState uts;

        UpdateUtilState(UtilTrunkState uts) {
            this.uts = uts;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(uts.getTickMillis());
            for (int fanId = 0; fanId < UtilTrunkFans.NUM_FANS; fanId++) {
                FanState st = uts.getFanState(fanId);
                st = st == null ? FanState.OFFLINE : st;
                JLabel label = fanState[fanId];
                label.setText(st.toString());
                label.setForeground(st == FanState.TEMP ? UiConstants.GREEN : st == FanState.SPEED ? UiConstants.YELLOW :
                                    st == FanState.OFF ? Color.BLACK : UiConstants.BLUE);
                st = uts.getFanBaseState(fanId);
                JRadioButton selButton = st == FanState.TEMP ? fanTempRB[fanId] :
                                         st == FanState.SPEED ? fanSpeedRB[fanId] : fanOffRB[fanId];
                selButton.setSelected(true);
                fanSpeedTF[fanId].update(uts.getFanSpeed(fanId), true);
                fanTempTF[fanId].update(uts.getDeltaTemp(fanId), true);
                fanState[fanId].setEnabled(true);
                fanOffRB[fanId].setEnabled(true);
                fanSpeedRB[fanId].setEnabled(true);
                fanTempRB[fanId].setEnabled(true);
            }
            for (int htrId = 0; htrId < UtilTrunkHeaters.NUM_HEATERS; htrId++) {
                HeaterState st = uts.getHeaterState(htrId);
                st = st == null ? HeaterState.OFFLINE : st;
                JLabel label = heaterState[htrId];
                label.setText(st.toString());
                label.setForeground(st == HeaterState.TEMP ? UiConstants.GREEN : st == HeaterState.ON ? UiConstants.YELLOW :
                                    st == HeaterState.OFF ? Color.BLACK : UiConstants.BLUE);
                st = uts.getHeaterBaseState(htrId);
                JRadioButton selButton = st == HeaterState.ON ? heaterOnRB[htrId] : heaterOffRB[htrId];
                selButton.setSelected(true);
                heaterValueTF[htrId].update(uts.getHeaterValue(htrId), true);
                heaterState[htrId].setEnabled(true);
                heaterOffRB[htrId].setEnabled(true);
                heaterOnRB[htrId].setEnabled(true);
            }
            for (int vlvId = 0; vlvId < UtilTrunkValves.NUM_VALVES; vlvId++) {
                ValveState st = uts.getValveState(vlvId);
                st = st == null ? ValveState.OFFLINE : st;
                JLabel label = valveState[vlvId];
                label.setText(st.toString());
                label.setForeground(st == ValveState.TEMP ? UiConstants.GREEN : st == ValveState.POSN ? UiConstants.YELLOW :
                                    st == ValveState.SHUT ? Color.BLACK : UiConstants.BLUE);
                st = uts.getValveBaseState(vlvId);
                JRadioButton selButton = st == ValveState.POSN ? valveOpenRB[vlvId] : valveShutRB[vlvId];
                selButton.setSelected(true);
                valvePosnTF[vlvId].update(100.0 * uts.getValvePosition(vlvId), true);
                valveState[vlvId].setEnabled(true);
                valveShutRB[vlvId].setEnabled(true);
                valveOpenRB[vlvId].setEnabled(true);
            }
            VpcControlState vst = uts.getVpcState();
            vst = vst == null ? VpcControlState.MANUAL : vst;
            JLabel label = vpcState;
            label.setText(vst.toString());
            label.setForeground(vst == VpcControlState.TEMP ? UiConstants.GREEN : Color.BLACK);
            JRadioButton selButton = vst == VpcControlState.TEMP ? vpcTempRB : vpcManualRB;
            selButton.setSelected(true);
            vpcState.setEnabled(true);
            vpcManualRB.setEnabled(true);
            vpcTempRB.setEnabled(true);

            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            statusPanel.disablePanel();
            for (int fanId = 0; fanId < UtilTrunkFans.NUM_FANS; fanId++) {
                fanState[fanId].setEnabled(false);
                fanOffRB[fanId].setEnabled(false);
                fanSpeedRB[fanId].setEnabled(false);
                fanTempRB[fanId].setEnabled(false);
                fanSpeedTF[fanId].setDisabled();
                fanTempTF[fanId].setDisabled();
            }
            for (int htrId = 0; htrId < UtilTrunkHeaters.NUM_HEATERS; htrId++) {
                heaterState[htrId].setEnabled(false);
                heaterOffRB[htrId].setEnabled(false);
                heaterOnRB[htrId].setEnabled(false);
                heaterValueTF[htrId].setDisabled();
            }
            for (int vlvId = 0; vlvId < UtilTrunkValves.NUM_VALVES; vlvId++) {
                valveState[vlvId].setEnabled(false);
                valveShutRB[vlvId].setEnabled(false);
                valveOpenRB[vlvId].setEnabled(false);
                valvePosnTF[vlvId].setDisabled();
            }
            vpcState.setEnabled(false);
            vpcManualRB.setEnabled(false);
            vpcTempRB.setEnabled(false);
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
