package org.lsst.ccs.subsystem.utility.ui;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.utility.constants.FanState;
import org.lsst.ccs.subsystem.utility.constants.UTTestFans;
import org.lsst.ccs.subsystem.utility.constants.UTTestSwitches;
import org.lsst.ccs.subsystem.utility.constants.UTTestValves;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.constants.ValveState;
import org.lsst.ccs.subsystem.utility.data.UTTestState;

/**
 *  Implements the purge test control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class UTTestControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    private static final Map<Integer, String> switchNames = new LinkedHashMap<>();
    static {
        switchNames.put(UTTestSwitches.SW_BFR_24V_PDU, "BFR 24V PDU");
        switchNames.put(UTTestSwitches.SW_BFR_48V_PDU, "BFR 48V PDU");
        switchNames.put(UTTestSwitches.SW_PDU_24V_MAIN, "24V PDU Main");
        switchNames.put(UTTestSwitches.SW_PDU_48V_MAIN, "48V PDU Main");
        switchNames.put(UTTestSwitches.SW_PDU_ELEX, "Electronics");
    }
    private static final int switchStateWidth = UiUtilities.maxEnumLabelWidth(SwitchState.class);
    private static final int fanStateWidth = UiUtilities.maxEnumLabelWidth(FanState.class);
    private static final int valveStateWidth = UiUtilities.maxEnumLabelWidth(ValveState.class);

    private final CommandSender sender;
    private final UiUtilities uiUtils;
    private SystemStatusPanel statusPanel;
    private JPanel switchPanel, utPanel;
    private final JLabel[] switchLabel = new JLabel[UTTestSwitches.NUM_PWR_SWITCHES];
    private final JLabel[] switchState = new JLabel[UTTestSwitches.NUM_PWR_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[UTTestSwitches.NUM_PWR_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[UTTestSwitches.NUM_PWR_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[UTTestSwitches.NUM_PWR_SWITCHES];
    private final JLabel[] fanState = new JLabel[UTTestFans.NUM_FANS];
    private final ButtonGroup[] fanBG = new ButtonGroup[UTTestFans.NUM_FANS];
    private final JRadioButton[] fanOffRB = new JRadioButton[UTTestFans.NUM_FANS];
    private final JRadioButton[] fanSpeedRB = new JRadioButton[UTTestFans.NUM_FANS];
    private final TextFieldX[] fanSpeedTF = new TextFieldX[UTTestFans.NUM_FANS];
    private final JRadioButton[] fanTempRB = new JRadioButton[UTTestFans.NUM_FANS];
    private final TextFieldX[] fanTempTF = new TextFieldX[UTTestFans.NUM_FANS];
    private final JLabel[] valveState = new JLabel[UTTestValves.NUM_VALVES];
    private final ButtonGroup[] valveBG = new ButtonGroup[UTTestValves.NUM_VALVES];
    private final JRadioButton[] valveShutRB = new JRadioButton[UTTestValves.NUM_VALVES];
    private final JRadioButton[] valveOpenRB = new JRadioButton[UTTestValves.NUM_VALVES];
    private final TextFieldX[] valvePosnTF = new TextFieldX[UTTestValves.NUM_VALVES];
    private final Map<Integer, Integer> switchLines = new HashMap<>();

    public UTTestControlPanel(String subsys) {
        sender = new CommandSender(subsys, this);
        uiUtils = new UiUtilities(this);
        initComponents();
        (new DisablePanel()).run();
    }

    public void initPanel() {
        sender.sendCommand(true, null, "getFullState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
        updatePanel((UTTestState)reply);
    }

    public void updatePanel(UTTestState rs) {
        SwingUtilities.invokeLater(new UpdatePurgeState(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }

    private void initComponents() {
        // Switch control elements
        int line = 0;
        for (int swId : switchNames.keySet()) {
            switchLines.put(swId, line);
            switchLabel[line] = UiUtilities.newLabel(switchNames.get(swId) + ":", 0);
            switchState[line] = UiUtilities.newLabel("XXX", switchStateWidth);
            switchOnRB[line] = uiUtils.newRadioButton("On", "ST" + swId);
            switchOffRB[line] = uiUtils.newRadioButton("Off", "SF" + swId);
            ButtonGroup bg = switchBG[line] = new ButtonGroup();
            bg.add(switchOnRB[line]);
            bg.add(switchOffRB[line]);
            line++;
        }

        // Fan control elements
        for (int fanId = 0; fanId < UTTestFans.NUM_FANS; fanId++) {
            fanState[fanId] = UiUtilities.newLabel("XXX", fanStateWidth);
            fanOffRB[fanId] = uiUtils.newRadioButton("Off", "FF" + fanId);
            fanSpeedRB[fanId] = uiUtils.newRadioButton("Speed", "FS" + fanId);
            fanTempRB[fanId] = uiUtils.newRadioButton("Temp", "FT" + fanId);
            ButtonGroup bg = fanBG[fanId] = new ButtonGroup();
            bg.add(fanOffRB[fanId]);
            bg.add(fanSpeedRB[fanId]);
            bg.add(fanTempRB[fanId]);
            fanSpeedTF[fanId] = uiUtils.newTextFieldX("0000.00", "S" + fanId, TextFieldX.TYPE_DOUBLE, true);
            fanTempTF[fanId] = uiUtils.newTextFieldX("0000.00", "T" + fanId, TextFieldX.TYPE_DOUBLE, true);
        }

        // Valve control elements
        for (int vlvId = 0; vlvId < UTTestValves.NUM_VALVES; vlvId++) {
            valveState[vlvId] = UiUtilities.newLabel("XXX", valveStateWidth);
            valveShutRB[vlvId] = uiUtils.newRadioButton("Shut", "VF" + vlvId);
            valveOpenRB[vlvId] = uiUtils.newRadioButton("Open", "VT" + vlvId);
            ButtonGroup bg = valveBG[vlvId] = new ButtonGroup();
            bg.add(valveShutRB[vlvId]);
            bg.add(valveOpenRB[vlvId]);
            valvePosnTF[vlvId] = uiUtils.newTextFieldX("0000.00", "V" + vlvId, TextFieldX.TYPE_DOUBLE, true);
        }

        //** Construct panels

        // Status line panel
        statusPanel = new SystemStatusPanel(sender);

        // Switch panel
        switchPanel = UiUtilities.newBorderedPanel("Power Control");
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.WEST;
        gbs.insets = new Insets(0, 0, 4, 4);
        int numColm = 2;
        int colm = 0;
        gbs.gridy = -1;
        gbs.insets.top = 0;
        for (line = 0; line < switchLabel.length; line++) {
            if (switchLabel[line] == null) continue;
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = gbs.gridy == 0 ? 4 : 0;
            gbs.insets.left = colm == 0 ? 4 : 40;
            switchPanel.add(switchLabel[line], gbs);
            gbs.gridx++;
            gbs.insets.left = 4;
            switchPanel.add(switchState[line], gbs);
            gbs.gridx++;
            switchPanel.add(switchOffRB[line], gbs);
            gbs.gridx++;
            switchPanel.add(switchOnRB[line], gbs);
            gbs.gridx++;
            colm = (colm + 1) % numColm;
        }
        switchPanel.setMinimumSize(switchPanel.getPreferredSize());

        // UT cooling control panel
        utPanel = UiUtilities.newBorderedPanel("UT Cooling Control");
        GridBagConstraints gbu = new GridBagConstraints();
        gbu.anchor = GridBagConstraints.WEST;
        gbu.insets = new Insets(4, 4, 0, 4);
        gbu.gridx = 0;
        gbu.gridy = 0;
        addFanLine(UTTestFans.FAN_UT_ID, utPanel, gbu);
        gbu.gridy++;
        gbu.insets.top = 0;
        gbu.gridx = 0;
        addValveLine(UTTestValves.VALVE_UT_ID, utPanel, gbu);
        utPanel.setMinimumSize(utPanel.getPreferredSize());

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(statusPanel, gbc);
        gbc.gridy++;
        add(switchPanel, gbc);
        gbc.gridy++;
        add(utPanel, gbc);
        gbc.gridy++;
    }

    private void addFanLine(int fanId, JPanel panel, GridBagConstraints gbc) {
        panel.add(UiUtilities.newLabel("Fan:", 0), gbc);
        gbc.gridx++;
        gbc.insets.left = 4;
        panel.add(fanState[fanId], gbc);
        gbc.gridx++;
        panel.add(fanOffRB[fanId], gbc);
        gbc.gridx++;
        panel.add(fanSpeedRB[fanId], gbc);
        gbc.gridx++;
        panel.add(fanSpeedTF[fanId], gbc);
        gbc.gridx++;
        gbc.insets.left = 0;
        panel.add(UiUtilities.newLabel("D/C", 0), gbc);
        gbc.insets.left = 4;
        gbc.gridx++;
        if (UTTestFans.hasTempState[fanId]) {
            panel.add(fanTempRB[fanId], gbc);
            gbc.gridx++;
            panel.add(fanTempTF[fanId], gbc);
            gbc.gridx++;
            gbc.insets.left = 0;
            panel.add(UiUtilities.newLabel("\u00b0C", 0), gbc);
            gbc.insets.left = 4;
            gbc.gridx++;
        }
    }

    private void addValveLine(int valveId, JPanel panel, GridBagConstraints gbc) {
        panel.add(UiUtilities.newLabel("Valve:", 0), gbc);
        gbc.gridx++;
        gbc.insets.left = 4;
        panel.add(valveState[valveId], gbc);
        gbc.gridx++;
        panel.add(valveShutRB[valveId], gbc);
        gbc.gridx++;
        panel.add(valveOpenRB[valveId], gbc);
        gbc.gridx++;
        panel.add(valvePosnTF[valveId], gbc);
        gbc.gridx++;
    }

    @Override
    public void handleRadioButton(String name) {
        char type = name.charAt(0);
        char action = name.charAt(1);
        int id = Integer.valueOf(name.substring(2));
        switch (type) {
        case 'S':
            sender.sendCommand(null, "setSwitchOn", UTTestSwitches.switchIdToName.get(id), action == 'T');
            break;
        case 'F':
            FanState fState = action == 'T' ? FanState.TEMP : action == 'S' ? FanState.SPEED : FanState.OFF;
            sender.sendCommand(null, "setFanState", UTTestFans.fanIdToName.get(id), fState);
            break;
        case 'V':
            ValveState vState = action == 'T' ? ValveState.POSN : ValveState.SHUT;
            sender.sendCommand(null, "setValveState", UTTestValves.valveIdToName.get(id), vState);
            break;
        }
    }

    @Override
    public void handleTextFieldX(String name, Object value) {
        char type = name.charAt(0);
        int id = Integer.valueOf(name.substring(1));
        switch (type) {
        case 'S':
            sender.sendCommand(null, "setFanSpeed", UTTestFans.fanIdToName.get(id), (Double)value);
            break;
        case 'T':
            sender.sendCommand(null, "setFanDeltaTemp", UTTestFans.fanIdToName.get(id), (Double)value);
            break;
        case 'V':
            sender.sendCommand(null, "setValvePosition", UTTestValves.valveIdToName.get(id), (Double)value);
            break;
        }
    }

    class UpdatePurgeState implements Runnable {

        private final UTTestState ps;

        UpdatePurgeState(UTTestState ps) {
            this.ps = ps;
        }

        @Override
        public void run() {
            statusPanel.updatePanel(ps.getTickMillis());
            for (int swId : switchNames.keySet()) {
                int line = switchLines.get(swId);
                SwitchState st = ps.getSwitchState(swId);
                st = st == null ? SwitchState.OFFLINE : st;
                JLabel label = switchState[line];
                label.setText(st.toString());
                label.setForeground(st == SwitchState.ON ? UiConstants.GREEN :
                                    st == SwitchState.OFF ? Color.BLACK : UiConstants.BLUE);
                st = ps.getSwitchBaseState(swId);
                JRadioButton selButton = st == SwitchState.ON ? switchOnRB[line] : switchOffRB[line];
                selButton.setSelected(true);
                switchState[line].setEnabled(true);
                switchOffRB[line].setEnabled(true);
                switchOnRB[line].setEnabled(true);
            }

            for (int fanId = 0; fanId < UTTestFans.NUM_FANS; fanId++) {
                FanState st = ps.getFanState(fanId);
                st = st == null ? FanState.OFFLINE : st;
                JLabel label = fanState[fanId];
                label.setText(st.toString());
                label.setForeground(st == FanState.TEMP ? UiConstants.GREEN : st == FanState.SPEED ? UiConstants.YELLOW :
                                    st == FanState.OFF ? Color.BLACK : UiConstants.BLUE);
                st = ps.getFanBaseState(fanId);
                JRadioButton selButton = st == FanState.TEMP ? fanTempRB[fanId] :
                                         st == FanState.SPEED ? fanSpeedRB[fanId] : fanOffRB[fanId];
                selButton.setSelected(true);
                fanSpeedTF[fanId].update(ps.getFanSpeed(fanId), true);
                fanTempTF[fanId].update(ps.getDeltaTemp(fanId), true);
                fanState[fanId].setEnabled(true);
                fanOffRB[fanId].setEnabled(true);
                fanSpeedRB[fanId].setEnabled(true);
                fanTempRB[fanId].setEnabled(true);
            }

            for (int vlvId = 0; vlvId < UTTestValves.NUM_VALVES; vlvId++) {
                ValveState st = ps.getValveState(vlvId);
                st = st == null ? ValveState.OFFLINE : st;
                JLabel label = valveState[vlvId];
                label.setText(st.toString());
                label.setForeground(st == ValveState.TEMP ? UiConstants.GREEN : st == ValveState.POSN ? UiConstants.YELLOW :
                                    st == ValveState.SHUT ? Color.BLACK : UiConstants.BLUE);
                st = ps.getValveBaseState(vlvId);
                JRadioButton selButton = st == ValveState.POSN ? valveOpenRB[vlvId] : valveShutRB[vlvId];
                selButton.setSelected(true);
                valvePosnTF[vlvId].update(ps.getValvePosition(vlvId), true);
                valveState[vlvId].setEnabled(true);
                valveShutRB[vlvId].setEnabled(true);
                valveOpenRB[vlvId].setEnabled(true);
            }

            repaint();
        }

    }

    class DisablePanel implements Runnable {

        @Override
        public void run() {
            statusPanel.disablePanel();
            for (int swId : switchNames.keySet()) {
                int line = switchLines.get(swId);
                switchState[line].setEnabled(false);
                switchOffRB[line].setEnabled(false);
                switchOnRB[line].setEnabled(false);
            }
            for (int fanId = 0; fanId < UTTestFans.NUM_FANS; fanId++) {
                fanState[fanId].setEnabled(false);
                fanOffRB[fanId].setEnabled(false);
                fanSpeedRB[fanId].setEnabled(false);
                fanTempRB[fanId].setEnabled(false);
                fanSpeedTF[fanId].setDisabled();
                fanTempTF[fanId].setDisabled();
            }
            for (int vlvId = 0; vlvId < UTTestValves.NUM_VALVES; vlvId++) {
                valveState[vlvId].setEnabled(false);
                valveShutRB[vlvId].setEnabled(false);
                valveOpenRB[vlvId].setEnabled(false);
                valvePosnTF[vlvId].setDisabled();
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
