package org.lsst.ccs.subsystem.utility.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.utility.constants.FanState;
import org.lsst.ccs.subsystem.utility.constants.HeaterState;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestFans;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestHeaters;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestSwitches;
import org.lsst.ccs.subsystem.utility.constants.PurgeTestValves;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.constants.ValveState;
import org.lsst.ccs.subsystem.utility.constants.VpcControlState;
import org.lsst.ccs.subsystem.utility.data.PurgeTestState;

/**
 *  Implements the purge test control & monitoring panel.
 *
 *  @author Owen Saxton
 */
public class PurgeTestControlPanel extends JPanel {

    private static final Map<Integer, String> switchNames = new LinkedHashMap<>();
    static {
        switchNames.put(PurgeTestSwitches.SW_BFR_24V_PDU, "BFR 24V PDU");
        switchNames.put(PurgeTestSwitches.SW_BFR_48V_PDU, "BFR 48V PDU");
        switchNames.put(PurgeTestSwitches.SW_PDU_24V_MAIN, "24V PDU Main");
        switchNames.put(PurgeTestSwitches.SW_PDU_48V_MAIN, "48V PDU Main");
        switchNames.put(PurgeTestSwitches.SW_PDU_ELEX, "Electronics");
    }
    private static final String[] heaterNames = new String[PurgeTestHeaters.NUM_HEATERS];
    static {
        heaterNames[PurgeTestHeaters.HEATER_VPC1_ID] = "Heater1";
        heaterNames[PurgeTestHeaters.HEATER_VPC2_ID] = "Heater2";
        heaterNames[PurgeTestHeaters.HEATER_SIM_ID] = "Heater";
    }
    private static final int switchNameWidth, switchStateWidth, fanStateWidth, valveStateWidth,
                             heaterNameWidth, heaterStateWidth, vpcStateWidth;
    private static final int rbOffset = -4, tfOffset = -2;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (String name : switchNames.values()) {
            label.setText(name + ":");
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchNameWidth = width;
        width = 0;
        for (SwitchState state : SwitchState.values()) {
            label.setText(state.toString());
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchStateWidth = width;
        width = 0;
        for (FanState state : FanState.values()) {
            label.setText(state.toString());
            width = Math.max(width, label.getPreferredSize().width);
        }
        fanStateWidth = width;
        width = 0;
        for (ValveState state : ValveState.values()) {
            label.setText(state.toString());
            width = Math.max(width, label.getPreferredSize().width);
        }
        valveStateWidth = width;
        width = 0;
        for (String name : heaterNames) {
            label.setText(name + ":");
            width = Math.max(width, label.getPreferredSize().width);
        }
        heaterNameWidth = width;
        width = 0;
        for (HeaterState state : HeaterState.values()) {
            label.setText(state.toString());
            width = Math.max(width, label.getPreferredSize().width);
        }
        heaterStateWidth = width;
        width = 0;
        for (VpcControlState state : VpcControlState.values()) {
            label.setText(state.toString());
            width = Math.max(width, label.getPreferredSize().width);
        }
        vpcStateWidth = width;
    }

    private final CommandSender sender;
    private SystemStatusPanel statusPanel;
    private JPanel switchPanel, mpcPanel, vpcPanel, utPanel, simPanel;
    private final JLabel[] switchLabel = new JLabel[PurgeTestSwitches.NUM_PWR_SWITCHES];
    private final JLabel[] switchState = new JLabel[PurgeTestSwitches.NUM_PWR_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[PurgeTestSwitches.NUM_PWR_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[PurgeTestSwitches.NUM_PWR_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[PurgeTestSwitches.NUM_PWR_SWITCHES];
    private final JLabel[] fanLabel = new JLabel[PurgeTestFans.NUM_FANS];
    private final JLabel[] fanState = new JLabel[PurgeTestFans.NUM_FANS];
    private final ButtonGroup[] fanBG = new ButtonGroup[PurgeTestFans.NUM_FANS];
    private final JRadioButton[] fanOffRB = new JRadioButton[PurgeTestFans.NUM_FANS];
    private final JRadioButton[] fanSpeedRB = new JRadioButton[PurgeTestFans.NUM_FANS];
    private final JTextField[] fanSpeedTF = new JTextField[PurgeTestFans.NUM_FANS];
    private final JRadioButton[] fanTempRB = new JRadioButton[PurgeTestFans.NUM_FANS];
    private final JTextField[] fanTempTF = new JTextField[PurgeTestFans.NUM_FANS];
    private final JLabel[] heaterLabel = new JLabel[PurgeTestHeaters.NUM_HEATERS];
    private final JLabel[] heaterState = new JLabel[PurgeTestHeaters.NUM_HEATERS];
    private final ButtonGroup[] heaterBG = new ButtonGroup[PurgeTestHeaters.NUM_HEATERS];
    private final JRadioButton[] heaterOffRB = new JRadioButton[PurgeTestHeaters.NUM_HEATERS];
    private final JRadioButton[] heaterOnRB = new JRadioButton[PurgeTestHeaters.NUM_HEATERS];
    private final JTextField[] heaterPowerTF = new JTextField[PurgeTestHeaters.NUM_HEATERS];
    private final JLabel[] valveLabel = new JLabel[PurgeTestValves.NUM_VALVES];
    private final JLabel[] valveState = new JLabel[PurgeTestValves.NUM_VALVES];
    private final ButtonGroup[] valveBG = new ButtonGroup[PurgeTestValves.NUM_VALVES];
    private final JRadioButton[] valveShutRB = new JRadioButton[PurgeTestValves.NUM_VALVES];
    private final JRadioButton[] valveOpenRB = new JRadioButton[PurgeTestValves.NUM_VALVES];
    private final JTextField[] valvePosnTF = new JTextField[PurgeTestValves.NUM_VALVES];
    private JLabel vpcLabel;
    private JLabel vpcState;
    private ButtonGroup vpcBG;
    private JRadioButton vpcManualRB;
    private JRadioButton vpcTempRB;
    private final Map<Integer, Integer> switchLines = new HashMap<>();
    private final double[] fanSpeed = new double[PurgeTestFans.NUM_FANS];
    private final double[] fanTemp = new double[PurgeTestFans.NUM_FANS];
    private final double[] heaterPower = new double[PurgeTestHeaters.NUM_HEATERS];
    private final double[] valvePosition = new double[PurgeTestValves.NUM_VALVES];

    public PurgeTestControlPanel(CommandSender sender) {
        this.sender = sender;
        initComponents();
        (new UpdatePurgeStatus(new PurgeTestState())).run();
        (new DisableSystem()).run();
    }

    private void initComponents()
    {
        // Switch control elements
        int line = 0;
        for (int swId : switchNames.keySet()) {
            switchLines.put(swId, line);
            switchLabel[line] = newLabel(switchNames.get(swId) + ":", switchNameWidth);
            switchState[line] = newLabel("XXX", switchStateWidth);
            switchOnRB[line] = newRadioButton("On", "ST" + swId);
            switchOffRB[line] = newRadioButton("Off", "SF" + swId);
            ButtonGroup bg = switchBG[line] = new ButtonGroup();
            bg.add(switchOnRB[line]);
            bg.add(switchOffRB[line]);
            line++;
        }

        // VPC state elements
        vpcLabel = newLabel("Mode:", 0);
        vpcState = newLabel("XXX", vpcStateWidth);
        vpcManualRB = newRadioButton("Manual", "MF0");
        vpcTempRB = newRadioButton("Temp", "MT0");
        vpcBG = new ButtonGroup();
        vpcBG.add(vpcManualRB);
        vpcBG.add(vpcTempRB);

        // Fan control elements
        for (int fanId = 0; fanId < PurgeTestFans.NUM_FANS; fanId++) {
            fanLabel[fanId] = newLabel("Fan:", 0);
            fanState[fanId] = newLabel("XXX", fanStateWidth);
            fanOffRB[fanId] = newRadioButton("Off", "FF" + fanId);
            fanSpeedRB[fanId] = newRadioButton("Speed", "FS" + fanId);
            fanTempRB[fanId] = newRadioButton("Temp", "FT" + fanId);
            ButtonGroup bg = fanBG[fanId] = new ButtonGroup();
            bg.add(fanOffRB[fanId]);
            bg.add(fanSpeedRB[fanId]);
            bg.add(fanTempRB[fanId]);
            fanSpeedTF[fanId] = newTextField("000000", "S" + fanId);
            fanTempTF[fanId] = newTextField("000.00", "T" + fanId);
        }

        // Heater control elements
        for (int htrId = 0; htrId < PurgeTestHeaters.NUM_HEATERS; htrId++) {
            heaterLabel[htrId] = newLabel(heaterNames[htrId] + ":", heaterNameWidth);
            heaterState[htrId] = newLabel("XXX", heaterStateWidth);
            heaterOffRB[htrId] = newRadioButton("Off", "HF" + htrId);
            heaterOnRB[htrId] = newRadioButton("On", "HT" + htrId);
            ButtonGroup bg = heaterBG[htrId] = new ButtonGroup();
            bg.add(heaterOffRB[htrId]);
            bg.add(heaterOnRB[htrId]);
            heaterPowerTF[htrId] = newTextField("00.000", "H" + htrId);
        }

        // Valve control elements
        for (int vlvId = 0; vlvId < PurgeTestValves.NUM_VALVES; vlvId++) {
            valveLabel[vlvId] = newLabel("Valve:", 0);
            valveState[vlvId] = newLabel("XXX", valveStateWidth);
            valveShutRB[vlvId] = newRadioButton("Shut", "VF" + vlvId);
            valveOpenRB[vlvId] = newRadioButton("Open", "VT" + vlvId);
            ButtonGroup bg = valveBG[vlvId] = new ButtonGroup();
            bg.add(valveShutRB[vlvId]);
            bg.add(valveOpenRB[vlvId]);
            valvePosnTF[vlvId] = newTextField("00.000", "V" + vlvId);
        }

        //** Construct panels

        // Status line panel
        statusPanel = new SystemStatusPanel(sender);

        // Switch panel
        switchPanel = newBorderedPanel("Power Control");
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.NORTHWEST;
        int numColm = 2;
        int colm = 0;
        gbs.gridy = 0;
        gbs.insets.top = 0;
        for (line = 0; line < switchLabel.length; line++) {
            if (switchLabel[line] == null) continue;
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = line < numColm ? 4 : 0;
            gbs.insets.left = colm == 0 ? 4 : 40;
            addSwitchLine(line, switchPanel, gbs);
            gbs.gridx++;
            colm = (colm + 1) % numColm;
        }
        switchPanel.setMinimumSize(switchPanel.getPreferredSize());

        // MPC control panel
        mpcPanel = newBorderedPanel("MPC Control");
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.anchor = GridBagConstraints.NORTHWEST;
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.insets.top = 4;
        gbm.insets.left = 4;
        addFanLine(PurgeTestFans.FAN_MPC_ID, mpcPanel, gbm);
        mpcPanel.setMinimumSize(mpcPanel.getPreferredSize());

        // VPC control panel
        vpcPanel = newBorderedPanel("VPC Control");
        GridBagConstraints gbv = new GridBagConstraints();
        gbv.anchor = GridBagConstraints.NORTHWEST;
        gbv.gridx = 0;
        gbv.gridy = 0;
        gbv.insets.top = 4;
        gbv.insets.left = 4;
        addModeLine(vpcPanel, gbv);
        gbv.gridy++;
        gbv.insets.top = 0;
        gbv.insets.left = 4;
        gbv.gridx = 0;
        addValveLine(PurgeTestValves.VALVE_VPC_ID, vpcPanel, gbv);
        gbv.gridy++;
        gbv.insets.left = 4;
        gbv.gridx = 0;
        addHeaterLine(PurgeTestHeaters.HEATER_VPC1_ID, vpcPanel, gbv);
        gbv.gridy++;
        gbv.insets.left = 4;
        gbv.gridx = 0;
        addHeaterLine(PurgeTestHeaters.HEATER_VPC2_ID, vpcPanel, gbv);
        gbv.gridy++;
        gbv.insets.left = 4;
        gbv.gridx = 0;
        addFanLine(PurgeTestFans.FAN_VPC_ID, vpcPanel, gbv);
        vpcPanel.setMinimumSize(vpcPanel.getPreferredSize());

        // Simulator control panel
        simPanel = newBorderedPanel("Simulator Control");
        GridBagConstraints gbe = new GridBagConstraints();
        gbe.anchor = GridBagConstraints.NORTHWEST;
        gbe.gridx = 0;
        gbe.gridy = 0;
        gbe.insets.top = 4;
        gbe.insets.left = 4;
        addHeaterLine(PurgeTestHeaters.HEATER_SIM_ID, simPanel, gbe);
        simPanel.setMinimumSize(simPanel.getPreferredSize());

        // UT cooling control panel
        utPanel = newBorderedPanel("UT Cooling Control");
        GridBagConstraints gbu = new GridBagConstraints();
        gbu.anchor = GridBagConstraints.NORTHWEST;
        gbu.gridx = 0;
        gbu.gridy = 0;
        gbu.insets.top = 4;
        gbu.insets.left = 4;
        addFanLine(PurgeTestFans.FAN_UT_ID, utPanel, gbu);
        gbu.gridy++;
        gbu.insets.top = 0;
        gbu.insets.left = 4;
        gbu.gridx = 0;
        addValveLine(PurgeTestValves.VALVE_UT_ID, utPanel, gbu);
        utPanel.setMinimumSize(utPanel.getPreferredSize());

        // Whole panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.gridx = 0;
        gbc.gridy = 0;
        add(statusPanel, gbc);
        gbc.gridy++;
        add(switchPanel, gbc);
        gbc.gridy++;
        add(mpcPanel, gbc);
        gbc.gridy++;
        add(vpcPanel, gbc);
        gbc.gridy++;
        add(utPanel, gbc);
        gbc.gridy++;
        add(simPanel, gbc);
    }

    private static JLabel newLabel(String text, int width) {
        JLabel label = new JLabel(text);
        label.setFont(UiConstants.FONT);
        Dimension d = label.getPreferredSize();
        if (width > 0) {
            d.width = width;
        }
        label.setPreferredSize(d);
        return label;
    }

    private JRadioButton newRadioButton(String title, String name) {
        JRadioButton rb = new JRadioButton(title);
        rb.setFont(UiConstants.FONT);
        rb.setFocusable(false);
        rb.setName(name);
        rb.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                handleRadioButton(((JRadioButton)evt.getSource()).getName());
            }
        });
        return rb;
    }

    private JTextField newTextField(String text, String name) {
        JTextField tf = new JTextField(text);
        tf.setHorizontalAlignment(SwingConstants.CENTER);
        tf.setPreferredSize(tf.getPreferredSize());
        tf.setName(name);
        tf.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                JTextField source = (JTextField)evt.getSource();
                handleTextField(source.getName(), source.getText());
            }
        });
        return tf;
    }

    private static JPanel newBorderedPanel(String title) {
        JPanel panel = new JPanel();
        TitledBorder border = new TitledBorder(new LineBorder(Color.BLACK), title);
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        panel.setBorder(border);
        panel.setLayout(new GridBagLayout());
        return panel;
    }

    private void addSwitchLine(int line, JPanel panel, GridBagConstraints gbc) {
        panel.add(switchLabel[line], gbc);
        gbc.gridx++;
        gbc.insets.left = 6;
        panel.add(switchState[line], gbc);
        gbc.gridx++;
        gbc.insets.top += rbOffset;
        panel.add(switchOffRB[line], gbc);
        gbc.gridx++;
        gbc.insets.right = 4;
        panel.add(switchOnRB[line], gbc);
        gbc.gridx++;
        gbc.insets.right = 0;
        gbc.insets.top -= rbOffset;
    }

    private void addFanLine(int fanId, JPanel panel, GridBagConstraints gbc) {
        panel.add(fanLabel[fanId], gbc);
        gbc.gridx++;
        gbc.insets.left = 6;
        panel.add(fanState[fanId], gbc);
        gbc.gridx++;
        gbc.insets.top += rbOffset;
        panel.add(fanOffRB[fanId], gbc);
        gbc.gridx++;
        panel.add(fanSpeedRB[fanId], gbc);
        gbc.gridx++;
        gbc.insets.top += tfOffset - rbOffset;
        panel.add(fanSpeedTF[fanId], gbc);
        gbc.gridx++;
        gbc.insets.top -= tfOffset;
        if (PurgeTestFans.hasTempState[fanId]) {
            gbc.insets.top += rbOffset;
            panel.add(fanTempRB[fanId], gbc);
            gbc.gridx++;
            gbc.insets.top += tfOffset - rbOffset;
            gbc.insets.right = 4;
            panel.add(fanTempTF[fanId], gbc);
            gbc.gridx++;
            gbc.insets.top -= tfOffset;
            gbc.insets.right = 0;
        }
    }

    private void addModeLine(JPanel panel, GridBagConstraints gbc) {
        panel.add(vpcLabel, gbc);
        gbc.gridx++;
        gbc.insets.left = 6;
        panel.add(vpcState, gbc);
        gbc.gridx++;
        gbc.insets.top += rbOffset;
        panel.add(vpcManualRB, gbc);
        gbc.gridx++;
        panel.add(vpcTempRB, gbc);
        gbc.gridx++;
        gbc.insets.top += tfOffset - rbOffset;
        panel.add(fanTempTF[PurgeTestFans.FAN_VPC_ID], gbc);
        gbc.gridx++;
        gbc.insets.top -= tfOffset;
    }

    private void addValveLine(int valveId, JPanel panel, GridBagConstraints gbc) {
        panel.add(valveLabel[valveId], gbc);
        gbc.gridx++;
        gbc.insets.left = 6;
        panel.add(valveState[valveId], gbc);
        gbc.gridx++;
        gbc.insets.top += rbOffset;
        panel.add(valveShutRB[valveId], gbc);
        gbc.gridx++;
        panel.add(valveOpenRB[valveId], gbc);
        gbc.gridx++;
        gbc.insets.top += tfOffset - rbOffset;
        panel.add(valvePosnTF[valveId], gbc);
        gbc.gridx++;
        gbc.insets.top -= tfOffset;
    }

    private void addHeaterLine(int htrId, JPanel panel, GridBagConstraints gbc) {
        panel.add(heaterLabel[htrId], gbc);
        gbc.gridx++;
        gbc.insets.left = 6;
        panel.add(heaterState[htrId], gbc);
        gbc.gridx++;
        gbc.insets.top += rbOffset;
        panel.add(heaterOffRB[htrId], gbc);
        gbc.gridx++;
        panel.add(heaterOnRB[htrId], gbc);
        gbc.gridx++;
        gbc.insets.top -= rbOffset;
        if (htrId == PurgeTestHeaters.HEATER_SIM_ID) {
            gbc.insets.top += tfOffset;
            gbc.insets.right = 4;
            panel.add(heaterPowerTF[htrId], gbc);
            gbc.insets.top -= tfOffset;
            gbc.insets.right = 0;
        }
        gbc.gridx++;
    }

    private void handleRadioButton(String name) {
        char type = name.charAt(0);
        char action = name.charAt(1);
        int id = Integer.valueOf(name.substring(2, 3));
        switch (type) {
        case 'S':
            sender.sendCommand(null, "setSwitchOn", PurgeTestSwitches.switchIdToName.get(id), action == 'T');
            break;
        case 'M':
            VpcControlState mState = action == 'T' ? VpcControlState.TEMP : VpcControlState.MANUAL;
            sender.sendCommand(null, "setVpcState", mState);
            break;
        case 'F':
            FanState fState = action == 'T' ? FanState.TEMP : action == 'S' ? FanState.SPEED : FanState.OFF;
            sender.sendCommand(null, "setFanState", PurgeTestFans.fanIdToName.get(id), fState);
            break;
        case 'V':
            ValveState vState = action == 'T' ? ValveState.POSN : ValveState.SHUT;
            sender.sendCommand(null, "setValveState", PurgeTestValves.valveIdToName.get(id), vState);
            break;
        case 'H':
            HeaterState hState = action == 'T' ? HeaterState.ON : HeaterState.OFF;
            sender.sendCommand(null, "setHeaterState", PurgeTestHeaters.heaterIdToName.get(id), hState);
            break;
        }
    }

    private void handleTextField(String name, String text) {
        char type = name.charAt(0);
        int id = Integer.valueOf(name.substring(1, 2));
        double value;
        switch (type) {
        case 'S':
            try {
                value = Double.valueOf(text);
                fanSpeed[id] = value;
                sender.sendCommand(null, "setFanSpeed", PurgeTestFans.fanIdToName.get(id), value);
            }
            catch (NumberFormatException e) {
                fanSpeedTF[id].setText(String.valueOf(fanSpeed[id]));
            }
            break;
        case 'T':
            try {
                value = Double.valueOf(text);
                fanTemp[id] = value;
                sender.sendCommand(null, "setDeltaTemp", PurgeTestFans.fanIdToName.get(id), value);
            }
            catch (NumberFormatException e) {
                fanTempTF[id].setText(String.valueOf(fanTemp[id]));
            }
            break;
        case 'V':
            try {
                value = Double.valueOf(text);
                valvePosition[id] = value;
                sender.sendCommand(null, "setValvePosition", PurgeTestValves.valveIdToName.get(id), value);
            }
            catch (NumberFormatException e) {
                valvePosnTF[id].setText(String.valueOf(valvePosition[id]));
            }
            break;
        case 'H':
            try {
                value = Double.valueOf(text);
                heaterPower[id] = value;
                sender.sendCommand(null, "setHeaterPower", PurgeTestHeaters.heaterIdToName.get(id), value);
            }
            catch (NumberFormatException e) {
                heaterPowerTF[id].setText(String.valueOf(heaterPower[id]));
            }
            break;
        }
    }

    public void updateControlPanel(PurgeTestState rs) {
        SwingUtilities.invokeLater(new UpdatePurgeStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdatePurgeStatus implements Runnable {

        private final PurgeTestState ps;

        UpdatePurgeStatus(PurgeTestState ps) {
            this.ps = ps;
        }

        @Override
        public void run() {
            statusPanel.updateStatus(ps.getTickMillis());
            for (int swId : switchNames.keySet()) {
                int line = switchLines.get(swId);
                SwitchState st = ps.getSwitchState(swId);
                st = st == null ? SwitchState.OFFLINE : st;
                JLabel label = switchState[line];
                label.setText(st.toString());
                label.setForeground(st == SwitchState.ON ? UiConstants.GREEN :
                                    st == SwitchState.OFF ? Color.BLACK : UiConstants.BLUE);
                st = ps.getSwitchBaseState(swId);
                JRadioButton selButton = st == SwitchState.ON ? switchOnRB[line] : switchOffRB[line];
                selButton.setSelected(true);
                switchState[line].setEnabled(true);
                switchOffRB[line].setEnabled(true);
                switchOnRB[line].setEnabled(true);
            }

            for (int fanId = 0; fanId < PurgeTestFans.NUM_FANS; fanId++) {
                FanState st = ps.getFanState(fanId);
                st = st == null ? FanState.OFFLINE : st;
                JLabel label = fanState[fanId];
                label.setText(st.toString());
                label.setForeground(st == FanState.TEMP ? UiConstants.GREEN : st == FanState.SPEED ? UiConstants.YELLOW :
                                    st == FanState.OFF ? Color.BLACK : UiConstants.BLUE);
                st = ps.getFanBaseState(fanId);
                JRadioButton selButton = st == FanState.TEMP ? fanTempRB[fanId] :
                                         st == FanState.SPEED ? fanSpeedRB[fanId] : fanOffRB[fanId];
                selButton.setSelected(true);
                double speed = ps.getFanSpeed(fanId);
                fanSpeed[fanId] = speed;
                fanSpeedTF[fanId].setText(String.format("%.2f", speed));
                double temp = ps.getDeltaTemp(fanId);
                fanTemp[fanId] = temp;
                fanTempTF[fanId].setText(String.format("%.2f", temp));
                fanState[fanId].setEnabled(true);
                fanOffRB[fanId].setEnabled(true);
                fanSpeedRB[fanId].setEnabled(true);
                fanTempRB[fanId].setEnabled(true);
                fanSpeedTF[fanId].setEnabled(true);
                fanTempTF[fanId].setEnabled(true);
            }

            for (int htrId = 0; htrId < PurgeTestHeaters.NUM_HEATERS; htrId++) {
                HeaterState st = ps.getHeaterState(htrId);
                st = st == null ? HeaterState.OFFLINE : st;
                JLabel label = heaterState[htrId];
                label.setText(st.toString());
                label.setForeground(st == HeaterState.TEMP ? UiConstants.GREEN : st == HeaterState.ON ? UiConstants.YELLOW :
                                    st == HeaterState.OFF ? Color.BLACK : UiConstants.BLUE);
                st = ps.getHeaterBaseState(htrId);
                JRadioButton selButton = st == HeaterState.ON ? heaterOnRB[htrId] : heaterOffRB[htrId];
                selButton.setSelected(true);
                double power = ps.getHeaterValue(htrId);
                heaterPower[htrId] = power;
                heaterPowerTF[htrId].setText(String.format("%.2f", power));
                heaterState[htrId].setEnabled(true);
                heaterOffRB[htrId].setEnabled(true);
                heaterOnRB[htrId].setEnabled(true);
                heaterPowerTF[htrId].setEnabled(true);
            }

            for (int vlvId = 0; vlvId < PurgeTestValves.NUM_VALVES; vlvId++) {
                ValveState st = ps.getValveState(vlvId);
                st = st == null ? ValveState.OFFLINE : st;
                JLabel label = valveState[vlvId];
                label.setText(st.toString());
                label.setForeground(st == ValveState.TEMP ? UiConstants.GREEN : st == ValveState.POSN ? UiConstants.YELLOW :
                                    st == ValveState.SHUT ? Color.BLACK : UiConstants.BLUE);
                st = ps.getValveBaseState(vlvId);
                JRadioButton selButton = st == ValveState.POSN ? valveOpenRB[vlvId] : valveShutRB[vlvId];
                selButton.setSelected(true);
                double position = ps.getValvePosition(vlvId);
                valvePosition[vlvId] = position;
                valvePosnTF[vlvId].setText(String.format("%.2f", position));
                valveState[vlvId].setEnabled(true);
                valveShutRB[vlvId].setEnabled(true);
                valveOpenRB[vlvId].setEnabled(true);
                valvePosnTF[vlvId].setEnabled(true);
            }

            VpcControlState vst = ps.getVpcState();
            vst = vst == null ? VpcControlState.MANUAL : vst;
            JLabel label = vpcState;
            label.setText(vst.toString());
            label.setForeground(vst == VpcControlState.TEMP ? UiConstants.GREEN : Color.BLACK);
            JRadioButton selButton = vst == VpcControlState.TEMP ? vpcTempRB : vpcManualRB;
            selButton.setSelected(true);
            vpcState.setEnabled(true);
            vpcManualRB.setEnabled(true);
            vpcTempRB.setEnabled(true);

            repaint();
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            statusPanel.disableSystem();
            for (int swId : switchNames.keySet()) {
                int line = switchLines.get(swId);
                switchState[line].setEnabled(false);
                switchOffRB[line].setEnabled(false);
                switchOnRB[line].setEnabled(false);
            }
            for (int fanId = 0; fanId < PurgeTestFans.NUM_FANS; fanId++) {
                fanState[fanId].setEnabled(false);
                fanOffRB[fanId].setEnabled(false);
                fanSpeedRB[fanId].setEnabled(false);
                fanTempRB[fanId].setEnabled(false);
                fanSpeedTF[fanId].setEnabled(false);
                fanTempTF[fanId].setEnabled(false);
            }
            for (int htrId = 0; htrId < PurgeTestHeaters.NUM_HEATERS; htrId++) {
                heaterState[htrId].setEnabled(false);
                heaterOffRB[htrId].setEnabled(false);
                heaterOnRB[htrId].setEnabled(false);
                heaterPowerTF[htrId].setEnabled(false);
            }
            for (int vlvId = 0; vlvId < PurgeTestValves.NUM_VALVES; vlvId++) {
                valveState[vlvId].setEnabled(false);
                valveShutRB[vlvId].setEnabled(false);
                valveOpenRB[vlvId].setEnabled(false);
                valvePosnTF[vlvId].setEnabled(false);
            }
            vpcState.setEnabled(false);
            vpcManualRB.setEnabled(false);
            vpcTempRB.setEnabled(false);
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
