package org.lsst.ccs.subsystem.utility.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.border.TitledBorder;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.SystemStatusPanel;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.utility.constants.ConditionState;
import org.lsst.ccs.subsystem.utility.constants.LatchState;
import org.lsst.ccs.subsystem.utility.constants.MpmConditions;
import org.lsst.ccs.subsystem.utility.constants.MpmLatches;
import org.lsst.ccs.subsystem.utility.constants.MpmLimits;
import org.lsst.ccs.subsystem.utility.constants.MpmPlcs;
import org.lsst.ccs.subsystem.utility.constants.MpmSwitches;
import org.lsst.ccs.subsystem.utility.constants.PLCState;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.data.MpmSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class MpmControlPanel extends javax.swing.JPanel {

    private static final String[] plcNames = new String[MpmPlcs.NUM_PLCS];
    static {
        plcNames[MpmPlcs.PLC_TRUNK] = "Trunk";
        plcNames[MpmPlcs.PLC_COLD]  = "Cold";
        plcNames[MpmPlcs.PLC_CRYO]  = "Cryo";
    }
    private static final String[] switchNames = new String[MpmSwitches.NUM_SWITCHES];
    static {
        switchNames[MpmSwitches.SW_BLOCK_COLD_HEAT] = "Block Cold Heater";
        switchNames[MpmSwitches.SW_BLOCK_COLD_REFG] = "Block Cold Refrig";
        switchNames[MpmSwitches.SW_BLOCK_CRYO_HEAT] = "Block Cryo Heater";
        switchNames[MpmSwitches.SW_BLOCK_CRYO_REFG] = "Block Cryo Refrig";
        switchNames[MpmSwitches.SW_BLOCK_COOLANT]   = "Block UT Coolant";
        switchNames[MpmSwitches.SW_BLOCK_REB_POWER] = "Block REB Power";
        switchNames[MpmSwitches.SW_BLOCK_UT_POWER]  = "Block UT Power";
    }
    private static final String[] latchNames = new String[MpmLatches.NUM_LATCHES];
    static {
        latchNames[MpmLatches.LATCH_COLD_TEMP_HIGH] = "Cold Temp > 99C";
        latchNames[MpmLatches.LATCH_COLD_TEMP_LOW]  = "Cold Temp < -99C";
        latchNames[MpmLatches.LATCH_CRYO_TEMP_HIGH] = "Cryo Temp > 99C";
        latchNames[MpmLatches.LATCH_CRYO_TEMP_LOW]  = "Cryo Temp < -999C";
        latchNames[MpmLatches.LATCH_CRYO_VACUUM]    = "Cryo Vacuum Bad";
        latchNames[MpmLatches.LATCH_HEX_VACUUM]     = "Hex Vacuum Bad";
        latchNames[MpmLatches.LATCH_UT_LEAK]        = "UT Leak";
        latchNames[MpmLatches.LATCH_UT_LEAK_FAULT]  = "UT Leak Fault";
        latchNames[MpmLatches.LATCH_UT_SMOKE]       = "UT Smoke";
        latchNames[MpmLatches.LATCH_UT_SMOKE_FAULT] = "UT Smoke Fault";
        latchNames[MpmLatches.LATCH_UT_TEMP]        = "UT Temp High";
    }
    private static final String[] conditionNames = new String[MpmConditions.NUM_CONDITIONS];
    static {
        conditionNames[MpmConditions.COND_UT_POWER]  = "UT Power Permit";
        conditionNames[MpmConditions.COND_REB_POWER] = "REB Power Permit";
        conditionNames[MpmConditions.COND_COOLANT]   = "UT Coolant Permit";
        conditionNames[MpmConditions.COND_COLD_REFG] = "Cold Refrig Permit";
        conditionNames[MpmConditions.COND_COLD_HEAT] = "Cold Heater Permit";
        conditionNames[MpmConditions.COND_CRYO_REFG] = "Cryo Refrig Permit";
        conditionNames[MpmConditions.COND_CRYO_HEAT] = "Cryo Heater Permit";
    }
    private static final int plcStateWidth, switchStateWidth, latchStateWidth, conditionStateWidth;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (PLCState state : PLCState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        plcStateWidth = width;
        width = 0;
        for (SwitchState state : SwitchState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchStateWidth = width;
        width = 0;
        for (LatchState state : LatchState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        latchStateWidth = width;
        width = 0;
        for (ConditionState state : ConditionState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        conditionStateWidth = width;
    }

    private SystemStatusPanel systemPanel;

    private JPanel plcPanel;
    private final JLabel[] plcLabel = new JLabel[MpmPlcs.NUM_PLCS];
    private final JLabel[] plcStatus = new JLabel[MpmPlcs.NUM_PLCS];

    private JPanel switchPanel;
    private final JLabel[] switchLabel = new JLabel[MpmSwitches.NUM_SWITCHES];
    private final JLabel[] switchStatus = new JLabel[MpmSwitches.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[MpmSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[MpmSwitches.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[MpmSwitches.NUM_SWITCHES];

    private JPanel latchPanel;
    private final JLabel[] latchLabel = new JLabel[MpmLatches.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[MpmLatches.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[MpmLatches.NUM_LATCHES];

    private JPanel conditionPanel;
    private final JLabel[] conditionLabel = new JLabel[MpmConditions.NUM_CONDITIONS];
    private final JLabel[] conditionStatus = new JLabel[MpmConditions.NUM_CONDITIONS];

    private final CommandSender sender;
    private boolean gotTempLimits = false;

    public MpmControlPanel(CommandSender cons) {
        this.sender = cons;
        initComponents();
        (new UpdateSysStatus(new MpmSysState())).run();
        (new DisableSystem()).run();
    }

    private void initComponents() {

        // System status
        systemPanel = new SystemStatusPanel(sender);

        // PLC states
        for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
            plcLabel[plc] = newLabel(plcNames[plc] + ":", 0);
            plcStatus[plc] = newLabel("XXX", plcStateWidth);
        }

        // Switch states
        for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
            switchLabel[sw] = newLabel(switchNames[sw] + ":", 0);
            switchStatus[sw] = newLabel("XXX", switchStateWidth);
            switchOffRB[sw] = newRadioButton("Off", "XF" + sw);
            switchOnRB[sw] = newRadioButton("On", "XT" + sw);
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(switchOffRB[sw]);
            bg.add(switchOnRB[sw]);
        }

        // Latched condition states
        for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
            latchLabel[cond] = newLabel(latchNames[cond] + ":", 0);
            latchStatus[cond] = newLabel("XXX", latchStateWidth);
            latchResetBtn[cond] = newButton("Reset", "XX" + cond);
        }

        // Condition states
        for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
            conditionLabel[cond] = newLabel(conditionNames[cond] + ":", 0);
            conditionStatus[cond] = newLabel("XXX", conditionStateWidth);
        }

        // Layout the PLC panel
        plcPanel = newBorderedPanel("PLC States");
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.anchor = GridBagConstraints.NORTHWEST;
        gbp.gridx = 0;
        gbp.gridy = 0;
        gbp.insets = new Insets(4, 5, 4, 5);
        for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
            addPlcLine(plc, plcPanel, gbp);
            gbp.insets.left = 35;
        }

        // Layout the switch panel
        switchPanel = newBorderedPanel("Switches");
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.NORTHWEST;
        gbs.gridy = 0;
        int numColm = 2;
        int colm = 0;
        for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = sw < numColm ? 4 : 0;
            gbs.insets.left = colm == 0 ? 5 : 40;
            addSwitchLine(sw, switchPanel, gbs);
            colm = (colm + 1) % numColm;
        }

        // Layout the latched conditions panel
        latchPanel = newBorderedPanel("PLC Error Conditions");
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.NORTHWEST;
        gbl.insets.bottom = 4;
        gbl.gridy = 0;
        numColm = 2;
        colm = 0;
        for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
            if (colm == 0) {
                gbl.gridx = 0;
                gbl.gridy++;
            }
            gbl.insets.top = cond < numColm ? 4 : 0;
            gbl.insets.left = colm == 0 ? 5 : 40;
            addLatchLine(cond, latchPanel, gbl);
            colm = (colm + 1) % numColm;
        }

        // Layout the conditions panel
        conditionPanel = newBorderedPanel("PLC Running Conditions");
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = new Insets(0, 0, 4, 5);
        gbc.gridy = 0;
        numColm = 2;
        colm = 0;
        for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
            if (colm == 0) {
                gbc.gridx = 0;
                gbc.gridy++;
            }
            gbc.insets.top = cond < numColm ? 4 : 0;
            gbc.insets.left = colm == 0 ? 5 : 35;
            addConditionLine(cond, conditionPanel, gbc);
            colm = (colm + 1) % numColm;
        }

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(systemPanel, gbm);
        gbm.gridy++;
        add(plcPanel, gbm);
        gbm.gridy++;
        gbm.insets.top = 6;
        add(switchPanel, gbm);
        gbm.gridy++;
        add(latchPanel, gbm);
        gbm.gridy++;
        gbm.insets.bottom = 6;
        add(conditionPanel, gbm);

    }

    private static JLabel newLabel(String text, int width) {
        JLabel label = new JLabel(text);
        label.setFont(UiConstants.FONT);
        Dimension d = label.getPreferredSize();
        if (width > 0) {
            d.width = width;
        }
        label.setPreferredSize(d);
        return label;
    }

    private JRadioButton newRadioButton(String title, String name) {
        JRadioButton rb = new JRadioButton(title);
        rb.setFont(UiConstants.FONT);
        rb.setFocusable(false);
        rb.setName(name);
        rb.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                handleRadioButton(((JRadioButton)evt.getSource()).getName());
            }
        });
        return rb;
    }

    private JButton newButton(String title, String name) {
        JButton btn = new JButton(title);
        Insets i = btn.getMargin();
        i.left -= 3;
        i.right -= 3;
        btn.setMargin(i);
        Dimension d = btn.getPreferredSize();
        d.height -= 9;
        btn.setPreferredSize(d);
        btn.setFont(UiConstants.FONT);
        btn.setFocusable(false);
        btn.setName(name);
        btn.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                handleButton(((JButton)evt.getSource()).getName());
            }
        });
        return btn;
    }

    private static JPanel newBorderedPanel(String title) {
        JPanel panel = new JPanel();
        TitledBorder border = new TitledBorder(new LineBorder(Color.BLACK), title);
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        panel.setBorder(border);
        panel.setLayout(new GridBagLayout());
        return panel;
    }

    private void addPlcLine(int plc, JPanel panel, GridBagConstraints gbc) {
        panel.add(plcLabel[plc], gbc);
        gbc.gridx++;
        gbc.insets.left = 0;
        panel.add(plcStatus[plc], gbc);
        gbc.gridx++;
    }

    private void addSwitchLine(int sw, JPanel panel, GridBagConstraints gbc) {
        panel.add(switchLabel[sw], gbc);
        gbc.insets.left = 6;
        gbc.gridx++;
        panel.add(switchStatus[sw], gbc);
        gbc.gridx++;
        gbc.insets.left = 8;
        gbc.insets.top -= 4;
        panel.add(switchOffRB[sw], gbc);
        gbc.gridx++;
        gbc.insets.left = 3;
        gbc.insets.right = 3;
        panel.add(switchOnRB[sw], gbc);
        gbc.gridx++;
        gbc.insets.right = 0;
    }

    private void addLatchLine(int cond, JPanel panel, GridBagConstraints gbc) {
        panel.add(latchLabel[cond], gbc);
        gbc.insets.left = 5;
        gbc.gridx++;
        panel.add(latchStatus[cond], gbc);
        gbc.gridx++;
        gbc.insets.top -= 1;
        gbc.insets.right = 4;
        panel.add(latchResetBtn[cond], gbc);
        gbc.insets.right = 0;
        gbc.gridx++;
    }

    private void addConditionLine(int cond, JPanel panel, GridBagConstraints gbc) {
        panel.add(conditionLabel[cond], gbc);
        gbc.insets.left = 0;
        gbc.gridx++;
        panel.add(conditionStatus[cond], gbc);
        gbc.gridx++;
    }

    private void handleRadioButton(String name) {
        char type = name.charAt(0);
        char action = name.charAt(1);
        String swName = MpmSwitches.ID_MAP.get(Integer.valueOf(name.substring(2, 3)));
        sender.sendCommand(null, "setNamedSwitchOn", swName, action == 'T');
    }

    private void handleButton(String name) {
        char type = name.charAt(0);
        char action = name.charAt(1);
        String lName = MpmLatches.ID_MAP.get(Integer.valueOf(name.substring(2, 3)));
        sender.sendCommand(null, "clearLatch", lName);
    }

    public void updateControlPanel(MpmSysState rs) {
        SwingUtilities.invokeLater(new UpdateSysStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdateSysStatus implements Runnable {

        private final MpmSysState ps;

        UpdateSysStatus(MpmSysState ps) {
            this.ps = ps;
        }

        @Override
        public void run() {
            systemPanel.updateStatus(ps.getTickMillis());

            for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
                PLCState state = ps.getPlcState(plc);
                plcStatus[plc].setText(state.name());
                plcStatus[plc].setForeground(state == PLCState.OFFLINE ? UiConstants.BLUE :
                                             state == PLCState.DEAD ? UiConstants.RED : UiConstants.GREEN);
                plcStatus[plc].setEnabled(true);
            }
            for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
                SwitchState state = ps.getSwitchState(sw);
                String text;
                Color color;
                text = state.name();
                color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? UiConstants.GREEN : UiConstants.BLUE;
                switchStatus[sw].setText(text);
                switchStatus[sw].setForeground(color);

                JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
                LatchState state = ps.getLatch(cond);
                latchStatus[cond].setText(state.name());
                latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
                                                state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
                latchStatus[cond].setEnabled(true);
                latchResetBtn[cond].setEnabled(true);
            }
            for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
                ConditionState state = ps.getCondition(cond);
                conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
            }
            if (!gotTempLimits && ps.getLimit(0) != Integer.MAX_VALUE) {
                latchLabel[MpmLatches.LATCH_COLD_TEMP_HIGH].setText("Cold Temp > " + ps.getLimit(MpmLimits.LIMIT_COLD_HIGH) + "C:");
                latchLabel[MpmLatches.LATCH_COLD_TEMP_LOW].setText("Cold Temp < " + ps.getLimit(MpmLimits.LIMIT_COLD_LOW) + "C:");
                latchLabel[MpmLatches.LATCH_CRYO_TEMP_HIGH].setText("Cryo Temp > " + ps.getLimit(MpmLimits.LIMIT_CRYO_HIGH) + "C:");
                latchLabel[MpmLatches.LATCH_CRYO_TEMP_LOW].setText("Cryo Temp < " + ps.getLimit(MpmLimits.LIMIT_CRYO_LOW) + "C:");
                gotTempLimits = true;
            }
            repaint();
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            systemPanel.disableSystem();
            for (int plc = 0; plc < MpmPlcs.NUM_PLCS; plc++) {
                plcStatus[plc].setEnabled(false);
            }
            for (int sw = 0; sw < MpmSwitches.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < MpmLatches.NUM_LATCHES; cond++) {
                latchStatus[cond].setEnabled(false);
                latchResetBtn[cond].setEnabled(false);
            }
            for (int cond = 0; cond < MpmConditions.NUM_CONDITIONS; cond++) {
                conditionStatus[cond].setEnabled(false);
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
