package org.lsst.ccs.subsystem.utility.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.utility.data.UtilityState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class UTMonitorPanel extends javax.swing.JPanel {

    static final Font FONT = new java.awt.Font("Tahoma", 1, 12);
    static final Color RED = new Color(150, 0, 0), GREEN = new Color(0, 150, 0);
    final private CommandSender gui;
    private String subsys;
    private double mainPeriod = 5.0;
    private double loopGain = 0.0;
    private double loopTimeConst = 0.0;
    private double dutyCycle = 0.0;
    private double coldTemp = 0.0;

    private final JPanel mainPanel = new JPanel();

    //private final JPanel statePanel = new JPanel();
    private final JLabel stateLabel = new JLabel("Utility State: ");
    private final JLabel stateStatus = new JLabel("STOPPED");

    private final JPanel periodPanel = new JPanel();
    private final JLabel periodLabel = new JLabel("Update Period: ");
    private final JTextField periodTextField = new JTextField();
    private final JLabel periodUnitsLabel = new JLabel("sec");

    private final JLabel gainLabel = new JLabel("Loop gain: ");
    private final JTextField gainTextField = new JTextField();

    private final JPanel timePanel = new JPanel();
    private final JLabel timeLabel = new JLabel("Time constant: ");
    private final JTextField timeTextField = new JTextField();
    private final JLabel timeUnitsLabel = new JLabel("sec");

    private final JLabel fanControlLabel = new JLabel("Fan Control: ");
    private final JLabel fanControlStatus = new JLabel("OFF");
    private final ButtonGroup fanControlBG = new ButtonGroup();
    private final JRadioButton fanControlOffRB = new JRadioButton("Off");
    private final JRadioButton fanControlManualRB = new JRadioButton("Manual");
    private final JRadioButton fanControlAutoRB = new JRadioButton("Auto");

    private final JPanel dutyPanel = new JPanel();
    private final JTextField dutyTextField = new JTextField();

    private final JPanel tempPanel = new JPanel();
    private final JTextField tempTextField = new JTextField();
    private final JLabel tempUnitsLabel = new JLabel("°C");

    public UTMonitorPanel(CommandSender gui) {
        this.gui = gui;
        initComponents();
        (new UpdateUTStatus(new UtilityState())).run();
        (new DisableSystem()).run();
    }

    public void setSubsystem(String name) {
        this.subsys = name;
    }

    private void initComponents() {

        // System state
        stateLabel.setFont(FONT);
        stateStatus.setFont(FONT);
        Dimension d = stateStatus.getPreferredSize();
        d.width = 75;
        stateStatus.setMinimumSize(d);
        stateStatus.setPreferredSize(d);

        // Update period
        periodLabel.setFont(FONT);
        d = periodTextField.getPreferredSize();
        d.width = 45;
        periodTextField.setMinimumSize(d);
        periodTextField.setPreferredSize(d);
        periodTextField.setHorizontalAlignment(SwingConstants.CENTER);
        periodTextField.setText("...");
        periodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });
        periodUnitsLabel.setFont(FONT);

        // Loop gain
        gainLabel.setFont(FONT);
        d = gainTextField.getPreferredSize();
        d.width = 45;
        gainTextField.setMinimumSize(d);
        gainTextField.setPreferredSize(d);
        gainTextField.setHorizontalAlignment(SwingConstants.CENTER);
        gainTextField.setText("...");
        gainTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setLoopGain();
            }
        });

        // Loop time constant
        timeLabel.setFont(FONT);
        d = timeTextField.getPreferredSize();
        d.width = 60;
        timeTextField.setMinimumSize(d);
        timeTextField.setPreferredSize(d);
        timeTextField.setHorizontalAlignment(SwingConstants.CENTER);
        timeTextField.setText("...");
        timeTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setLoopTimeConst();
            }
        });
        timeUnitsLabel.setFont(FONT);

        // Fan control state
        fanControlLabel.setFont(FONT);
        fanControlStatus.setFont(FONT);
        d = fanControlStatus.getPreferredSize();
        d.width = 60;
        fanControlStatus.setMinimumSize(d);
        fanControlStatus.setPreferredSize(d);
        fanControlOffRB.setFont(FONT);
        fanControlOffRB.setFocusable(false);
        fanControlOffRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setControlState(UtilityState.OPS_CTRL_OFF);
            }
        });
        fanControlManualRB.setFont(FONT);
        fanControlManualRB.setFocusable(false);
        fanControlManualRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setControlState(UtilityState.OPS_CTRL_MANUAL);
            }
        });
        fanControlAutoRB.setFont(FONT);
        fanControlAutoRB.setFocusable(false);
        fanControlAutoRB.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setControlState(UtilityState.OPS_CTRL_AUTO);
            }
        });
        fanControlBG.add(fanControlOffRB);
        fanControlBG.add(fanControlManualRB);
        fanControlBG.add(fanControlAutoRB);

        // Duty cycle
        d = dutyTextField.getPreferredSize();
        d.width = 50;
        dutyTextField.setMinimumSize(d);
        dutyTextField.setPreferredSize(d);
        dutyTextField.setHorizontalAlignment(SwingConstants.CENTER);
        dutyTextField.setText("...");
        dutyTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setControlDuty();
            }
        });

        // Temperature
        d = tempTextField.getPreferredSize();
        d.width = 60;
        tempTextField.setMinimumSize(d);
        tempTextField.setPreferredSize(d);
        tempTextField.setHorizontalAlignment(SwingConstants.CENTER);
        tempTextField.setText("...");
        tempTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setControlTemp();
            }
        });
        tempUnitsLabel.setFont(FONT);

        // Lay out all the sub-panels
        periodPanel.add(periodTextField);
        periodPanel.add(periodUnitsLabel);
        timePanel.add(timeTextField);
        timePanel.add(timeUnitsLabel);
        dutyPanel.add(dutyTextField);
        tempPanel.add(tempTextField);
        tempPanel.add(tempUnitsLabel);

        // Add state & period items to main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(5, 0, 5, 0);
        gbm.gridx = 0;
        gbm.gridy = 0;
        gbm.anchor = GridBagConstraints.EAST;
        mainPanel.add(stateLabel, gbm);
        gbm.gridx++;
        gbm.anchor = GridBagConstraints.WEST;
        mainPanel.add(stateStatus, gbm);
        gbm.gridx++;
        gbm.gridwidth = 2;
        gbm.anchor = GridBagConstraints.EAST;
        mainPanel.add(periodLabel, gbm);
        gbm.gridx += 2;
        gbm.gridwidth = 1;
        gbm.anchor = GridBagConstraints.WEST;
        mainPanel.add(periodPanel, gbm);
        gbm.gridx = 0;
        gbm.gridy++;

        // Add gain & time constant items to main panel
        gbm.anchor = GridBagConstraints.EAST;
        gbm.insets.top = -4;
        mainPanel.add(gainLabel, gbm);
        gbm.gridx++;
        gbm.anchor = GridBagConstraints.WEST;
        mainPanel.add(gainTextField, gbm);
        gbm.gridx++;
        gbm.gridwidth = 2;
        gbm.anchor = GridBagConstraints.EAST;
        mainPanel.add(timeLabel, gbm);
        gbm.gridx += 2;
        gbm.gridwidth = 1;
        gbm.anchor = GridBagConstraints.WEST;
        mainPanel.add(timePanel, gbm);
        gbm.gridx = 0;
        gbm.gridy++;

        // Add fan control to main panel
        gbm.insets.top = 0;
        gbm.anchor = GridBagConstraints.NORTHEAST;
        mainPanel.add(fanControlLabel, gbm);
        gbm.gridx++;
        gbm.anchor = GridBagConstraints.NORTHWEST;
        mainPanel.add(fanControlStatus, gbm);
        gbm.gridx++;
        gbm.insets.top = -4;
        mainPanel.add(fanControlOffRB, gbm);
        gbm.gridx++;
        gbm.insets.left = 6;
        mainPanel.add(fanControlManualRB, gbm);
        gbm.gridx++;
        mainPanel.add(fanControlAutoRB, gbm);
        gbm.gridx = 0;
        gbm.gridy++;

        // Add duty & temperature panels to main panel
        gbm.insets.top = -10;
        gbm.insets.left = 6;
        gbm.gridx = 3;
        mainPanel.add(dutyPanel, gbm);
        gbm.gridx++;
        mainPanel.add(tempPanel, gbm);

        // Lay out the complete panel
        add(mainPanel);

    }

    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(periodTextField.getText());
            gui.sendCommand(subsys, null, "setUpdatePeriod", (int)(1000 * value));
            periodTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            periodTextField.setText(String.valueOf(mainPeriod));
        }
    }

    private void setLoopGain() {
        try {
            double value = Double.valueOf(gainTextField.getText());
            gui.sendCommand(subsys, null, "setGain", value);
            gainTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            gainTextField.setText(String.valueOf(loopGain));
        }
    }

    private void setLoopTimeConst() {
        try {
            double value = Double.valueOf(timeTextField.getText());
            gui.sendCommand(subsys, null, "setTimeConstant", value);
            timeTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            timeTextField.setText(String.valueOf(loopTimeConst));
        }
    }

    private void setControlState(int value) {
        fanControlOffRB.setEnabled(false);
        fanControlManualRB.setEnabled(false);
        fanControlAutoRB.setEnabled(false);
        gui.sendCommand(subsys, null, "setControlState", value);
    }
    
    private void setControlDuty() {
        try {
            double value = Double.valueOf(dutyTextField.getText());
            gui.sendCommand(subsys, null, "setFanSpeed", value);
            dutyTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            dutyTextField.setText(String.valueOf(dutyCycle));
        }
    }

    private void setControlTemp() {
        try {
            double value = Double.valueOf(tempTextField.getText());
            gui.sendCommand(subsys, null, "setColdTemp", value);
            tempTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            tempTextField.setText(String.valueOf(coldTemp));
        }
    }

    public void updateControlPanel(UtilityState rs) {
        SwingUtilities.invokeLater(new UpdateUTStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdateUTStatus implements Runnable {

        private final UtilityState us;

        UpdateUTStatus(UtilityState us) {
            this.us = us;
        }

        @Override
        public void run() {
            stateStatus.setText("RUNNING");
            stateStatus.setForeground(GREEN);
            periodTextField.setEnabled(true);
            mainPeriod = us.getTickMillis() / 1000.0;
            periodTextField.setText(String.valueOf(mainPeriod));
            gainTextField.setEnabled(true);
            loopGain = us.getLoopGain();
            gainTextField.setText(String.valueOf(loopGain));
            timeTextField.setEnabled(true);
            loopTimeConst = us.getLoopTimeConst();
            timeTextField.setText(String.valueOf(loopTimeConst));
            int state = us.getOpState();
            fanControlStatus.setEnabled(true);
            fanControlStatus.setText(state == UtilityState.OPS_CTRL_MANUAL ? "MANUAL" :
                                     state == UtilityState.OPS_CTRL_AUTO ? "AUTO" : "OFF");
            fanControlStatus.setForeground(state == UtilityState.OPS_CTRL_MANUAL ? GREEN :
                                           state == UtilityState.OPS_CTRL_AUTO ? RED : Color.black);
            JRadioButton selButton = state == UtilityState.OPS_CTRL_MANUAL ? fanControlManualRB :
                                     state == UtilityState.OPS_CTRL_AUTO ? fanControlAutoRB : fanControlOffRB;
            selButton.setSelected(true);
            fanControlOffRB.setEnabled(selButton != fanControlOffRB);
            fanControlManualRB.setEnabled(selButton != fanControlManualRB);
            fanControlAutoRB.setEnabled(selButton != fanControlAutoRB);
            dutyTextField.setEnabled(true);
            dutyCycle = us.getDutyCycle();
            dutyTextField.setText(String.valueOf(dutyCycle));
            tempTextField.setEnabled(true);
            coldTemp = us.getCoolTemp();
            tempTextField.setText(String.valueOf(coldTemp));
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            stateStatus.setText("STOPPED");
            stateStatus.setForeground(RED);
            periodTextField.setEnabled(false);
            gainTextField.setEnabled(false);
            timeTextField.setEnabled(false);
            fanControlStatus.setEnabled(false);
            fanControlOffRB.setEnabled(false);
            fanControlManualRB.setEnabled(false);
            fanControlAutoRB.setEnabled(false);
            dutyTextField.setEnabled(false);
            tempTextField.setEnabled(false);
        } 
    }

    private static final long serialVersionUID = 1L;
}
