package org.lsst.ccs.subsystem.utility.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.border.TitledBorder;
import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;
import org.lsst.ccs.subsystem.utility.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.utility.constants.ConditionState;
import org.lsst.ccs.subsystem.utility.constants.LatchState;
import org.lsst.ccs.subsystem.utility.constants.PLCState;
import org.lsst.ccs.subsystem.utility.constants.SwitchState;
import org.lsst.ccs.subsystem.utility.data.MpmSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
public class MpmControlPanel extends javax.swing.JPanel {

    static final int SWTP_PUMP = 0, SWTP_VALVE = 1;
    static final String[] plcNames = new String[MpmSysState.NUM_PLCS];
    static {
        plcNames[MpmSysState.PLC_TRUNK] = "Trunk";
        plcNames[MpmSysState.PLC_COLD]  = "Cold";
        plcNames[MpmSysState.PLC_CRYO]  = "Cryo";
    }
    static final String[] switchNames = new String[MpmSysState.NUM_SWITCHES];
    static {
        switchNames[MpmSysState.SW_BLOCK_COLD_HEAT] = "Block Cold Heater";
        switchNames[MpmSysState.SW_BLOCK_COLD_REFG] = "Block Cold Refrig";
        switchNames[MpmSysState.SW_BLOCK_CRYO_HEAT] = "Block Cryo Heater";
        switchNames[MpmSysState.SW_BLOCK_CRYO_REFG] = "Block Cryo Refrig";
        switchNames[MpmSysState.SW_BLOCK_COOLANT]   = "Block UT Coolant";
        switchNames[MpmSysState.SW_BLOCK_REB_POWER] = "Block REB Power";
        switchNames[MpmSysState.SW_BLOCK_UT_POWER]  = "Block UT Power";
    }
    static final String[] latchNames = new String[MpmSysState.NUM_LATCHES];
    static {
        latchNames[MpmSysState.LATCH_COLD_TEMP_HIGH]  = "Cold Temp High";
        latchNames[MpmSysState.LATCH_COLD_TEMP_LOW]  = "Cold Temp Low";
        latchNames[MpmSysState.LATCH_CRYO_TEMP_HIGH]  = "Cryo Temp High";
        latchNames[MpmSysState.LATCH_CRYO_TEMP_LOW]  = "Cryo Temp Low";
        latchNames[MpmSysState.LATCH_CRYO_VACUUM]  = "Cryo Vacuum Bad";
        latchNames[MpmSysState.LATCH_HEX_VACUUM]  = "Hex Vacuum Bad";
        latchNames[MpmSysState.LATCH_UT_LEAK]  = "UT Leak";
        latchNames[MpmSysState.LATCH_UT_LEAK_FAULT]  = "UT Leak Fault";
        latchNames[MpmSysState.LATCH_UT_SMOKE]  = "UT Smoke";
        latchNames[MpmSysState.LATCH_UT_SMOKE_FAULT]  = "UT Smoke Fault";
        latchNames[MpmSysState.LATCH_UT_TEMP]  = "UT Temp High";
    }
    static final String[] conditionNames = new String[MpmSysState.NUM_CONDITIONS];
    static {
        conditionNames[MpmSysState.COND_UT_POWER] = "UT Power Permit";
        conditionNames[MpmSysState.COND_REB_POWER] = "REB Power Permit";
        conditionNames[MpmSysState.COND_COOLANT] = "UT Coolant Permit";
        conditionNames[MpmSysState.COND_COLD_REFG] = "Cold Refrig Permit";
        conditionNames[MpmSysState.COND_COLD_HEAT] = "Cold Heater Permit";
        conditionNames[MpmSysState.COND_CRYO_REFG] = "Cryo Refrig Permit";
        conditionNames[MpmSysState.COND_CRYO_HEAT] = "Cryo Heater Permit";
    }
    static final int plcNameWidth, plcStateWidth, switchNameWidth, switchStateWidth,
                     latchNameWidth, latchStateWidth, conditionNameWidth, conditionStateWidth;
    static {
        JLabel label = new JLabel();
        label.setFont(UiConstants.FONT);
        int width = 0;
        for (String name : plcNames) {
            label.setText(name + ":");
            width = Math.max(width, label.getPreferredSize().width);
        }
        plcNameWidth = width;
        width = 0;
        for (PLCState state : PLCState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        plcStateWidth = width;
        width = 0;
        for (String name : switchNames) {
            label.setText(name + ":");
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchNameWidth = width;
        width = 0;
        for (SwitchState state : SwitchState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        switchStateWidth = width;
        width = 0;
        for (String name : latchNames) {
            label.setText(name + ":");
            width = Math.max(width, label.getPreferredSize().width);
        }
        latchNameWidth = width;
        width = 0;
        for (LatchState state : LatchState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        latchStateWidth = width;
        width = 0;
        for (String name : conditionNames) {
            label.setText(name + ":");
            width = Math.max(width, label.getPreferredSize().width);
        }
        conditionNameWidth = width;
        width = 0;
        for (ConditionState state : ConditionState.values()) {
            label.setText(state.name());
            width = Math.max(width, label.getPreferredSize().width);
        }
        conditionStateWidth = width;
    }

    final private CommandSender sender;

    //private final JPanel mainPanel = new JPanel();
    private SystemStatusPanel systemPanel;

    private final JPanel plcPanel = new JPanel();
    private final JLabel[] plcLabel = new JLabel[MpmSysState.NUM_PLCS];
    private final JLabel[] plcStatus = new JLabel[MpmSysState.NUM_PLCS];

    private final JPanel switchPanel = new JPanel();
    private final JLabel[] switchLabel = new JLabel[MpmSysState.NUM_SWITCHES];
    private final JLabel[] switchStatus = new JLabel[MpmSysState.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[MpmSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[MpmSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[MpmSysState.NUM_SWITCHES];

    private final JPanel latchPanel = new JPanel();
    private final JLabel[] latchLabel = new JLabel[MpmSysState.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[MpmSysState.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[MpmSysState.NUM_LATCHES];

    private final JPanel conditionPanel = new JPanel();
    private final JLabel[] conditionLabel = new JLabel[MpmSysState.NUM_CONDITIONS];
    private final JLabel[] conditionStatus = new JLabel[MpmSysState.NUM_CONDITIONS];

    public MpmControlPanel(CommandSender cons) {
        this.sender = cons;
        initComponents();
        (new UpdateSysStatus(new MpmSysState())).run();
        (new DisableSystem()).run();
    }

    private void initComponents() {

        Dimension d;

        // System status
        systemPanel = new SystemStatusPanel(sender);

        // PLC states
        for (int plc = 0; plc < MpmSysState.NUM_PLCS; plc++) {
            JLabel label = plcLabel[plc] = new JLabel(plcNames[plc] + ":");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = plcNameWidth;
            label.setPreferredSize(d);
            label = plcStatus[plc] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = plcStateWidth;
            label.setPreferredSize(d);
        }

        // Switch states
        for (int sw = 0; sw < MpmSysState.NUM_SWITCHES; sw++) {
            JLabel label = switchLabel[sw] = new JLabel(switchNames[sw] + ":");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = switchNameWidth;
            label.setPreferredSize(d);
            label = switchStatus[sw] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = switchStateWidth;
            label.setPreferredSize(d);
            JRadioButton buttonOff = switchOffRB[sw] = new JRadioButton("Off");
            buttonOff.setFont(UiConstants.FONT);
            //buttonOff.setText("Off");
            buttonOff.setFocusable(false);
            buttonOff.setName(Integer.toString(sw));
            buttonOff.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), false);
                }
            });
            JRadioButton buttonOn = switchOnRB[sw] = new JRadioButton("On");
            buttonOn.setFont(UiConstants.FONT);
            //buttonOn.setText("On");
            buttonOn.setFocusable(false);
            buttonOn.setName(Integer.toString(sw));
            buttonOn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), true);
                }
            });
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(buttonOff);
            bg.add(buttonOn);
        }

        // Latched condition states
        for (int cond = 0; cond < MpmSysState.NUM_LATCHES; cond++) {
            JLabel label = latchLabel[cond] = new JLabel(latchNames[cond] + ":");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = latchNameWidth;
            label.setPreferredSize(d);
            label = latchStatus[cond] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = latchStateWidth;
            label.setPreferredSize(d);
            JButton button = latchResetBtn[cond] = new JButton("Reset");
            Insets i = button.getMargin();
            i.left -= 3;
            i.right -= 3;
            button.setMargin(i);
            d = button.getPreferredSize();
            d.height -= 9;
            //d.width -= 10;
            //button.setMinimumSize(d);
            button.setPreferredSize(d);
            button.setFont(UiConstants.FONT);
            button.setFocusable(false);
            button.setName(Integer.toString(cond));
            button.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JButton)evt.getSource()).getName();
                    clearLatch(Integer.valueOf(name));
                }
            });
        }

        // Condition states
        for (int cond = 0; cond < MpmSysState.NUM_CONDITIONS; cond++) {
            JLabel label = conditionLabel[cond] = new JLabel(conditionNames[cond] + ":");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = conditionNameWidth;
            label.setMinimumSize(d);
            label.setPreferredSize(d);
            label = conditionStatus[cond] = new JLabel("XXX");
            label.setFont(UiConstants.FONT);
            d = label.getPreferredSize();
            d.width = conditionStateWidth;
            label.setMinimumSize(d);
            label.setPreferredSize(d);
        }

        // Layout the PLC panel
        TitledBorder border = BorderFactory.createTitledBorder(new LineBorder(Color.BLACK), "PLC States");
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        plcPanel.setBorder(border);
        plcPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbp = new GridBagConstraints();
        gbp.anchor = GridBagConstraints.NORTHWEST;
        gbp.gridx = 0;
        gbp.gridy = 0;
        gbp.insets.bottom = 4;
        gbp.insets.right = 5;
        int numColm = 3;
        int colm = 0;
        for (int plc = 0; plc < MpmSysState.NUM_PLCS; plc++) {
            if (colm == 0) {
                gbp.gridx = 0;
                gbp.gridy++;
            }
            gbp.insets.top = plc < numColm ? 4 : 0;
            gbp.insets.left = colm == 0 ? 5 : 35;
            plcPanel.add(plcLabel[plc], gbp);
            gbp.insets.left = 0;
            gbp.gridx++;
            plcPanel.add(plcStatus[plc], gbp);
            gbp.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Layout the switch panel
        border = BorderFactory.createTitledBorder(new LineBorder(Color.BLACK), "Switches");
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        switchPanel.setBorder(border);
        switchPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.NORTHWEST;
        gbs.gridy = 0;
        numColm = 2;
        colm = 0;
        for (int sw = 0; sw < MpmSysState.NUM_SWITCHES; sw++) {
            if (colm == 0) {
                gbs.gridx = 0;
                gbs.gridy++;
            }
            gbs.insets.top = sw < numColm ? 4 : 0;
            gbs.insets.left = colm == 0 ? 5 : 40;
            //gbs.anchor = GridBagConstraints.NORTHEAST;
            switchPanel.add(switchLabel[sw], gbs);
            gbs.insets.left = 5;
            gbs.gridx++;
            //gbs.anchor = GridBagConstraints.NORTHWEST;
            switchPanel.add(switchStatus[sw], gbs);
            gbs.gridx++;
            gbs.insets.top -= 4;
            switchPanel.add(switchOffRB[sw], gbs);
            gbs.gridx++;
            gbs.insets.left = 6;
            gbs.insets.right = 3;
            switchPanel.add(switchOnRB[sw], gbs);
            gbs.insets.right = 0;
            gbs.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Layout the latched conditions panel
        border = BorderFactory.createTitledBorder(new LineBorder(Color.BLACK), "PLC Error Conditions");
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        latchPanel.setBorder(border);
        latchPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.anchor = GridBagConstraints.NORTHWEST;
        gbl.gridy = 0;
        numColm = 2;
        colm = 0;
        for (int cond = 0; cond < MpmSysState.NUM_LATCHES; cond++) {
            if (colm == 0) {
                gbl.gridx = 0;
                gbl.gridy++;
            }
            gbl.insets.top = cond < numColm ? 4 : 0;
            gbl.insets.bottom = 4;
            gbl.insets.left = colm == 0 ? 5 : 40;
            latchPanel.add(latchLabel[cond], gbl);
            gbl.insets.left = 5;
            gbl.gridx++;
            latchPanel.add(latchStatus[cond], gbl);
            gbl.gridx++;
            gbl.insets.top -= 1;
            gbl.insets.right = 4;
            latchPanel.add(latchResetBtn[cond], gbl);
            gbl.insets.right = 0;
            gbl.gridx++;
            colm = (colm + 1) % numColm;
        }

        // Layout the conditions panel
        border = BorderFactory.createTitledBorder(new LineBorder(Color.BLACK), "PLC Running Conditions");
        border.setTitleJustification(TitledBorder.CENTER);
        border.setTitleFont(UiConstants.FONT);
        border.setTitleColor(UiConstants.BLUE);
        conditionPanel.setBorder(border);
        conditionPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.gridy = 0;
        gbc.insets.bottom = 4;
        gbc.insets.right = 5;
        numColm = 2;
        colm = 0;
        for (int cond = 0; cond < MpmSysState.NUM_CONDITIONS; cond++) {
            if (colm == 0) {
                gbc.gridx = 0;
                gbc.gridy++;
            }
            gbc.insets.top = cond < numColm ? 4 : 0;
            gbc.insets.left = colm == 0 ? 5 : 35;
            conditionPanel.add(conditionLabel[cond], gbc);
            gbc.insets.left = 0;
            gbc.gridx++;
            conditionPanel.add(conditionStatus[cond], gbc);
            gbc.gridx++;
            colm = (colm + 1) % numColm;
        }

        /*
        // Lay out the main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        mainPanel.add(systemPanel, gbm);
        gbm.gridy++;
        mainPanel.add(plcPanel, gbm);
        gbm.gridy++;
        gbm.insets.top = 6;
        mainPanel.add(switchPanel, gbm);
        gbm.gridy++;
        mainPanel.add(latchPanel, gbm);
        gbm.gridy++;
        mainPanel.add(conditionPanel, gbm);
        */

        // Lay out the complete panel
        setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        add(systemPanel, gbm);
        gbm.gridy++;
        add(plcPanel, gbm);
        gbm.gridy++;
        gbm.insets.top = 6;
        add(switchPanel, gbm);
        gbm.gridy++;
        add(latchPanel, gbm);
        gbm.gridy++;
        gbm.insets.bottom = 6;
        add(conditionPanel, gbm);
        //add(mainPanel);

    }

    private void setSwitchOn(int sw, boolean value) {
        switchOffRB[sw].setEnabled(false);
        switchOnRB[sw].setEnabled(false);
        sender.sendCommand(null, "setSwitchOn", sw, value);
    }
    
    private void clearLatch(int cond) {
        latchResetBtn[cond].setEnabled(false);
        sender.sendCommand(null, "clearLatch", cond);
    }
    
    public void updateControlPanel(MpmSysState rs) {
        SwingUtilities.invokeLater(new UpdateSysStatus(rs));
    }

    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }

    class UpdateSysStatus implements Runnable {

        private final MpmSysState vs;

        UpdateSysStatus(MpmSysState vs) {
            this.vs = vs;
        }

        @Override
        public void run() {
            systemPanel.updateStatus(vs.getTickMillis());

            for (int plc = 0; plc < MpmSysState.NUM_PLCS; plc++) {
                PLCState state = vs.getPlcState(plc);
                plcStatus[plc].setText(state.name());
                plcStatus[plc].setForeground(state == PLCState.OFFLINE ? UiConstants.BLUE :
                                             state == PLCState.DEAD ? UiConstants.RED : UiConstants.GREEN);
                plcStatus[plc].setEnabled(true);
            }
            for (int sw = 0; sw < MpmSysState.NUM_SWITCHES; sw++) {
                SwitchState state = vs.getSwitchState(sw);
                String text;
                Color color;
                text = state.name();
                color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? UiConstants.GREEN : UiConstants.BLUE;
                switchStatus[sw].setText(text);
                switchStatus[sw].setForeground(color);

                JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            for (int cond = 0; cond < MpmSysState.NUM_LATCHES; cond++) {
                LatchState state = vs.getLatch(cond);
                latchStatus[cond].setText(state.name());
                latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
                                                state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
                latchStatus[cond].setEnabled(true);
                latchResetBtn[cond].setEnabled(true);
            }
            for (int cond = 0; cond < MpmSysState.NUM_CONDITIONS; cond++) {
                ConditionState state = vs.getCondition(cond);
                conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
            }
            repaint();
        }

    }

    class DisableSystem implements Runnable {

        @Override
        public void run() {
            systemPanel.disableSystem();
            for (int plc = 0; plc < MpmSysState.NUM_PLCS; plc++) {
                plcStatus[plc].setEnabled(false);
            }
            for (int sw = 0; sw < MpmSysState.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < MpmSysState.NUM_LATCHES; cond++) {
                latchStatus[cond].setEnabled(false);
                latchResetBtn[cond].setEnabled(false);
            }
            for (int cond = 0; cond < MpmSysState.NUM_CONDITIONS; cond++) {
                conditionStatus[cond].setEnabled(false);
            }
            repaint();
        } 
    }

    private static final long serialVersionUID = 1L;
}
