package org.lsst.ccs.subsystem.utility.constants;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 * Utility trunk fan definitions.
 *
 * @author The CCS team
 */
public class UtilTrunkFans {

    public static final String
        FAN_UT  = "UtFan",
        FAN_VPC = "VpcFan",
        FAN_MPC = "MpcFan";

    public static final int
        FAN_UT_ID  = 0,
        FAN_VPC_ID = 1,
        FAN_MPC_ID = 2,
        NUM_FANS = 3;

    private static final Map<Integer, Integer> fanSwitchMap = new LinkedHashMap<>();
    static {
        //fanSwitchMap.put(FAN_MPC_ID, UtilTrunkSwitches.SW_PDU_MPC_FAN);
    }

    private static final Map<String, Integer> fanNameToId = new LinkedHashMap<>();
    static {
        fanNameToId.put(FAN_UT, FAN_UT_ID);
        fanNameToId.put(FAN_VPC, FAN_VPC_ID);
        fanNameToId.put(FAN_MPC, FAN_MPC_ID);
    }

    private static final Map<Integer, String> fanIdToName = new LinkedHashMap<>();
    static {
        for (Map.Entry entry : fanNameToId.entrySet()) {
            fanIdToName.put((Integer)entry.getValue(), (String)entry.getKey());
        }
    }

    private static final boolean[] hasTempState = new boolean[NUM_FANS];
    static {
        hasTempState[FAN_UT_ID] = true;
        hasTempState[FAN_VPC_ID] = false;
        hasTempState[FAN_MPC_ID] = true;
    }

    /**
     *  Gets the list of fan names.
     * 
     *  @return  The list of names
     */
    public static List<String> getNames()
    {
        return new ArrayList<>(fanNameToId.keySet());
    }

    /**
     *  Gets a fan name from its ID.
     * 
     *  @param  id  The fan id
     *  @return  The fan name
     */
    public static String getName(int id)
    {
        return fanIdToName.get(id);
    }

    /**
     *  Gets a fan ID from its name.
     * 
     *  @param  name  The fan name
     *  @return  The fan ID
     *  @throws  UtilityException
     */
    public static int getId(String name) throws UtilityException
    {
        Integer id = fanNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid fan name: " + name);
        }
        return id;
    }

    /**
     *  Gets a fan switch from its ID.
     * 
     *  @param  id  The fan ID
     *  @return  The fan switch ID
     */
    public static Integer getSwitch(int id)
    {
        return fanSwitchMap.get(id);
    }

    /**
     *  Gets whether a fan has a temperature control state.
     * 
     *  @param  id  The fan id
     *  @return  Whether it has a TEMP state
     */
    public static boolean hasTempState(int id)
    {
        return hasTempState[id];
    }

}
