package org.lsst.ccs.subsystem.utility.constants;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Definitions of the MPM system latches.
 *
 *  @author The CCS team
 */
public class MpmLatches {

    public static final int
        LATCH_UT_TEMP        = 0,
        LATCH_UT_LEAK        = 1,
        LATCH_UT_LEAK_FAULT  = 2,
        LATCH_UT_SMOKE       = 3,
        LATCH_UT_SMOKE_FAULT = 4,
        LATCH_COLD_TEMP_HIGH = 5,
        LATCH_COLD_TEMP_LOW  = 6,
        LATCH_CRYO_TEMP_HIGH = 7,
        LATCH_CRYO_TEMP_LOW  = 8,
        LATCH_CRYO_VACUUM    = 9,
        LATCH_HEX_VACUUM     = 10,
        NUM_LATCHES          = 11;

    private static final Map<String, Integer> NAME_MAP = new LinkedHashMap<>();
    static {
        NAME_MAP.put("UTTemp", LATCH_UT_TEMP);
        NAME_MAP.put("UTLeak", LATCH_UT_LEAK);
        NAME_MAP.put("UTLeakFault", LATCH_UT_LEAK_FAULT);
        NAME_MAP.put("UTSmoke", LATCH_UT_SMOKE);
        NAME_MAP.put("UTSmokeFault", LATCH_UT_SMOKE_FAULT);
        NAME_MAP.put("ColdTempHigh", LATCH_COLD_TEMP_HIGH);
        NAME_MAP.put("ColdTempLow", LATCH_COLD_TEMP_LOW);
        NAME_MAP.put("CryoTempHigh", LATCH_CRYO_TEMP_HIGH);
        NAME_MAP.put("CryoTempLow", LATCH_CRYO_TEMP_LOW);
        NAME_MAP.put("CryoVacuum", LATCH_CRYO_VACUUM);
        NAME_MAP.put("HexVacuum", LATCH_HEX_VACUUM);
    }

    private static final Map<Integer, String> ID_MAP= new LinkedHashMap<>();
    static {
        for (Map.Entry e : NAME_MAP.entrySet()) {
            ID_MAP.put((int)e.getValue(), (String)e.getKey());
        }
    }

    /**
     *  Gets the list of latch names.
     * 
     *  @return  The list of names
     */
    public static List<String> getNames()
    {
        return new ArrayList<>(NAME_MAP.keySet());
    }

    /**
     *  Gets a latch name from its ID.
     * 
     *  @param  id  The latch id
     *  @return  The latch name
     */
    public static String getName(int id) {
        return ID_MAP.get(id);
    }

    /**
     *  Gets a latch ID from its name.
     * 
     *  @param  name  The latch name
     *  @return  The latch ID
     *  @throws  UtilityException
     */
    public static int getId(String name) throws UtilityException {
        Integer id = NAME_MAP.get(name);
        if (id == null) {
            throw new UtilityException("Invalid latch name: " + name);
        }
        return id;
    }

}
