package org.lsst.ccs.subsystem.utility.constants;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 *  Definitions of the MPM system switches.
 *
 *  @author The CCS team
 */
public class MpmSwitches {

    public static final int
        SW_BLOCK_UT_POWER   = 0,
        SW_BLOCK_REB_POWER  = 1,
        SW_BLOCK_COOLANT    = 2,
        SW_BLOCK_COLD_REFG  = 3,
        SW_BLOCK_COLD_HEAT  = 4,
        SW_BLOCK_CRYO_REFG  = 5,
        SW_BLOCK_CRYO_HEAT  = 6,
        NUM_SWITCHES        = 7;

    private static final Map<String, Integer> NAME_MAP = new LinkedHashMap<>();
    static {
        NAME_MAP.put("BlockColdHeat", SW_BLOCK_COLD_HEAT);
        NAME_MAP.put("BlockColdRefg", SW_BLOCK_COLD_REFG);
        NAME_MAP.put("BlockCryoHeat", SW_BLOCK_CRYO_HEAT);
        NAME_MAP.put("BlockCryoRefg", SW_BLOCK_CRYO_REFG);
        NAME_MAP.put("BlockCoolant", SW_BLOCK_COOLANT);
        NAME_MAP.put("BlockRebPower", SW_BLOCK_REB_POWER);
        NAME_MAP.put("BlockUtPower", SW_BLOCK_UT_POWER);
    }

    private static final Map<Integer, String> ID_MAP= new LinkedHashMap<>();
    static {
        for (Map.Entry e : NAME_MAP.entrySet()) {
            ID_MAP.put((int)e.getValue(), (String)e.getKey());
        }
    }

    /**
     *  Gets the list of switch names.
     * 
     *  @return  The list of names
     */
    public static List<String> getNames()
    {
        return new ArrayList<>(NAME_MAP.keySet());
    }

    /**
     *  Gets a switch name from its ID.
     * 
     *  @param  id  The switch id
     *  @return  The switch name
     */
    public static String getName(int id) {
        return ID_MAP.get(id);
    }

    /**
     *  Gets a switch ID from its name.
     * 
     *  @param  name  The switch name
     *  @return  The switch ID
     *  @throws  UtilityException
     */
    public static int getId(String name) throws UtilityException {
        Integer id = NAME_MAP.get(name);
        if (id == null) {
            throw new UtilityException("Invalid switch name: " + name);
        }
        return id;
    }

}
