package org.lsst.ccs.subsystem.utility.constants;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.utility.data.UtilityException;

/**
 * Utility trunk heater definitions.
 *
 * @author The CCS team
 */
public class UtilTrunkHeaters {

    public static final String
        HEATER_VPC1 = "VpcHeater1",
        HEATER_VPC2 = "VpcHeater2";

    public static final int
        HEATER_VPC1_ID = 0,
        HEATER_VPC2_ID = 1,
        NUM_VPC_HEATERS = 2,
        NUM_HEATERS = 2;

    private static final Map<Integer, Integer> heaterSwitchMap = new LinkedHashMap<>();
    static {
        heaterSwitchMap.put(HEATER_VPC1_ID, UtilTrunkSwitches.SW_BFR_HEATER_1);
        heaterSwitchMap.put(HEATER_VPC2_ID, UtilTrunkSwitches.SW_BFR_HEATER_2);
    }

    private static final Map<String, Integer> heaterNameToId = new LinkedHashMap<>();
    static {
        heaterNameToId.put(HEATER_VPC1, HEATER_VPC1_ID);
        heaterNameToId.put(HEATER_VPC2, HEATER_VPC2_ID);
    }

    private static final Map<Integer, String> heaterIdToName = new LinkedHashMap<>();
    static {
        for (Map.Entry entry : heaterNameToId.entrySet()) {
            heaterIdToName.put((Integer)entry.getValue(), (String)entry.getKey());
        }
    }

    /**
     *  Gets the list of heater names.
     * 
     *  @return  The list of names
     */
    public static List<String> getNames()
    {
        return new ArrayList<>(heaterNameToId.keySet());
    }

    /**
     *  Gets a heater name from its ID.
     * 
     *  @param  id  The heater id
     *  @return  The heater name
     */
    public static String getName(int id)
    {
        return heaterIdToName.get(id);
    }

    /**
     *  Gets a heater ID from its name.
     * 
     *  @param  name  The heater name
     *  @return  The heater ID
     *  @throws  UtilityException
     */
    public static int getId(String name) throws UtilityException
    {
        Integer id = heaterNameToId.get(name);
        if (id == null) {
            throw new UtilityException("Invalid heater name: " + name);
        }
        return id;
    }

    /**
     *  Gets a heater switch from its ID.
     * 
     *  @param  id  The heater ID
     *  @return  The heater switch ID
     */
    public static int getSwitch(int id)
    {
        return heaterSwitchMap.get(id);
    }

}
