package org.lsst.ccs.subsystem.ts8;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import nom.tam.fits.FitsException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.daq.utilities.FitsService;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.rafts.GlobalProc;
import org.lsst.ccs.subsystem.rafts.REBDevice;
import org.lsst.ccs.subsystem.rafts.RaftsCommands;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.DACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.ImageState;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.RaftState;
import org.lsst.ccs.subsystem.rafts.data.RaftsAgentProperties;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.GeometryUtils;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.drivers.reb.SlowAdcs;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.Monitor.AlarmHandler;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.rafts.TempControl;
import org.lsst.ccs.subsystem.rafts.states.SequencerMainState;
import org.lsst.ccs.utilities.image.FitsHeadersSpecifications;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * The TS8Subsystem is designed specifically for running test stand 8 at BNL,
 * and the equivalent test stand at SLAC for I&T. It makes use of the Rafts
 * subsystem, which contains the code common to all use of the DAQ system both
 * for test stands and (eventually) for the construction of the full focal plane
 * during I&T and eventually the operation of the camera.
 *
 * TOOD: Review whether the split between code which is in the TS8Subsystem and
 * the rafts subsystem is consistent with the division described above.
 */
public class TS8Subsystem implements HasLifecycle, Monitor.AlarmHandler {

    private final FitsUtilities fitsUtils = new FitsUtilities();

    private static final Logger LOG = Logger.getLogger("org.lsst.ccs.subsystem.ts8");

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final List<REBDevice> rebDevices = new ArrayList();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, REBDevice> rebDevicesMap = new HashMap();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final List<TempControl> tempCtrlList = new ArrayList();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private AlarmHandler alarmHandler;

    // TODO: The comment below seems confusing because there are only two protected variables, and in 
    // general variables to do not to be protected for testing.
    // TODO: The outputDirectory and flieNamePattern are just defaults, I think they can be overriden. Should check and if
    // so give them more appropriate names.
    //The next three variables are protected so that they can be tested
    protected final String outputDirectory = "${output_base_dir}/rafts/${raftId}/${runId}/${acq_type}/${test_version}/${job_id}/s${sensorLoc}";
    protected final String fileNamePattern = "${CCDSerialLSST}_${TestType}_${ImageType}_${SequenceInfo}_${RunNumber}_${timestamp}.fits";

    @ConfigurationParameter(isFinal = true)
    private String shutterREB = "R00.Reb0";
    @ConfigurationParameter(isFinal = true)
    private String xedREB = "R00.Reb1";

    @ConfigurationParameter(category = REBDevice.RAFTS)
    private String ccdType = null; // The CCD type

    private final Geometry geometry;
    // TODO: Why do we have sLog and log above?
    private final Logger sLog;

    //Image acquisition default values
    private long exposureTimeInMillis = 2000;
    private boolean openShutter = true;
    private boolean actuateXED = false;

    private String imageType = "";
    
    private final Map<String, ChanParams> monChans = new HashMap<>();

    private final ClientFactory clientFactory;
    private RaftsCommands raftsCommands;
    private String last_alertStr;

    private final GlobalProc globalProc = new GlobalProc();

    private TempControl tempCtrl;

    private boolean useInfoAlerts = System.getProperty("org.lsst.ccs.subsystem.ts8PowerInfoAlerts", "false").contains("true");
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    AgentStatusAggregatorService aggregatorService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    /**
     * Main constructor for the TS8Subsystem, normally invoked from the
     * corresponding groovy file.
     *
     * @param geometry The initial geometry. This is used for ???
     * @param clientFactory The ClientFactoryused to create the interfaces to
     * the DAQ, or <code>null</code> to use the default ClientFactory. See
     * {@link org.lsst.ccs.subsystem.ts8.sim.TS8ClientFactorySimulation} for an
     * alternative factory which can be used for simulations.
     */
    public TS8Subsystem(Geometry geometry, ClientFactory clientFactory) {

        this.geometry = geometry;
        this.clientFactory = clientFactory;

        sLog = Logger.getLogger("org.lsst.ccs.subsystem.ts8");
    }

    @Override
    public void postInit() {

        // By setting RAFT_TYPE_AGENT_PROPERTY we signal to consoles that this subsystem is compatible with the rafts subsystm GUI
        subsys.getAgentService(AgentPropertiesService.class).setAgentProperty(RaftsAgentProperties.RAFT_TYPE_AGENT_PROPERTY, TS8Subsystem.class.getCanonicalName());

        // Get temperature control object
        if (!tempCtrlList.isEmpty()) {
            tempCtrl = (TempControl) tempCtrlList.get(0);
            tempCtrl.initialize(sLog);
        } else {
            sLog.info("No temperature control available");
        }

        raftsCommands = new RaftsCommands(subsys, rebDevices, globalProc, geometry, tempCtrl);

        //Add the RaftCommands to the CommandSet
        subsys.getAgentService(AgentCommandDictionaryService.class).addCommandSetToObject(raftsCommands, "");


        sLog.info("Configured the status aggregator to listen to subsystems: " + System.getProperty("org.lsst.ccs.subsystem.teststand", "ts") + " " + System.getProperty("org.lsst.ccs.subsystem.rebps", "ccs-rebps") + " " + System.getProperty("org.lsst.ccs.subsystem.ts8", "ts8"));

        // Initialize for triggering
        globalProc.initialize(rebDevices, clientFactory, sLog);

        //TO-DO, Why is this needed?
        Properties filePatternProperties = BootstrapResourceUtils.getBootstrapProperties("output_location");
        for (Object key : BootstrapResourceUtils.getAllKeysInProperties(filePatternProperties)) {
            String propertyName = (String) key;
            raftsCommands.setPatternProperty(propertyName, filePatternProperties.getProperty(propertyName));
        }

        raftsCommands.setFitsFileNamePattern(fileNamePattern);
        raftsCommands.setDefaultImageDirectory(outputDirectory);

        // setup channel data
        monChans.put("Temp1", new ChanParams(REBDevice.TYPE_BD_TEMP, 0));
        monChans.put("Temp2", new ChanParams(REBDevice.TYPE_BD_TEMP, 1));
        monChans.put("Temp3", new ChanParams(REBDevice.TYPE_BD_TEMP, 2));
        monChans.put("Temp4", new ChanParams(REBDevice.TYPE_BD_TEMP, 3));
        monChans.put("Temp5", new ChanParams(REBDevice.TYPE_BD_TEMP, 4));
        monChans.put("Temp6", new ChanParams(REBDevice.TYPE_BD_TEMP, 5));
        monChans.put("Temp7", new ChanParams(REBDevice.TYPE_BD_TEMP, 6));
        monChans.put("Temp8", new ChanParams(REBDevice.TYPE_BD_TEMP, 7));
        monChans.put("Temp9", new ChanParams(REBDevice.TYPE_BD_TEMP, 8));
        monChans.put("Temp10", new ChanParams(REBDevice.TYPE_BD_TEMP, 9));
        monChans.put("Atemp2U", new ChanParams(REBDevice.TYPE_ASP_TEMP, 4));
        monChans.put("Atemp1U", new ChanParams(REBDevice.TYPE_ASP_TEMP, 2));
        monChans.put("Atemp0U", new ChanParams(REBDevice.TYPE_ASP_TEMP, 0));
        monChans.put("Atemp2L", new ChanParams(REBDevice.TYPE_ASP_TEMP, 5));
        monChans.put("Atemp1L", new ChanParams(REBDevice.TYPE_ASP_TEMP, 3));
        monChans.put("Atemp0L", new ChanParams(REBDevice.TYPE_ASP_TEMP, 1));
        monChans.put("CCDTemp0", new ChanParams(REBDevice.TYPE_RTD, 0));
        monChans.put("CCDTemp1", new ChanParams(REBDevice.TYPE_RTD, 1));
        monChans.put("CCDTemp2", new ChanParams(REBDevice.TYPE_RTD, 2));
        monChans.put("RTDTemp", new ChanParams(REBDevice.TYPE_RTD, 3));
        monChans.put("DigI", new ChanParams(REBDevice.TYPE_BD_POWER, 1));
        monChans.put("AnaI", new ChanParams(REBDevice.TYPE_BD_POWER, 3));
        monChans.put("ClkHI", new ChanParams(REBDevice.TYPE_BD_POWER, 5));
        monChans.put("ClkLI", new ChanParams(REBDevice.TYPE_BD_POWER, 9));
        monChans.put("ODI", new ChanParams(REBDevice.TYPE_BD_POWER, 7));

        monChans.put("OD0V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OD_0));
        monChans.put("OG0V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OG_0));
        monChans.put("RD0V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_RD_0));
        monChans.put("GD0V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_GD_0));

        monChans.put("OD1V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OD_1));
        monChans.put("OG1V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OG_1));
        monChans.put("RD1V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_RD_1));
        monChans.put("GD1V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_GD_1));

        monChans.put("OD2V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OD_2));
        monChans.put("OG2V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OG_2));
        monChans.put("RD2V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_RD_2));
        monChans.put("GD2V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_GD_2));

        for (int iccd = 0; iccd < 3; iccd++) {
            for (int ihalf = 0; ihalf < 2; ihalf++) {
                String tb = ihalf == 0 ? "upper" : "lower";
                for (int jc = 0; jc < 8; jc++) {
                    int ch = jc + 8 * ihalf + 16 * iccd;
                    String name = "CCDI" + iccd + ihalf + jc;
                    monChans.put(name, new ChanParams(REBDevice.TYPE_CCD_CURR, ch));
                }
            }
        }
    }

    @Override
    public void build() {
        //Add a periodic task to check if Sequencers are running
        //This will be replaced once the DAQ v2 will be able to call us back
        //when the sequencer is done running
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("sequencerMonitor",
                () -> {
                    checkSequencerState();
                }).withIsFixedRate(false).withPeriod(Duration.ofSeconds(5)));
    }

    private void checkSequencerState() {
        if (raftsCommands == null) {
            throw new RuntimeException("RaftsCommands have not been initialized");
        }
        raftsCommands.checkSequencerState();
    }

    @Override
    public void postStart() {
        sLog.info("TS8 subsystem started");
        //At start we don't know what is the main state of the Sequencer
        //so we set it to UNKNOWN
        subsys.getAgentService(AgentStateService.class).updateAgentState(SequencerMainState.UNKNOWN);
        if (ccdType == null) {
            raftsCommands.setCcdType(((Raft) geometry).getType().getName());
        } else {
            raftsCommands.changeCcdType(ccdType);
        }
    }

    @ConfigurationParameterChanger
    public void setCcdType(String ccdType) {
        //Should setCcdType throw an exception?
        //NOTE: this method could be invoked right at startup, when the raftsCommands
        //object has not been built yet. So we have to check for that and invoke
        //the same method in the postStart method to make sure it's invoked 
        //after the ccdType has been invoked during startup.
        if (raftsCommands != null) {
            raftsCommands.changeCcdType(ccdType);
        }
        this.ccdType = ccdType;
    }

    @Override
    public boolean processAlarm(int event, int parm, String cause, String alarmName) {
        if (alarmHandler != null) {
            return alarmHandler.processAlarm(event, parm, cause, alarmName);
        }
        return false;
    }

    @Command(description = "Print Header specifications", type = Command.CommandType.QUERY)
    @Deprecated
    public String printHeaderSpecifications() {
        StringBuilder sb = new StringBuilder();
        Map<String, HeaderSpecification> config = FitsHeadersSpecifications.getHeaderSpecifications();
        for (String header : config.keySet()) {
            sb.append("***************************\n");
            sb.append("     Header: ").append(header).append("\n");
            sb.append("***************************\n");
            HeaderSpecification spec = config.get(header);
            for (HeaderLine line : spec.getHeaders()) {
                sb.append("## ").append(line.getKeyword()).append(" ").append(line.getMetaName()).append(" ").append(line.getDataType()).append(" ").append(line.getComment()).append("\n");
            }
        }
        return sb.toString();
    }

    @Command(description = "Print Geometry", type = Command.CommandType.QUERY)
    public String printGeometry(@Argument(defaultValue = "0") int depth) {
        return GeometryUtils.showGeometryTree(geometry, depth);
    }

    /**
     * Set the name of the raft. This will be called by the script running TS8
     * using data obtained from the eTraveler, and will enable appropriate
     * headers to be added to the FitsFile generated.
     *
     * @param raftName The name of the raft.
     */
    @Command(description = "Set the name of the raft", type = Command.CommandType.QUERY)
    public void setRaftName(String raftName) {
        setHeader("RaftName", raftName, true);
    }

    /**
     * Set the name of the reb at the given parallel position. This will be
     * called by the script running TS8 using data obtained from the eTraveler,
     * and will enable appropriate headers to be added to the FitsFile
     * generated.
     *
     * @param parallel The parallel position of the Reb.
     * @param rebName The new name of the reb
     * @throws IllegalArgumentException if the parallel position is illegal.
     */
    @Command(description = "Set the name of the reb at a given parallel position", type = Command.CommandType.QUERY)
    public void setRebName(int parallel, String rebName) throws IllegalArgumentException {
        Reb reb = (Reb) geometry.getChild(parallel, 0);
        if (reb == null) {
            throw new RuntimeException("Could not find Reb for parallel position " + parallel);
        }
        for (CCD ccd : reb.getChildrenList()) {
            setCCDHeader(ccd.getUniqueId(), "RebName", rebName, true);
        }
    }

    /**
     * Set the RunNumber.
     *
     * @param runNumber The run Number
     */
    @Command(description = "Set the run number", type = Command.CommandType.QUERY)
    public void setRunNumber(String runNumber) {
        setHeader("RunNumber", runNumber, true);
    }

    /**
     * Clear REB config
     *
     * @param rebnum The number of the REB to have its config cleared
     * @return A String representing the state of the devices after they are
     * powered off.
     */
    @Command(type = Command.CommandType.ACTION, description = "clear REB config for a given REB ID")
    public String clearREBCfg(int rebnum) throws Exception {
        StringBuilder outstr = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() == rebnum) {
                outstr.append(resetREBCfg(rebDevice));
            }
        }
        return outstr.toString();
    }

    /**
     * Clear REB config
     *
     * @param rebDevice The REB device to have its config cleared
     * @return A String representing the state of the devices after they are
     * powered off.
     */
    @Command(type = Command.CommandType.ACTION, description = "reset REB config")
    public String resetREBCfg(REBDevice rebDevice) throws Exception {

        LOG.info("state before reset = " + this.showREBCfg(rebDevice));

        REB reb = rebDevice.getREBConfig();
        BiasDACS[] biases = reb.getBiases();
        for (BiasDACS bias : biases) {
            double[] values = bias.getPValues();
            values[BiasDACS.GD] = 0.;
            values[BiasDACS.OD] = 0.;
            values[BiasDACS.OG] = 0.;
            values[BiasDACS.RD] = 0.;
            values[BiasDACS.CS_GATE] = 0.;
        }

        DACS dacs = reb.getDacs();

        double[] dacvalues = dacs.getPValues();
        dacvalues[DACS.PCLK_HIGH] = 0.;
        dacvalues[DACS.PCLK_LOW] = 0.;
        dacvalues[DACS.SCLK_HIGH] = 0.;
        dacvalues[DACS.SCLK_LOW] = 0.;
        dacvalues[DACS.RG_HIGH] = 0.;
        dacvalues[DACS.RG_LOW] = 0.;

        LOG.info("setting new config");
        rebDevice.setREBConfig(reb);

        String msg = "state after reset:\n" + this.showREBCfg(rebDevice);
        return msg;
    }

    @Command(type = Command.CommandType.QUERY, description = "raise subsystem alert")
    public void raiseAlert(String alertmsg, AlertState severity) {
        org.lsst.ccs.bus.data.Alert a = new org.lsst.ccs.bus.data.Alert("TS8 Alert", "information");
        if (!(severity == AlertState.NOMINAL) || (severity == AlertState.NOMINAL && this.useInfoAlerts)) {
            for (String line : alertmsg.split("\n")) {
                alertService.raiseAlert(a, severity, line.substring(0, Math.min(254, line.length())));
            }
        }
        last_alertStr = alertmsg;
    }

    /**
     * Get the configuration for all REB devices
     *
     * @return A string representing the configuration of the devices.
     */
    @Command(type = Command.CommandType.ACTION, description = "show REB config for all REBS")
    public String showAllREBCfg() {
        StringBuilder outstr = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            outstr.append(showREBCfg(rebDevice));
        }
        return outstr.toString();
    }

    @Command(type = Command.CommandType.ACTION, description = "show REB config for a given REB ID")
    public String showREBCfg(int rebnum) {
        StringBuilder outstr = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() == rebnum) {
                outstr.append(showREBCfg(rebDevice));
            }
        }
        return outstr.toString();
    }

    /**
     * Get the configuration of a particular REB device.
     *
     * @param rebDevice The device for which the configuration is to be
     * obtained.
     * @return A string representing the device's configuration.
     */
//    @Command(type = Command.CommandType.ACTION, description = "show REB config")
    private String showREBCfg(REBDevice rebDevice) {

        StringBuilder newcfgsstr = new StringBuilder("\nShowing config for REB:").append(rebDevice.getRebNumber()).append(" : \n");

        REB reb = rebDevice.getREBConfig();
        BiasDACS[] biases = reb.getBiases();
        int i_ccd = 0;
        for (BiasDACS bias : biases) {
            // TODO: i_ccd is always going to be zero, this can't be what was intended??
            newcfgsstr.append("CCD").append(i_ccd).append(":\n");
            newcfgsstr.append("The BiasDACS voltage settings are (V):\n")
                    .append("GD=").append(bias.getPValues()[BiasDACS.GD]).append(" ,\n")
                    .append("RD=").append(bias.getPValues()[BiasDACS.RD]).append(" ,\n")
                    .append("OD=").append(bias.getPValues()[BiasDACS.OD]).append(" ,\n")
                    .append("OG=").append(bias.getPValues()[BiasDACS.OG]).append(" ,\n");
            i_ccd++;
        }

        DACS dacs = reb.getDacs();

        newcfgsstr.append("new DACS config:\n" + "PCLK_HIGH=").append(dacs.getPValues()[DACS.PCLK_HIGH]).append(" ,\n")
                .append("PCLK_LOW=").append(dacs.getPValues()[DACS.PCLK_LOW]).append(" ,\n")
                .append("SCLK_HIGH=").append(dacs.getPValues()[DACS.SCLK_HIGH]).append(" ,\n")
                .append("SCLK_LOW=").append(dacs.getPValues()[DACS.SCLK_LOW]).append(" ,\n")
                .append("RG_HIGH=").append(dacs.getPValues()[DACS.RG_HIGH]).append(" ,\n")
                .append("RG_LOW=").append(dacs.getPValues()[DACS.RG_LOW]).append(" ,\n");

        i_ccd++;
        return newcfgsstr.toString();
    }

    /**
     * Choose the test voltage to use based on the target voltage
     */
    public double chooseTestVolt(double proposedVoltage, double full_target) {
        return (Math.abs(proposedVoltage) < Math.abs(full_target) ? Math.copySign(proposedVoltage, full_target) : full_target / 5.0);
    }

    private String dateStr() {
        return (new SimpleDateFormat("yyyy-MM-dd-HH:mm:ss").format(new Date(System.currentTimeMillis())));
    }

    /**
     * Read a register address on a Reb.
     *
     * @param rebId The rebId: "Reb0", "R11:Reb0"
     * @param address The address of the register to read.
     * @return The value of the register.
     */
    @Command(description = "Read a register address", type = Command.CommandType.QUERY)
    public int readRegister(String rebId, int address) {
        Object reb = rebDevicesMap.get(rebId);
        if (reb == null || !(reb instanceof REBDevice)) {
            throw new IllegalArgumentException("Reb id : " + rebId + " is not valid.");
        }
        try {
            return ((REBDevice) reb).getRegister(address, 1).getValues()[0];
        } catch (Exception x) {
            throw new RuntimeException("Error during register read ", x);
        }
    }

    /**
     * loadSequence with modified geometry
     *
     * @param seq
     * @param prerows
     * @param readrows
     * @param postrows
     * @param precols
     * @param readcols
     * @param postcols
     * @return
     * @throws java.lang.Exception
     */
    @Command(description = "load sequencer and adjust row and col parameters")
    public List<Integer> loadSequencerAndReDimension(
            @Argument(description = "sequencer filename") String seq,
            @Argument(description = "Number of rows to skip before window") int prerows,
            @Argument(description = "Number of rows of the window") int readrows,
            @Argument(description = "Number of rows after window") int postrows,
            @Argument(description = "Number of columns to skip before readout window, including prescan") int precols,
            @Argument(description = "Number of columns to read") int readcols,
            @Argument(description = "Number of columns to discard after window ") int postcols) throws Exception {
        List<Integer> seqrply = raftsCommands.loadSequencer(seq);
        getRaftsCommands().setSequencerParameter("PreRows", prerows);
        getRaftsCommands().setSequencerParameter("ReadRows", readrows);
        getRaftsCommands().setSequencerParameter("PostRows", postrows);
        getRaftsCommands().setSequencerParameter("PreCols", precols);
        getRaftsCommands().setSequencerParameter("ReadCols", readcols);
        getRaftsCommands().setSequencerParameter("PostCols", postcols);

        return (seqrply);
    }

    /**
     * Write a register address on a Reb.
     *
     * @param rebId The id of the Reb: "Reb0", "R11:Reb0"
     * @param address The address to write to.
     * @param value The value to set the address to.
     */
    @Command(description = "Write a register address", type = Command.CommandType.ACTION)
    public void writeRegister(String rebId, int address, int value) {
        Object reb = rebDevicesMap.get(rebId);
        if (reb == null || !(reb instanceof REBDevice)) {
            throw new IllegalArgumentException("Reb id : " + rebId + " is not valid.");
        }
        try {
            ((REBDevice) reb).setRegister(address, new int[]{value});
        } catch (Exception x) {
            throw new RuntimeException("Error during register write ", x);
        }
    }

    /**
     * Set a key for all the fits extensions.
     *
     * @param headerName The name of the Header to set
     * @param headerValue The corresponding value
     * @param sticky Boolean value to specify if the provided Header keyword
     * value should be used across exposures. The default value is true. If
     * false is provided, the provided Header keyword value will be reset after
     * each exposure.
     */
    @Command(description = "Set a header value for all extensions", type = Command.CommandType.QUERY)
    public void setHeader(String headerName, Object headerValue,
            @Argument(defaultValue = "true") boolean sticky) {
        raftsCommands.setHeaderKeyword(FitsService.COMMON_HEADER_NAME,headerName,(Serializable)headerValue,sticky);
    }

    /**
     * Set a key for the header of a given CCD identified by its id. The
     * Id could be an enumeration or a string ("S00", "R01:S00").
     *
     * @param ccdId The CCD id
     * @param headerName The name of the Header to set
     * @param headerValue The corresponding value
     * @param sticky Boolean value to specify if the provided Header keyword
     * value should be used across exposures. The default value is true. If
     * false is provided, the provided Header keyword value will be reset after
     * each exposure.
     */
    @Command(description = "Set a header value for a given CCD", type = Command.CommandType.QUERY)
    public void setCCDHeader(String ccdId, String headerName, Object headerValue,
            @Argument(defaultValue = "true") boolean sticky) {
        raftsCommands.setHeaderKeywordForCCD(ccdId, headerName, (Serializable)headerValue, sticky);        
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave() throws IOException, InterruptedException, REBException {
        return exposeAcquireAndSave(exposureTimeInMillis);
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave(long exposureTimeInMillis) throws IOException, InterruptedException, REBException {
        return exposeAcquireAndSave(exposureTimeInMillis, openShutter);
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @param openShutter True/false if the shutter should be open
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave(long exposureTimeInMillis, boolean openShutter)
            throws IOException, InterruptedException, REBException {
        return exposeAcquireAndSave(exposureTimeInMillis, openShutter, actuateXED);
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @param openShutter True/false if the shutter should be open
     * @param actuateXED True/false if the FE55 should be involved
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave(long exposureTimeInMillis, boolean openShutter,
            boolean actuateXED) throws IOException, InterruptedException, REBException {
        return exposeAcquireAndSave(exposureTimeInMillis, openShutter, actuateXED, null);
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @param openShutter True/false if the shutter should be open
     * @param actuateXED True/false if the FE55 should be involved
     * @param filePattern The filePattern for the output fits files
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave(long exposureTimeInMillis,
            boolean openShutter, boolean actuateXED,
            String filePattern) throws REBException, IOException, InterruptedException {

        LOG.info("Exposure requested: exptime = " + exposureTimeInMillis + ", shutter = " + openShutter + ", XED = " + actuateXED);
        List<Map<String, Integer>> mainmap = getRaftsCommands().getMainMap();
        boolean hasExposeMain = false;
        boolean hasBiasMain = false;
        boolean hasPocketPumpMain = false;
        boolean hasAcquireMain = false;


        setHeader("TemperatureSetPoint", raftsCommands.getControlTemp(), true);

        if ( imageType == null || imageType.isEmpty() ) {
            LOG.info("No image type set.");
        }

        for (REBDevice rebDevice : rebDevices) {
            setRebName(rebDevice.getRebNumber(), rebDevice.getName());
        }

        String mainRoutine = raftsCommands.getSequencerStart();

        for (Map<String, Integer> mymain : mainmap) {
            if (mymain.keySet().contains("PocketPump")) {
                hasPocketPumpMain = true;
            }
            if (mymain.keySet().contains("Expose")) {
                hasExposeMain = true;
            }
            if (mymain.keySet().contains("Acquire")) {
                hasAcquireMain = true;
            }
            if (mymain.keySet().contains("Bias")) {
                hasBiasMain = true;
            }
        }

        boolean runPreAcquisition = exposureTimeInMillis > 0;
        if (hasPocketPumpMain && hasExposeMain && hasAcquireMain && hasBiasMain) {
            if (exposureTimeInMillis > 0) {
                mainRoutine = "Acquire";
            } else {
                LOG.info("selecting bias mains");
                mainRoutine = "Bias";
            }
        } else {
            throw new RuntimeException("Required main routine missing!");
        }
        Predicate<BusMessage<?, ?>> imageStateStatusListenerFilter = BusMessageFilterFactory
                .messageClass(StatusSubsystemData.class).and(BusMessageFilterFactory.messageOrigin(subsys.getName()));
        ImageStatusDataListener imageStateListener = new ImageStatusDataListener(rebDevices.size());
        subsys.getMessagingAccess().addStatusMessageListener(imageStateListener,
                imageStateStatusListenerFilter);

        List<String> fitsFiles = new ArrayList<>();
        try {

//        if (System.getProperty("org.lsst.ccs.run.mode", "").equals("simulation")) {
//            // For now generate an image here. This will be moved somewhere else when the
//            // Sequencer loading works.
//            GeneratedImage generatedImage = PatternGeneratorFactory.generateImageForGeometry(geometry, "image", null);
//            PatternGeneratorFactory.exposeGeometryToGeneratedImage(geometry, generatedImage);
//        }
            //Start Acquisition
            try {
                getRaftsCommands().setSequencerParameter("ExposureTime", (int) (exposureTimeInMillis / 25.0)); // units are 25 ms        
                getRaftsCommands().setExposureTime(exposureTimeInMillis / 1000.); // set exposure time value in the metadata
                LOG.info("ExposureTime " + (int) (exposureTimeInMillis / 25.0) + " x 25ms periods");
            } catch (Exception e) {
                LOG.warn("Could not set exposure time on sequencer");
            }

            List<SequencerProc> sequencersToStartBeforeAcquisition = new ArrayList<>();
            //Open and Close the Shutter
            if (runPreAcquisition) {
                try {
                    if (!imageType.toLowerCase().contains("pump")) {
                        LOG.info("Setting Expose main for all devices ");
                        raftsCommands.setSequencerStart("Expose");
                    } else {
                        LOG.info("Setting PocketPump main for all devices ");
                        raftsCommands.setSequencerStart("PocketPump");
                    }
                } catch (Exception e) {
                    throw new RuntimeException("Error while setting the expose main ", e);
                }

                LOG.info("Exposure main setup: XED REB = " + xedREB + " , Shutter REB = " + shutterREB);
                for (REBDevice rebDevice : rebDevices) {
                    String rebname = rebDevice.getName();
                    try {

                        if ((openShutter && (rebname.toLowerCase().contains(shutterREB.toLowerCase())))
                                || (actuateXED && (rebname.toLowerCase().contains(xedREB.toLowerCase())))) {

                            int ExposureFlushPntr = -1;
                            try {
                                ExposureFlushPntr = rebDevice.getSequencer().getFunctionMap().get("ExposureFlush");
                            } catch (NullPointerException ne) {
                                ExposureFlushPntr = rebDevice.getSequencer().getSubroutineMap().get("ExposureFlush");
                            }

                            LOG.info("ExposureFlushPntr = " + ExposureFlushPntr);
                            rebDevice.getSequencer().setParameter("Exposure", ExposureFlushPntr);
                            LOG.info(rebname + " sequence set to actuate LVDS during exposure");

                        } else {

                            int SerialFlushPntr = -1;
                            try {
                                SerialFlushPntr = rebDevice.getSequencer().getFunctionMap().get("SerialFlush");
                            } catch (NullPointerException ne) {
                                SerialFlushPntr = rebDevice.getSequencer().getSubroutineMap().get("SerialFlush");
                            }

                            LOG.info("SerialFlushPntr = " + SerialFlushPntr);
                            rebDevice.getSequencer().setParameter("Exposure", SerialFlushPntr);
                            LOG.info(rebname + " sequence set to NOT actuate LVDS during exposure");

                        }

                        sequencersToStartBeforeAcquisition.add(rebDevice.getSequencer());

                    } catch (Exception e) {
                        throw new RuntimeException("Error while setting parameters ", e);
                    }
                }
            }

            //Starting the sequencers
            for (SequencerProc sequencerToStart : sequencersToStartBeforeAcquisition) {
                sequencerToStart.startSequencer();
            }

            if (exposureTimeInMillis > 0) {

                //Wait for the sequencers to terminate their pre-acquisition execution
                long waitForShutter = exposureTimeInMillis + 2000;
                if (sequencersToStartBeforeAcquisition.size() > 0) {
                    LOG.info("Waiting " + waitForShutter + " ms for exposure sequencers to terminate");
                    for (SequencerProc sequencerToStart : sequencersToStartBeforeAcquisition) {
                        sequencerToStart.waitDone((int) waitForShutter);
                    }
                }
            }

            //Set the acquisition main routine
            try {
                raftsCommands.setSequencerStart(mainRoutine);
            } catch (Exception e) {
                throw new RuntimeException("Error while setting the " + mainRoutine, e);
            }

            //Go on with the acquisition now.        
            for (REBDevice reb : rebDevices) {
                reb.getSequencer().resetError();
            }
            LOG.info("starting image acquisition at " + System.currentTimeMillis() + " millis");
            globalProc.acquireImage();

            imageStateListener.waitForImages(30000);

            LOG.info("finished image acquisition at " + System.currentTimeMillis() + " millis");

            try {
                boolean writeFits = true;
                for (REBDevice reb : rebDevices) {
                    if (filePattern != null) {
                        if (!filePattern.isEmpty()) {
                            reb.getImageProc().setFitsFileNamePattern(filePattern);
                        } else {
                            writeFits = false;
                        }
                    }
                    if (writeFits) {
                        List<String> rebFiles = reb.getImageProc().saveFitsImage(null, null);
                        for (String rebFile : rebFiles) {
                            fitsFiles.add(rebFile);
                        }
                    }
                }
            } catch (RaftException x) {
                //TODO: understand why a RaftException would be thrown here.
                throw new IOException("Error while writing FITS files", x);
            }
            LOG.info("exposure complete");

        } finally {
            LOG.info("Removing listener " + imageStateListener);
            subsys.getMessagingAccess().removeStatusMessageListener(imageStateListener);
            LOG.info("Clearing non sticky header keywords");
        }

        LOG.info("Returning " + fitsFiles);
        return fitsFiles;
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @param openShutter True/false if the shutter should be open
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public String exposeAcquireAndSaveVerbose(long exposureTimeInMillis,
            boolean openShutter, boolean actuateXED, String filePattern)
            throws IOException, InterruptedException, REBException {
        List<String> msg = exposeAcquireAndSave(exposureTimeInMillis, openShutter, actuateXED, filePattern);
        return (aggregatorService.getAllLast().toString());
    }

    /**
     * Set the default exposure time in milliseconds.
     *
     * @param millis Exposure time in milliseconds.
     */
    @Command(description = "Set the default exposure time.", type = Command.CommandType.ACTION)
    public void setDefaultExposureTime(int millis) throws Exception {
        if (millis < 0) {
            throw new IllegalArgumentException("Illegal exposure time: it cannot be negative.");
        }
        this.exposureTimeInMillis = millis;

    }

    /**
     * Set the default exposure time in milliseconds.
     *
     * @param millis Exposure time in milliseconds.
     */
    @Command(description = "Set the sequencer exposure time.", type = Command.CommandType.ACTION)
    public void setSequencerExposureTime(int millis) throws Exception {
        if (millis < 0) {
            throw new IllegalArgumentException("Illegal exposure time: it cannot be negative.");
        }
        getRaftsCommands().setSequencerParameter("ExposureTime", (int) (millis / 25.0)); // units are 25 ms        

    }

    /**
     * Set the default open shutter flag for an exposure.
     *
     * @param open True/False if the shutter should be opened during the
     * exposure.
     */
    @Command(description = "Set the exposure default open shutter flag.", type = Command.CommandType.ACTION)
    public void setDefaultOpenShutter(boolean open
    ) {
        this.openShutter = open;
    }

    /**
     * Set the exposure default Actuated XED.
     *
     * @param actuate True/False for XED actuation.
     */
    @Command(description = "Set the exposure default XED actuation.", type = Command.CommandType.ACTION)
    public void setDefaultActuateXED(boolean actuate
    ) {
        this.actuateXED = actuate;
    }

    /*
     * THE FOLLOWING METHODS ARE NEEDED BY THE GUI
     *
     * TODO: IS THERE A WAY TO GENERALIZE THIS PROCESS SINCE ALL THESE METHODS
     * ARE ALSO DEFINED IN RAFTSMAIN?
     *
     */
    /**
     * Gets the full monitoring state. This is intended to be called by GUIs
     * during initialization
     *
     * @return The full Raft state
     */
    @Command(type = Command.CommandType.QUERY,
            description = "Gets the full Raft module state")
    public RaftState getFullState() {
        return new RaftState(getTickPeriod());
    }
    /**
     *  Gets the tick period
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

    private class ImageStatusDataListener implements StatusMessageListener {

        private final CountDownLatch latch;

        ImageStatusDataListener(int nImages) {
            latch = new CountDownLatch(nImages);
        }

        @Override
        public void onStatusMessage(StatusMessage msg) {
            StatusSubsystemData statusData = (StatusSubsystemData) msg;
            if (statusData.getDataKey().equals(ImageState.KEY)) {
                latch.countDown();
            }
        }

        public void waitForImages(long timeout) throws InterruptedException {
            boolean done = latch.await(timeout, TimeUnit.MILLISECONDS);
            if (!done) {
                throw new RuntimeException("Exceeded " + timeout + "mS wait period to receive all images. " + latch.getCount() + " images have not been received.");
            }
        }

    }

    //Utility methods
    List<REBDevice> getRebDevices() {
        return rebDevices;
    }

    Geometry getGeometry() {
        return geometry;
    }


    RaftsCommands getRaftsCommands() {
        return raftsCommands;
    }

    class ChanParams {

        private final int type;
        private final int addr;

        ChanParams(int type, int addr) {
            this.type = type;
            this.addr = addr;
        }

    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the LSST assigned CCD Serial Number")
    public void setLsstSerialNumber(String ccdId, String serialNumber) {
        setCCDHeader(ccdId, "CCDSerialLSST", serialNumber, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the Measured CCD Temperature")
    public void setMeasuredCCDTemperature(String ccdId, double ccdtemp) {
        LOG.info("Setting measured CCD temperature to " + ccdtemp);
        setCCDHeader(ccdId, "MeasuredTemperature", ccdtemp, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the Measured CCD HVBias")
    public void setMeasuredCCDBSS(String ccdId, double hvbss) {
        LOG.info("Setting measured CCDBSS to " + hvbss);
        setCCDHeader(ccdId, "MeasuredCCDBSS", hvbss, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the Monochromator Wavelength")
    public void setMonoWavelength(double wl) {
        LOG.info("Setting header MonochromatorWL to " + wl);
        setHeader("MonochromatorWavelength", wl, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the Manufacturer assigned CCD Serial Number")
    public void setManufacturerSerialNumber(String ccdId, String serialNumber) {
        setCCDHeader(ccdId, "CCDSerialManufacturer", serialNumber, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Sets the test type property for the output filename")
    public void setTestType(String ttype) {
        setHeader("TestType", ttype, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Sets the name of the test stand station")
    public void setTestStand(String tsname) {
        setHeader("TestStand", tsname, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Sets the image type property for the output filename")
    public void setImageType(String itype) {
        imageType = itype;
        setHeader("ImageType", itype, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Sets the sequence info property for the output filename")
    public void setSeqInfo(Integer seqNum) {
        setHeader("SequenceInfo", String.format("%03d", seqNum), true);
        raftsCommands.setSequenceNumber(seqNum);
    }

    @Command
    public void showPhotoDiodeAnalysis(String pddatfile)
            throws IOException, FitsException {
        fitsUtils.showPhotoDiodeAnalysis(new File(pddatfile));
    }

    @Command
    public void addBinaryTable(String pddatfile, String fitsfile,
            String extnam, String c1name, String c2name, double tstart)
            throws IOException, FitsException {
        fitsUtils.updatePhotoDiodeValues(new File(pddatfile),
                new File(fitsfile), extnam, c1name, c2name, tstart);
    }

    @Command
    public double getFluxStats(String fitsfile) throws FitsException,
            IOException {
        return (fitsUtils.getFluxStats(new File(fitsfile)));
    }

    /**
     ***************************************************************************
     **
     ** Sleep - what a waste
     * **************************************************************************
     */
    public void Sleep(double secs) {
        if ( secs <= 0 ) {
            return;
        }
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            LOG.error("Rude awakening!" + ex);
        }
    }

}
