package org.lsst.ccs.subsystem.ts8;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Predicate;
import nom.tam.fits.FitsException;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.reb.ClientFactory;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusAggregator;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.rafts.GlobalProc;
import org.lsst.ccs.subsystem.rafts.ImageProc;
import org.lsst.ccs.subsystem.rafts.REBDevice;
import org.lsst.ccs.subsystem.rafts.RaftsCommands;
import org.lsst.ccs.subsystem.rafts.SequencerProc;
import org.lsst.ccs.subsystem.rafts.config.BiasDACS;
import org.lsst.ccs.subsystem.rafts.config.DACS;
import org.lsst.ccs.subsystem.rafts.config.REB;
import org.lsst.ccs.subsystem.rafts.data.ImageState;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.data.RaftFullState;
import org.lsst.ccs.subsystem.rafts.data.RaftState;
import org.lsst.ccs.subsystem.rafts.data.RaftsAgentProperties;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.GeometryUtils;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.drivers.reb.SlowAdcs;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.Monitor.AlarmHandler;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.subsystem.rafts.TempControl;
import org.lsst.ccs.subsystem.rafts.states.SequencerMainState;
import org.lsst.ccs.utilities.image.FitsHeadersSpecifications;
import org.lsst.ccs.utilities.image.HeaderSpecification;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;
import org.lsst.ccs.utilities.image.MetaDataSet;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * The TS8Subsystem is designed specifically for running test stand 8 at BNL,
 * and the equivalent test stand at SLAC for I&T. It makes use of the Rafts
 * subsystem, which contains the code common to all use of the DAQ system both
 * for test stands and (eventually) for the construction of the full focal plane
 * during I&T and eventually the operation of the camera.
 *
 * TOOD: Review whether the split between code which is in the TS8Subsystem and
 * the rafts subsystem is consistent with the division described above.
 */
public class TS8Subsystem implements HasLifecycle, Monitor.AlarmHandler {

    private final StatusAggregator sa = new StatusAggregator();
    private final FitsUtilities fitsUtils = new FitsUtilities();

    private static final Logger LOG = Logger.getLogger("org.lsst.ccs.subsystem.ts8");

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final List<REBDevice> rebDevices = new ArrayList();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, REBDevice> rebDevicesMap = new HashMap();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final List<TempControl> tempCtrlList = new ArrayList();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private AlarmHandler alarmHandler;

    // TODO: The comment below seems confusing because there are only two protected variables, and in 
    // general variables to do not to be protected for testing.
    // TODO: The outputDirectory and flieNamePattern are just defaults, I think they can be overriden. Should check and if
    // so give them more appropriate names.
    //The next three variables are protected so that they can be tested
    protected final String outputDirectory = "${output_base_dir}/rafts/${raftId}/${runId}/${acq_type}/${test_version}/${job_id}/s${sensorLoc}";
    protected final String fileNamePattern = "${CCDSerialLSST}_${TestType}_${ImageType}_${SequenceInfo}_${RunNumber}_${timestamp}.fits";

    @ConfigurationParameter(isFinal = true)
    private String shutterREB = "R00.Reb0";
    @ConfigurationParameter(isFinal = true)
    private String xedREB = "R00.Reb1";

    @ConfigurationParameter(category = "InternalLimits")
    private double temp_Max = 0.045; // max temperature
    @ConfigurationParameter(category = "InternalLimits")
    private double odI_Max = 0.250; // max current on OD lines
    @ConfigurationParameter(category = "InternalLimits")
    private double clkI_Max = 0.300; // max current on Clock lines
    @ConfigurationParameter(category = "InternalLimits")
    private double anaI_Max = 0.670; // max current on Analog lines
    @ConfigurationParameter(category = "InternalLimits")
    private double digI_Max = 0.750; // max current on Digital lines

//    private double temp_Min = 0.0; // max temperature
    @ConfigurationParameter(category = "InternalLimits")
    private double odI_Min = 0.060; // max current on OD lines
    @ConfigurationParameter(category = "InternalLimits")
    private double clkI_Min = 0.100; // max current on Clock lines
    @ConfigurationParameter(category = "InternalLimits")
    private double anaI_Min = 0.400; // max current on Analog lines
    @ConfigurationParameter(category = "InternalLimits")
    private double digI_Min = 0.400; // max current on Digital lines

    @ConfigurationParameter(category = "InternalLimits")
    private double CCDI_Max = 0.00500; // max current on CCD lines
    @ConfigurationParameter(category = "InternalLimits")
    private double CCDI_Min = 0.00080; // min current on CCD lines

    @ConfigurationParameter(category = REBDevice.RAFTS)
    private String ccdType = null; // The CCD type

    @ConfigurationParameter(category = "InternalLimits")
    private double biasDacVBand = 0.5; // one sided band on bias voltages

    @ConfigurationParameter(category = "InternalLimits")
    private double biasDacVWideBand_OD = 1.5; // one sided band on OD (different from other biases since it doesn't go below 1V)

    private final Geometry geometry;
    // TODO: Why do we have sLog and log above?
    private final Logger sLog;

    //Image acquisition default values
    private long exposureTimeInMillis = 2000;
    private boolean openShutter = true;
    private boolean actuateXED = false;

    //Header keyword values related objects
    private final Map<String, HeaderKeywordValue> globalHeaderKeywords = new HashMap<>();
    private final Map<String, Map<String, HeaderKeywordValue>> ccdSpecificHeaderKeywords = new HashMap<>();

    private final Map<String, ChanParams> monChans = new HashMap<>();

    private final ClientFactory clientFactory;
    private RaftsCommands raftsCommands;
    private String last_alertStr;

    private final GlobalProc globalProc = new GlobalProc();

    private TempControl tempCtrl;

    private boolean useInfoAlerts = System.getProperty("org.lsst.ccs.subsystem.ts8PowerInfoAlerts", "false").contains("true");

    /**
     * Main constructor for the TS8Subsystem, normally invoked from the
     * corresponding groovy file.
     *
     * @param geometry The initial geometry. This is used for ???
     * @param clientFactory The ClientFactoryused to create the interfaces to
     * the DAQ, or <code>null</code> to use the default ClientFactory. See
     * {@link org.lsst.ccs.subsystem.ts8.sim.TS8ClientFactorySimulation} for an
     * alternative factory which can be used for simulations.
     */
    public TS8Subsystem(Geometry geometry, ClientFactory clientFactory) {

        this.geometry = geometry;
        this.clientFactory = clientFactory;

        sLog = Logger.getLogger("org.lsst.ccs.subsystem.ts8");

        //Configuration of which spec files will be loaded.
        //TODO: Why use static methods here?
        FitsHeadersSpecifications.addSpecFile("ts8_primary", "primary");
        FitsHeadersSpecifications.addSpecFile("ccd_cond");
        FitsHeadersSpecifications.addSpecFile("reb_cond");
        FitsHeadersSpecifications.addSpecFile("test_cond");

    }

    @Override
    public void postInit() {

        fillHeaderKeywordMaps(geometry);

        // By setting RAFT_TYPE_AGENT_PROPERTY we signal to consoles that this subsystem is compatible with the rafts subsystm GUI
        subsys.setAgentProperty(RaftsAgentProperties.RAFT_TYPE_AGENT_PROPERTY, TS8Subsystem.class.getCanonicalName());

        // Get temperature control object
        if (!tempCtrlList.isEmpty()) {
            tempCtrl = (TempControl) tempCtrlList.get(0);
            tempCtrl.initialize(sLog);
        } else {
            sLog.info("No temperature control available");
        }

        raftsCommands = new RaftsCommands(subsys, rebDevices, globalProc, geometry, tempCtrl);

        //Add the RaftCommands to the CommandSet
        subsys.addCommandsFromObject(raftsCommands, "");

        //Initialize the StatusAggregator: it will aggregate all messages
        //coming from subsystems: ts,ccs-rebps,ts8
        //we just want the last value, no aggregate/history.
        sa.setAggregatePattern(System.getProperty("org.lsst.ccs.subsystem.teststand", "ts") + "/.*", -1, -1);
        sa.setAggregatePattern(System.getProperty("org.lsst.ccs.subsystem.rebps", "ccs-rebps") + "/.*", -1, -1);
        sa.setAggregatePattern(System.getProperty("org.lsst.ccs.subsystem.ts8", "ts8") + "/.*", -1, -1);
        sa.setAggregatePattern(System.getProperty("org.lsst.ccs.subsystem.ts8-bench", "ts8-bench") + "/.*", -1, -1);

        sLog.info("Configured the status aggregator to listen to subsystems: " + System.getProperty("org.lsst.ccs.subsystem.teststand", "ts") + " " + System.getProperty("org.lsst.ccs.subsystem.rebps", "ccs-rebps") + " " + System.getProperty("org.lsst.ccs.subsystem.ts8", "ts8"));

        // Initialize for triggering
        globalProc.initialize(rebDevices, clientFactory, sLog);

        //TO-DO, Why is this needed?
        Properties filePatternProperties = BootstrapResourceUtils.getBootstrapProperties("output_location");
        for (Object key : BootstrapResourceUtils.getAllKeysInProperties(filePatternProperties)) {
            String propertyName = (String) key;
            raftsCommands.setPatternProperty(propertyName, filePatternProperties.getProperty(propertyName));
        }

        raftsCommands.setFitsFileNamePattern(fileNamePattern);
        raftsCommands.setDefaultImageDirectory(outputDirectory);

        // setup channel data
        monChans.put("Temp1", new ChanParams(REBDevice.TYPE_BD_TEMP, 0));
        monChans.put("Temp2", new ChanParams(REBDevice.TYPE_BD_TEMP, 1));
        monChans.put("Temp3", new ChanParams(REBDevice.TYPE_BD_TEMP, 2));
        monChans.put("Temp4", new ChanParams(REBDevice.TYPE_BD_TEMP, 3));
        monChans.put("Temp5", new ChanParams(REBDevice.TYPE_BD_TEMP, 4));
        monChans.put("Temp6", new ChanParams(REBDevice.TYPE_BD_TEMP, 5));
        monChans.put("Temp7", new ChanParams(REBDevice.TYPE_BD_TEMP, 6));
        monChans.put("Temp8", new ChanParams(REBDevice.TYPE_BD_TEMP, 7));
        monChans.put("Temp9", new ChanParams(REBDevice.TYPE_BD_TEMP, 8));
        monChans.put("Temp10", new ChanParams(REBDevice.TYPE_BD_TEMP, 9));
        monChans.put("Atemp2U", new ChanParams(REBDevice.TYPE_ASP_TEMP, 4));
        monChans.put("Atemp1U", new ChanParams(REBDevice.TYPE_ASP_TEMP, 2));
        monChans.put("Atemp0U", new ChanParams(REBDevice.TYPE_ASP_TEMP, 0));
        monChans.put("Atemp2L", new ChanParams(REBDevice.TYPE_ASP_TEMP, 5));
        monChans.put("Atemp1L", new ChanParams(REBDevice.TYPE_ASP_TEMP, 3));
        monChans.put("Atemp0L", new ChanParams(REBDevice.TYPE_ASP_TEMP, 1));
        monChans.put("CCDTemp0", new ChanParams(REBDevice.TYPE_RTD, 0));
        monChans.put("CCDTemp1", new ChanParams(REBDevice.TYPE_RTD, 1));
        monChans.put("CCDTemp2", new ChanParams(REBDevice.TYPE_RTD, 2));
        monChans.put("RTDTemp", new ChanParams(REBDevice.TYPE_RTD, 3));
        monChans.put("DigI", new ChanParams(REBDevice.TYPE_BD_POWER, 1));
        monChans.put("AnaI", new ChanParams(REBDevice.TYPE_BD_POWER, 3));
        monChans.put("ClkHI", new ChanParams(REBDevice.TYPE_BD_POWER, 5));
        monChans.put("ClkLI", new ChanParams(REBDevice.TYPE_BD_POWER, 9));
        monChans.put("ODI", new ChanParams(REBDevice.TYPE_BD_POWER, 7));

        monChans.put("OD0V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OD_0));
        monChans.put("OG0V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OG_0));
        monChans.put("RD0V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_RD_0));
        monChans.put("GD0V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_GD_0));

        monChans.put("OD1V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OD_1));
        monChans.put("OG1V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OG_1));
        monChans.put("RD1V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_RD_1));
        monChans.put("GD1V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_GD_1));

        monChans.put("OD2V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OD_2));
        monChans.put("OG2V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_OG_2));
        monChans.put("RD2V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_RD_2));
        monChans.put("GD2V", new ChanParams(REBDevice.TYPE_BIAS_VOLT, SlowAdcs.CHAN_GD_2));

        for (int iccd = 0; iccd < 3; iccd++) {
            for (int ihalf = 0; ihalf < 2; ihalf++) {
                String tb = ihalf == 0 ? "upper" : "lower";
                for (int jc = 0; jc < 8; jc++) {
                    int ch = jc + 8 * ihalf + 16 * iccd;
                    String name = "CCDI" + iccd + ihalf + jc;
                    monChans.put(name, new ChanParams(REBDevice.TYPE_CCD_CURR, ch));
                }
            }
        }
    }

    @Override
    public void build() {
        //Add a periodic task to check if Sequencers are running
        //This will be replaced once the DAQ v2 will be able to call us back
        //when the sequencer is done running
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("sequencerMonitor",
                () -> {
                    checkSequencerState();
                }).withIsFixedRate(false).withPeriod(Duration.ofSeconds(5)));
    }

    private void checkSequencerState() {
        if (raftsCommands == null) {
            throw new RuntimeException("RaftsCommands have not been initialized");
        }
        raftsCommands.checkSequencerState();
    }

    @Override
    public void postStart() {
        sLog.info("TS8 subsystem started");
        //At start we don't know what is the main state of the Sequencer
        //so we set it to UNKNOWN
        subsys.updateAgentState(SequencerMainState.UNKNOWN);
        if (ccdType == null) {
            raftsCommands.setCcdType(((Raft) geometry).getType().getName());
        } else {
            raftsCommands.changeCcdType(ccdType);
        }
    }

    @ConfigurationParameterChanger
    public void setCcdType(String ccdType) {
        //Should setCcdType throw an exception?
        //NOTE: this method could be invoked right at startup, when the raftsCommands
        //object has not been built yet. So we have to check for that and invoke
        //the same method in the postStart method to make sure it's invoked 
        //after the ccdType has been invoked during startup.
        if (raftsCommands != null) {
            raftsCommands.changeCcdType(ccdType);
        }
        this.ccdType = ccdType;
    }

    @Override
    public void start() {
        //Register the StatusAggregator on the buses
        Predicate<BusMessage<?, ?>> allStatusSubsystemDataFilter = BusMessageFilterFactory
                .messageClass(StatusSubsystemData.class);
        subsys.getMessagingAccess().addStatusMessageListener(sa,
                allStatusSubsystemDataFilter);

    }

    @Override
    public boolean processAlarm(int event, int parm, String cause, String alarmName) {
        if (alarmHandler != null) {
            return alarmHandler.processAlarm(event, parm, cause, alarmName);
        }
        return false;
    }

    @Command(description = "Print Header specifications", type = Command.CommandType.QUERY)
    public String printHeaderSpecifications() {
        StringBuilder sb = new StringBuilder();
        Map<String, HeaderSpecification> config = FitsHeadersSpecifications.getHeaderSpecifications();
        for (String header : config.keySet()) {
            sb.append("***************************\n");
            sb.append("     Header: ").append(header).append("\n");
            sb.append("***************************\n");
            HeaderSpecification spec = config.get(header);
            for (HeaderLine line : spec.getHeaders()) {
                sb.append("## ").append(line.getKeyword()).append(" ").append(line.getMetaName()).append(" ").append(line.getDataType()).append(" ").append(line.getComment()).append("\n");
            }
        }
        return sb.toString();
    }

    @Command(description = "Print Geometry", type = Command.CommandType.QUERY)
    public String printGeometry(@Argument(defaultValue = "0") int depth) {
        return GeometryUtils.showGeometryTree(geometry, depth);
    }

    /**
     * Sets the name of the CCD at the given location. This will be called by
     * the script running TS8 using data obtained from the eTraveler, and will
     * enable appropriate headers to be added to the FitsFile generated. //TODO:
     * Perhaps a better name would be setNameOfCCDAtLocation?
     *
     * @param serial The serial position of the CCD
     * @param parallel The parallel position of the CCD
     * @param ccdName The name of the CCD at the given position
     * @throws IllegalArgumentException if the serial or parallel position are
     * illegal.
     */
    @Command(description = "Set the output directory for fits files", type = Command.CommandType.QUERY)
    public void setCCDLocation(int serial, int parallel, String ccdName) throws IllegalArgumentException {
        // TODO: REMOVE THIS ... use setLSSTSerialNumber instead
        throw new UnsupportedOperationException("use setLSSTSerialNumber instead");
    }

    /**
     * Set the name of the raft. This will be called by the script running TS8
     * using data obtained from the eTraveler, and will enable appropriate
     * headers to be added to the FitsFile generated.
     *
     * @param raftName The name of the raft.
     */
    @Command(description = "Set the name of the raft", type = Command.CommandType.QUERY)
    public void setRaftName(String raftName) {
        setHeader("RaftName", raftName, true);
    }

    /**
     * Set the name of the reb at the given parallel position. This will be
     * called by the script running TS8 using data obtained from the eTraveler,
     * and will enable appropriate headers to be added to the FitsFile
     * generated.
     *
     * @param parallel The parallel position of the Reb.
     * @param rebName The new name of the reb
     * @throws IllegalArgumentException if the parallel position is illegal.
     */
    @Command(description = "Set the name of the reb at a given parallel position", type = Command.CommandType.QUERY)
    public void setRebName(int parallel, String rebName) throws IllegalArgumentException {
        Reb reb = (Reb) geometry.getChild(parallel, 0);
        if (reb == null) {
            throw new RuntimeException("Could not find Reb for parallel position " + parallel);
        }
        for (CCD ccd : reb.getChildrenList()) {
            setCCDHeader(ccd.getUniqueId(), "RebName", rebName, true);
        }
    }

    /**
     * Set the RunNumber.
     *
     * @param runNumber The run Number
     */
    @Command(description = "Set the run number", type = Command.CommandType.QUERY)
    public void setRunNumber(String runNumber) {
        setHeader("RunNumber", runNumber, true);
    }

    @Command(type = Command.CommandType.ACTION, description = "checks all currents")
    public String checkCurrentsAndTemperatures() {

        StringBuilder results = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            results.append(checkCurrentsAndTemperatures(rebDevice.getRebNumber(), false));
        }
        return (results.toString());
    }

    @Command(type = Command.CommandType.ACTION, description = "checks all currents")
    public String checkTemperatures(int rebnum) {

        StringBuilder results = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() != rebnum) {
                continue;
            }
            String rebname = "REB" + rebnum;
            results.append("Checking temperatures for REB").append(rebnum).append(":\n");
            try {

                String[] temps = {"RTDTemp", "Atemp0U", "Atemp0L", "Atemp1U", "Atemp1L", "Atemp2U", "Atemp2L", "CCDTemp0", "CCDTemp1", "CCDTemp2",
                    "Temp1", "Temp2", "Temp3", "Temp4", "Temp5", "Temp6", "Temp7", "Temp8", "Temp9", "Temp10"};
                for (String temp : temps) {
                    last_alertStr = "";
                    getChanValue(rebDevice, temp, -1.0, temp_Max);
                    results.append(rebname).append(": ").append(last_alertStr);
                }

            } catch (RuntimeException e) {
                LOG.error(rebname + results.toString() + "\nSAFE CHANNEL READINGS CHECK FAILED WITH ERROR " + e);

                powerOff(rebnum);

                throw e;

            }
        }
        return results.toString();
    }

    @Command(type = Command.CommandType.ACTION, description = "checks all currents")
    public String checkCurrentsAndTemperatures(int rebnum) {
        return (checkCurrentsAndTemperatures(rebnum, false));
    }

    @Command(type = Command.CommandType.ACTION, description = "checks all currents")
    public String checkCurrentsAndTemperatures(int rebnum, boolean checkCCDI) {
        StringBuilder results = new StringBuilder();
        results.append(checkCurrents(rebnum, checkCCDI));
        results.append(checkTemperatures(rebnum));
        return results.toString();
    }

    @Command(type = Command.CommandType.ACTION, description = "checks all currents")
    public String checkCurrents(int rebnum) {
        return checkCurrents(rebnum, false);
    }

    @Command(type = Command.CommandType.ACTION, description = "checks all currents")
    public String checkCurrents(int rebnum, boolean checkCCDI) {

        StringBuilder results = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() != rebnum) {
                continue;
            }
            String rebname = "REB" + rebnum;
            results.append("Checking currents for REB").append(rebnum).append(":\n");

            try {
                getChanValue(rebDevice, "ODI", odI_Min, odI_Max);
                results.append(rebname).append(": ").append(last_alertStr);
                getChanValue(rebDevice, "DigI", digI_Min, digI_Max);
                results.append(rebname).append(": ").append(last_alertStr);
                getChanValue(rebDevice, "AnaI", anaI_Min, anaI_Max);
                results.append(rebname).append(": ").append(last_alertStr);
                getChanValue(rebDevice, "ClkLI", clkI_Min, clkI_Max);
                getChanValue(rebDevice, "ClkHI", clkI_Min, clkI_Max);
                results.append(rebname).append(": ").append(last_alertStr);

                if (checkCCDI) {
                    for (int iccd = 0; iccd < 3; iccd++) {
                        // check CCDI channels only if OD is up for the corresponding sensor
                        if (getChanValue(rebDevice, "OD" + iccd + "V", -0.5, 40.) > 15.0) {
                            for (int ihalf = 0; ihalf < 2; ihalf++) {
//                            String tb = ihalf == 0 ? "upper" : "lower";
                                for (int jc = 0; jc < 8; jc++) {
                                    int ch = jc + 8 * ihalf + 16 * iccd;
                                    String name = "CCDI" + iccd + ihalf + jc;
                                    getChanValue(rebDevice, name, CCDI_Min, CCDI_Max);
                                    results.append(rebname).append(": ").append(last_alertStr);
                                }
                            }
                        }
                    }
                }

            } catch (RuntimeException e) {
                LOG.error(rebname + results.toString() + "\nSAFE CHANNEL READINGS CHECK FAILED WITH ERROR " + e
                        + "\n setting all dacs to nominal levels!");
                powerOff(rebnum);

                throw e;
            }

        }
        return results.toString();
    }

    private double getChanValue(REBDevice rebDevice, String name, double lowLimit, double highLimit) {
        double value = rebDevice.readChannelNow(monChans.get(name).addr, monChans.get(name).type);

        String rebname = rebDevice.getName();

        if ((name.toLowerCase().contains("temp") || name.toLowerCase().contains("rtd"))) {
            if (value > 50. || value < -150. || Double.isNaN(value)) {
                String alertStr = rebname + ": " + name + " - Anomolous temperature reading for " + name + " with value " + String.format("%8.3f", value) + " C\n";
                LOG.info(alertStr);
                this.raiseAlert(alertStr, AlertState.NOMINAL);

                value = -999.;
            } else {
                String alertStr = rebname + ": " + name + " - OK temperature reading for " + name + " with value " + String.format("%8.3f", value) + " C\n";
                LOG.info(alertStr);
                this.raiseAlert(alertStr, AlertState.NOMINAL);
            }
        } else if (value < lowLimit) {
            String alertStr = rebname + ": " + name + " - I = " + String.format("%10.5f", value) + " is BELOW range " + lowLimit + " to " + highLimit + " Amps ";
            if (name.contains("CCDI")) {
                alertStr += " LOW CURRENT CCD CHANNEL!\n";
                LOG.warning(alertStr);
                this.raiseAlert(alertStr, AlertState.WARNING);
            } else {
                alertStr += "\n";
                LOG.error(alertStr);
                this.raiseAlert(alertStr, AlertState.ALARM);
                throw new RuntimeException("ALERT! Channel " + name + " reading " + value + " is below the limit " + lowLimit);
            }
        } else if (value > highLimit) {
            String alertStr = rebname + ": " + name + " = " + String.format("%10.5f", value) + " EXCEEDS range " + lowLimit + " to " + highLimit + "\n";
            LOG.error(alertStr);
            this.raiseAlert(alertStr, AlertState.ALARM);
            throw new RuntimeException("ALERT! Channel " + name + " reading " + value + " is above the limit " + highLimit);
        } else {
            String alertStr = rebname + ": " + name + " = " + String.format("%10.5f", value) + " is within range " + lowLimit + " to " + highLimit + "\n";
            LOG.info(alertStr);

            this.raiseAlert(alertStr, AlertState.NOMINAL);

        }

        return value;
    }

    /**
     * Clear REB config
     *
     * @param rebnum The number of the REB to have its config cleared
     * @return A String representing the state of the devices after they are
     * powered off.
     */
    @Command(type = Command.CommandType.ACTION, description = "clear REB config for a given REB ID")
    public String clearREBCfg(int rebnum) throws Exception {
        StringBuilder outstr = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() == rebnum) {
                outstr.append(resetREBCfg(rebDevice));
            }
        }
        return outstr.toString();
    }

    /**
     * Clear REB config
     *
     * @param rebDevice The REB device to have its config cleared
     * @return A String representing the state of the devices after they are
     * powered off.
     */
    @Command(type = Command.CommandType.ACTION, description = "reset REB config")
    public String resetREBCfg(REBDevice rebDevice) throws Exception {

        LOG.info("state before reset = " + this.showREBCfg(rebDevice));

        REB reb = rebDevice.getREBConfig();
        BiasDACS[] biases = reb.getBiases();
        for (BiasDACS bias : biases) {
            double[] values = bias.getPValues();
            values[BiasDACS.GD] = 0.;
            values[BiasDACS.OD] = 0.;
            values[BiasDACS.OG] = 0.;
            values[BiasDACS.RD] = 0.;
            values[BiasDACS.CS_GATE] = 0.;
        }

        DACS dacs = reb.getDacs();

        double[] dacvalues = dacs.getPValues();
        dacvalues[DACS.PCLK_HIGH] = 0.;
        dacvalues[DACS.PCLK_LOW] = 0.;
        dacvalues[DACS.SCLK_HIGH] = 0.;
        dacvalues[DACS.SCLK_LOW] = 0.;
        dacvalues[DACS.RG_HIGH] = 0.;
        dacvalues[DACS.RG_LOW] = 0.;

        LOG.info("setting new config");
        rebDevice.setREBConfig(reb);

        String msg = "state after reset:\n" + this.showREBCfg(rebDevice);
        return msg;
    }

    @Command(type = Command.CommandType.QUERY, description = "raise subsystem alert")
    public void raiseAlert(String alertmsg, AlertState severity) {
        org.lsst.ccs.bus.data.Alert a = new org.lsst.ccs.bus.data.Alert("TS8 Alert", "information");
        if (!(severity == AlertState.NOMINAL) || (severity == AlertState.NOMINAL && this.useInfoAlerts)) {
            for (String line : alertmsg.split("\n")) {
                subsys.getAlertService().raiseAlert(a, severity, line.substring(0, Math.min(254, line.length())));
            }
        }
        last_alertStr = alertmsg;
    }

    /**
     * Get the configuration for all REB devices
     *
     * @return A string representing the configuration of the devices.
     */
    @Command(type = Command.CommandType.ACTION, description = "show REB config for all REBS")
    public String showAllREBCfg() {
        StringBuilder outstr = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            outstr.append(showREBCfg(rebDevice));
        }
        return outstr.toString();
    }

    @Command(type = Command.CommandType.ACTION, description = "show REB config for a given REB ID")
    public String showREBCfg(int rebnum) {
        StringBuilder outstr = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() == rebnum) {
                outstr.append(showREBCfg(rebDevice));
            }
        }
        return outstr.toString();
    }

    /**
     * Get the configuration of a particular REB device.
     *
     * @param rebDevice The device for which the configuration is to be
     * obtained.
     * @return A string representing the device's configuration.
     */
//    @Command(type = Command.CommandType.ACTION, description = "show REB config")
    private String showREBCfg(REBDevice rebDevice) {

        StringBuilder newcfgsstr = new StringBuilder("\nShowing config for REB:").append(rebDevice.getRebNumber()).append(" : \n");

        REB reb = rebDevice.getREBConfig();
        BiasDACS[] biases = reb.getBiases();
        int i_ccd = 0;
        for (BiasDACS bias : biases) {
            // TODO: i_ccd is always going to be zero, this can't be what was intended??
            newcfgsstr.append("CCD").append(i_ccd).append(":\n");
            newcfgsstr.append("The BiasDACS voltage settings are (V):\n")
                    .append("GD=").append(bias.getPValues()[BiasDACS.GD]).append(" ,\n")
                    .append("RD=").append(bias.getPValues()[BiasDACS.RD]).append(" ,\n")
                    .append("OD=").append(bias.getPValues()[BiasDACS.OD]).append(" ,\n")
                    .append("OG=").append(bias.getPValues()[BiasDACS.OG]).append(" ,\n");
            i_ccd++;
        }

        DACS dacs = reb.getDacs();

        newcfgsstr.append("new DACS config:\n" + "PCLK_HIGH=").append(dacs.getPValues()[DACS.PCLK_HIGH]).append(" ,\n")
                .append("PCLK_LOW=").append(dacs.getPValues()[DACS.PCLK_LOW]).append(" ,\n")
                .append("SCLK_HIGH=").append(dacs.getPValues()[DACS.SCLK_HIGH]).append(" ,\n")
                .append("SCLK_LOW=").append(dacs.getPValues()[DACS.SCLK_LOW]).append(" ,\n")
                .append("RG_HIGH=").append(dacs.getPValues()[DACS.RG_HIGH]).append(" ,\n")
                .append("RG_LOW=").append(dacs.getPValues()[DACS.RG_LOW]).append(" ,\n");

        i_ccd++;
        return newcfgsstr.toString();
    }

    /**
     * Choose the test voltage to use based on the target voltage
     */
    public double chooseTestVolt(double proposedVoltage, double full_target) {
        return (Math.abs(proposedVoltage) < Math.abs(full_target) ? Math.copySign(proposedVoltage, full_target) : full_target / 5.0);
    }

    private String dateStr() {
        return (new SimpleDateFormat("yyyy-MM-dd-HH:mm:ss").format(new Date(System.currentTimeMillis())));
    }

    /**
     * Safely power on the Raft
     *
     * @return A string representing the final state of the system
     */
    @Command(type = Command.CommandType.ACTION, description = "safe power on")
    public String powerOn() {

        StringBuilder outmsg = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            outmsg.append(powerOn(rebDevice.getRebNumber()));
        }
        return (outmsg.toString());
    }

    /**
     * Safely power on the Raft
     *
     * @return A string representing the final state of the system
     */
    @Command(type = Command.CommandType.ACTION, description = "safe power on")
    public String powerOn(int rebnum) {

        StringBuilder outmsg = new StringBuilder();
        useInfoAlerts = System.getProperty("org.lsst.ccs.subsystem.ts8PowerInfoAlerts", "false").contains("true");

        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() != rebnum) {
                continue;
            }

            double noLim = 100.;
            if (Math.abs(getChanValue(rebDevice, "RD" + rebnum + "V", -noLim, noLim)) > 1.0
                    || Math.abs(getChanValue(rebDevice, "OD" + rebnum + "V", -noLim, noLim)) > 1.5
                    || Math.abs(getChanValue(rebDevice, "GD" + rebnum + "V", -noLim, noLim)) > 1.0
                    || Math.abs(getChanValue(rebDevice, "OG" + rebnum + "V", -noLim, noLim)) > 1.0) {

                String AbrtMsg = "Check of biases indicates sensors are not in an unpowered state.\n ABORTING powerOn procedure for REB" + rebnum + " !\n";
                LOG.error(AbrtMsg);
                this.raiseAlert(AbrtMsg, AlertState.NOMINAL);
                outmsg.append(AbrtMsg);
                continue;
            }

            String rebname = "REB" + rebnum;
            String msg = "Checking " + rebname + " at " + dateStr();
            LOG.warn(msg);
            this.raiseAlert(msg, AlertState.NOMINAL);

            // show the initial configuration before copying it then clearing it
            this.showREBCfg(rebDevice);
            outmsg.append("REB ").append(rebDevice.getRebNumber()).append(" :         " + dateStr() + "\n");

            // create a copy of the original REB settings
            REB reb = rebDevice.getREBConfig();
            REB rebOriginal = new REB();
            rebOriginal.copyFrom(reb);

            // copy the bias dac settings
            int n_ccd = reb.getBiases().length;
            BiasDACS[] biasesOriginal = new BiasDACS[n_ccd];

            for (int idx = 0; idx < n_ccd; idx++) {
                biasesOriginal[idx] = new BiasDACS();
                biasesOriginal[idx].copyFrom(reb.getBiases()[idx]);
            }

            BiasDACS[] biases = reb.getBiases();

            // set all values to idle levels in active config
            msg = "\n" + rebname + ": resetting config of all DACS to unpowered levels at " + dateStr() + "\n";
            outmsg.append(msg);
            this.raiseAlert(msg, AlertState.NOMINAL);
            LOG.info(msg);
            try {
                resetREBCfg(rebDevice);
                rebDevice.loadBiasDacs(true);
                rebDevice.loadDacs(true);
            } catch (Exception x) {
                throw new RuntimeException(rebname + ": Error while loading dacs", x);
            }

            // check currents and temperatures at this setting
            msg = rebname + ": Checking all currents with nominal levels\n";
            outmsg.append(msg);
            this.raiseAlert(msg, AlertState.NOMINAL);
            LOG.info(msg);

            LOG.info(rebname + ": Checking all currents with nominal DAC levels at " + dateStr());
            try {
                this.checkCurrentsAndTemperatures(rebnum);
            } catch (RuntimeException e) { // get all stat info into error message
                throw new RuntimeException(outmsg.toString() + e);
            }

// =----------------------------------------------------------------
//
// 15.1 Biasing sequence for E2V rafts:
// " Power-up the CCD biases and rails using the nominal operating values in LCA?15131, in the sequence [RM1] 
// (1) RD, (2) OD, (3) GD, (4) OG, (5) SClk_Low, (6) SClk_High, (7) PClk_Low, (8) PClk_High, (9) RG_Low, (10) RG_High."
//
            double test_voltage = 0.500; // 500 mV test voltage
            double volts = 0.0;
            double target = 0.0;
            if (raftsCommands.getCcdType().toLowerCase().contains("e2v")) {

                for (int idx = 0; idx < biases.length; idx++) {
                    BiasDACS bias = biases[idx];

                    outmsg.append("\n" + rebname + ": CCD ").append(idx).append(" :\n");

// Begin to apply the CCD frontside biases. Initial value of
// each voltage should be <500mV to verify that no short circuits are present (e.g. current <
// 10mA at 500mV); then bring biases to their recommended (ITL or E2V) levels.
                    // RD
                    outmsg.append(rebname + ": Testing RD for CCD " + idx + " at " + dateStr() + " - \n");
                    target = biasesOriginal[idx].getPValues()[BiasDACS.RD];
                    volts = biasDACChanOn("RD", rebDevice, idx, this.chooseTestVolt(test_voltage, target));
                    outmsg.append(rebname + ": RD set to " + volts + " V - status OK\n");
                    // GD
                    target = biasesOriginal[idx].getPValues()[BiasDACS.GD];
                    outmsg.append(rebname + ": Testing GD for CCD " + idx + " at " + dateStr() + " - \n");
                    volts = biasDACChanOn("GD", rebDevice, idx, this.chooseTestVolt(test_voltage, target));
                    outmsg.append(rebname + ": GD set to " + volts + " V - status OK\n");
                    // OG
                    outmsg.append(rebname + ": Testing OG for CCD " + idx + " at " + dateStr() + " - \n");
                    target = biasesOriginal[idx].getPValues()[BiasDACS.OG];
                    volts = biasDACChanOn("OG", rebDevice, idx, this.chooseTestVolt(test_voltage, target));
                    outmsg.append(rebname + ": OG set to " + volts + " V - status OK\n");

                    /* passed the short circuit checks , now  put the configured voltages */
                    // RD
                    outmsg.append(rebname + ": Applying RD for CCD " + idx + " at " + dateStr() + " - \n");
                    double volts_rd = biasDACChanOn("RD", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.RD]);
                    outmsg.append(rebname + ": RD set to " + volts_rd + " V - status OK\n");
                    // OD
                    outmsg.append(rebname + ": Applying OD for CCD " + idx + " at " + dateStr() + " - \n");
                    double volts_od = biasDACChanOn("OD", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.OD]);
                    outmsg.append(rebname + ": OD set to " + volts_od + " V - status OK\n");
                    // GD
                    outmsg.append(rebname + ": Applying GD for CCD " + idx + " at " + dateStr() + " - \n");
                    double volts_gd = biasDACChanOn("GD", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.GD]);
                    outmsg.append(rebname + ": GD set to " + volts_gd + " V - status OK\n");
                    // OG
                    outmsg.append(rebname + ": Applying OG for CCD " + idx + " at " + dateStr() + " - \n");
                    double volts_og = biasDACChanOn("OG", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.OG]);
                    outmsg.append(rebname + ": OG set to " + volts_og + " V - status OK\n");

                    // CSGATE on only now
                    outmsg.append(rebname + ": Turning on CSGATE for CCD " + idx + " at " + dateStr() + " - \n");
                    volts = biasDACChanOn("CS_GATE", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.CS_GATE]);

                    Sleep(0.1);
                    outmsg.append(rebname + ": RD measured = " + getChanValue(rebDevice, "RD" + idx + "V", volts_rd - biasDacVBand, volts_rd + biasDacVBand) + " V\n");
                    outmsg.append(rebname + ": OD measured = " + getChanValue(rebDevice, "OD" + idx + "V", volts_od - biasDacVBand, volts_od + biasDacVBand) + " V\n");
                    outmsg.append(rebname + ": GD measured = " + getChanValue(rebDevice, "GD" + idx + "V", volts_gd - biasDacVBand, volts_gd + biasDacVBand) + " V\n");
                    outmsg.append(rebname + ": OG measured = " + getChanValue(rebDevice, "OG" + idx + "V", volts_og - biasDacVBand, volts_og + biasDacVBand) + " V\n");
                    outmsg.append(rebname + ": CSGATE set to " + volts + " V - status OK\n");

                }

                // now for the DACS --------------------------------------------------------
                DACS dacs = reb.getDacs();
                DACS dacsOriginal = rebOriginal.getDacs();

                // serial clocks
                msg = rebname + ": bringing up serials at " + dateStr() + "\n";
                outmsg.append(msg);
                this.raiseAlert(msg, AlertState.NOMINAL);
                LOG.info(msg);
                double volts_hi;
                double volts_lo;
                outmsg.append(rebname).append(": testing serial lines\n");
                volts_hi = dacChanOn("SCLK_HIGH", rebDevice, dacsOriginal, test_voltage);
                volts_lo = dacChanOn("SCLK_LOW", rebDevice, dacsOriginal, test_voltage);
                outmsg.append(rebname).append(": serials set to ").append(volts_hi).append("/").append(volts_lo).append(" V - status OK\n");

                // parallel clocks
                msg = rebname + ": bringing up parallels at " + dateStr() + "\n";
                outmsg.append(msg);
                this.raiseAlert(msg, AlertState.NOMINAL);
                LOG.info(msg);
                outmsg.append(rebname).append(": testing parallel lines\n");
                volts_hi = dacChanOn("PCLK_HIGH", rebDevice, dacsOriginal, test_voltage);
                volts_lo = dacChanOn("PCLK_LOW", rebDevice, dacsOriginal, test_voltage);
                outmsg.append(rebname).append(": parallels set to ").append(volts_hi).append("/").append(volts_lo).append(" V - status OK\n");

                // RG
                msg = rebname + ": bringing up RG at " + dateStr() + "\n";
                outmsg.append(msg);
                this.raiseAlert(msg, AlertState.NOMINAL);
                LOG.info(msg);
                outmsg.append(rebname).append(": testing RG lines\n");
                volts_hi = dacChanOn("RG_HIGH", rebDevice, dacsOriginal, test_voltage);
                volts_lo = dacChanOn("RG_LOW", rebDevice, dacsOriginal, test_voltage);
                outmsg.append(rebname).append(": RG set to ").append(volts_hi).append("/").append(volts_lo).append(" V - status OK\n");
                // ================================= end of E2V front bias powering =====================

                // ================================= start of ITL front bias powering =====================
// 15.2 Biasing sequence for ITL rafts:
// " Power-up the CCD biases and rails using the nominal operating values in LCA?15131, in the sequence [RM2]
// (1) SClk_Low, (2) SClk_High, (3) Pclk_Low, (4) Pclk_High, (5) RG_Low, (6) RG_High, (7) OG, (8) GD, (9) RD, (10) OD".
//
            } else if (raftsCommands.getCcdType().toLowerCase().contains("itl")) {
                // DACS --------------->
                DACS dacs = reb.getDacs();
                DACS dacsOriginal = rebOriginal.getDacs();

                // serial clocks
                msg = rebname + ": bringing up serials at " + dateStr() + "\n";
                outmsg.append(msg);
                this.raiseAlert(msg, AlertState.NOMINAL);

                LOG.info(msg);
                double volts_hi;
                double volts_lo;
                outmsg.append(rebname).append(": testing serial lines\n");
                volts_hi = dacChanOn("SCLK_HIGH", rebDevice, dacsOriginal, test_voltage);
                volts_lo = dacChanOn("SCLK_LOW", rebDevice, dacsOriginal, test_voltage);
                outmsg.append(rebname).append(": serials set to ").append(volts_hi).append("/").append(volts_lo).append(" V - status OK\n");

                // parallel clocks
                msg = rebname + ": bringing up parallels at " + dateStr() + "\n";
                outmsg.append(msg);
                this.raiseAlert(msg, AlertState.NOMINAL);
                LOG.info(msg);
                outmsg.append(rebname).append(": testing parallel lines\n");
                volts_hi = dacChanOn("PCLK_HIGH", rebDevice, dacsOriginal, test_voltage);
                volts_lo = dacChanOn("PCLK_LOW", rebDevice, dacsOriginal, test_voltage);
                outmsg.append(rebname).append(": parallels set to ").append(volts_hi).append("/").append(volts_lo).append(" V - status OK\n");

                // RG
                msg = rebname + ": bringing up RG\n";
                outmsg.append(msg);
                this.raiseAlert(msg, AlertState.NOMINAL);
                LOG.info(msg);
                outmsg.append(rebname).append(": testing RG lines at " + dateStr() + "\n");
                volts_hi = dacChanOn("RG_HIGH", rebDevice, dacsOriginal, test_voltage);
                volts_lo = dacChanOn("RG_LOW", rebDevice, dacsOriginal, test_voltage);
                outmsg.append(rebname).append(": RG set to ").append(volts_hi).append("/").append(volts_lo).append(" V - status OK\n");

                // BIASDACS ----->
                for (int idx = 0; idx < biases.length; idx++) {
                    BiasDACS bias = biases[idx];

                    outmsg.append("\n" + rebname + ": CCD ").append(idx).append(" :\n");

// Begin to apply the CCD frontside biases. Initial value of
// each voltage should be <500mV to verify that no short circuits are present (e.g. current <
// 10mA at 500mV); then bring biases to their recommended (ITL or E2V) levels.
                    // OG
                    outmsg.append(rebname + ": Testing OG for CCD " + idx + " at " + dateStr() + " - \n");
                    target = biasesOriginal[idx].getPValues()[BiasDACS.OG];
                    volts = biasDACChanOn("OG", rebDevice, idx, this.chooseTestVolt(test_voltage, target));
                    outmsg.append(rebname + ": OG set to " + volts + " V - status OK\n");
                    // GD
                    target = biasesOriginal[idx].getPValues()[BiasDACS.GD];
                    outmsg.append(rebname + ": Testing GD for CCD " + idx + " at " + dateStr() + " - \n");
                    volts = biasDACChanOn("GD", rebDevice, idx, this.chooseTestVolt(test_voltage, target));
                    outmsg.append(rebname + ": GD set to " + volts + " V - status OK\n");

                    // RD
                    outmsg.append(rebname + ": Testing RD for CCD " + idx + " at " + dateStr() + " - \n");
                    target = biasesOriginal[idx].getPValues()[BiasDACS.RD];
                    volts = biasDACChanOn("RD", rebDevice, idx, this.chooseTestVolt(test_voltage, target));
                    outmsg.append(rebname + ": RD set to " + volts + " V - status OK\n");

                    /* passed the short circuit checks , now  put the configured voltages */
                    // OG
                    outmsg.append(rebname + ": Applying OG for CCD " + idx + " at " + dateStr() + " - \n");
                    double volts_og = biasDACChanOn("OG", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.OG]);
                    outmsg.append(rebname + ": OG set to " + volts_og + " V - status OK\n");

                    // GD
                    outmsg.append(rebname + ": Applying GD for CCD " + idx + " at " + dateStr() + " - \n");
                    double volts_gd = biasDACChanOn("GD", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.GD]);
                    outmsg.append(rebname + ": GD set to " + volts_gd + " V - status OK\n");

                    // RD
                    outmsg.append(rebname + ": Applying RD for CCD " + idx + " at " + dateStr() + " - \n");
                    double volts_rd = biasDACChanOn("RD", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.RD]);
                    outmsg.append(rebname + ": RD set to " + volts_rd + " V - status OK\n");

                    // OD
                    outmsg.append(rebname + ": Applying OD for CCD " + idx + " at " + dateStr() + " - \n");
                    double volts_od = biasDACChanOn("OD", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.OD]);
                    outmsg.append(rebname + ": OD set to " + volts_od + " V - status OK\n");

                    // CSGATE on only now
                    outmsg.append(rebname + ": Turning on CSGATE for CCD " + idx + " at " + dateStr() + " - \n");
                    volts = biasDACChanOn("CS_GATE", rebDevice, idx, biasesOriginal[idx].getPValues()[BiasDACS.CS_GATE]);

                    Sleep(0.1);
                    outmsg.append(rebname + ": OG measured = " + getChanValue(rebDevice, "OG" + idx + "V", volts_og - biasDacVBand, volts_og + biasDacVBand) + " V\n");
                    outmsg.append(rebname + ": GD measured = " + getChanValue(rebDevice, "GD" + idx + "V", volts_gd - biasDacVBand, volts_gd + biasDacVBand) + " V\n");
                    outmsg.append(rebname + ": RD measured = " + getChanValue(rebDevice, "RD" + idx + "V", volts_rd - biasDacVBand, volts_rd + biasDacVBand) + " V\n");
                    outmsg.append(rebname + ": OD measured = " + getChanValue(rebDevice, "OD" + idx + "V", volts_od - biasDacVBand, volts_od + biasDacVBand) + " V\n");

                    outmsg.append(rebname + ": CSGATE set to " + volts + " V - status OK\n");

                }

                // ================================= end of ITL front bias powering =====================
            } else {
                throw new RuntimeException("unrecognizable ccdType!!!");
            }

// get firmware version and serial number
            try {
                int pulled_firm = rebDevice.getHwVersion();
                long pulled_id = rebDevice.getSerialNumber();

                msg = "\n ------- " + rebname + " is now powered and reporting 1-wire ID " + String.format("%x", pulled_id) + " and firmware version " + String.format("%x", pulled_firm) + " ------- \n\n";
                outmsg.append(msg);
                this.raiseAlert(msg, AlertState.NOMINAL);
                LOG.info(msg);

                // a full clean final load
                rebDevice.loadAspics(true);
                rebDevice.loadBiasDacs(true);
                rebDevice.loadDacs(true);

            } catch (Exception e) {
                LOG.info(outmsg.toString()); // dump preceding log
                LOG.info("FAILED WHILE TRYING TO GET FIRMWARE VERSION AND 1-WIRE ID");
                throw new RuntimeException(e);
            }

        }

        return outmsg.append(checkCurrentsAndTemperatures(rebnum, true)).toString();
    }

    /**
     * check the     power state of a REB
     *
     * @return a boolean which if true indicate a valid state of the REB biases
     */
    @Command(type = Command.CommandType.ACTION, description = "check state of REB biases")
    public boolean checkREBState(int rebnum) {

        boolean REBStateOK = true;
        
        StringBuilder outmsg = new StringBuilder();

        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() != rebnum) {
                continue;
            }

            String rebname = "REB" + rebnum;
            String msg = "Checking " + rebname + " at " + dateStr();
            LOG.warn(msg);
            this.raiseAlert(msg, AlertState.NOMINAL);

// =----------------------------------------------------------------
            BiasDACS[] biases = rebDevice.getREBConfig().getBiases();

            for (int idx = 0; idx < biases.length; idx++) {
                BiasDACS bias = biases[idx];

                outmsg.append("\n" + rebname + ": CCD ").append(idx).append(" :\n");

                for (int idac = 0; idac < 4; idac++) {
                    double band = biasDacVBand;
                    String dacname = "";
                    double volts = 0.0;
                    if (idac == 0) {
                        dacname = "RD";
                        volts = biases[idx].getPValues()[BiasDACS.RD];
                    } else if (idac == 1) {
                        dacname = "OD";
                        volts = biases[idx].getPValues()[BiasDACS.OD];
                        band = biasDacVWideBand_OD;
                    } else if (idac == 2) {
                        dacname = "GD";
                        volts = biases[idx].getPValues()[BiasDACS.GD];
                    } else if (idac == 3) {
                        dacname = "OG";
                        volts = biases[idx].getPValues()[BiasDACS.OG];
                    }
                    try {
                        outmsg.append(rebname + ": " + dacname + " measured = " + getChanValue(rebDevice, dacname + idx + "V", volts - band, volts + band) + " V\n");
                    } catch (Exception e) {
                        LOG.warn("FAILED " + dacname + " STATE CHECK FOR REB "+rebnum+"CCD "+idx);
                        REBStateOK = false;
                    }
                }

            }
        }
        return(REBStateOK);
    }

    /**
     * Safely power off
     *
     * @return A string representing the state at the end of the power off
     * exercise
     */
    @Command(type = Command.CommandType.ACTION, description = "safe power off")
    public String powerOff() {

        StringBuilder outmsg = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            outmsg.append(powerOff(rebDevice.getRebNumber()));
        }
        return (outmsg.toString());
    }

    /**
     * Safely power off
     *
     * @return A string representing the state at the end of the power off
     * exercise
     */
    @Command(type = Command.CommandType.ACTION, description = "safe power off")
    public String powerOff(int rebnum) {

        StringBuilder outmsg = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() != rebnum) {
                continue;
            }
            try {
                if (rebDevice.isBackBiasOn()) {
                    LOG.error("REB " + rebDevice.getRebNumber() + ": Attempted to to power off clocks while HV bias switch still ON. ABORTING!");
                    outmsg.append("REB " + rebDevice.getRebNumber() + ": Attempted to to power off clocks while HV bias switch still ON. ABORTING!");
                    continue;
                }
            } catch (Exception ex) {
                LOG.error("REB " + rebDevice.getRebNumber() + ": Unable to determine if HV bias is on ... ABORTING!");
                continue;
            }

            double noLim = 100.;
            if (Math.abs(getChanValue(rebDevice, "OD" + rebnum + "V", -noLim, noLim)) < this.biasDacVWideBand_OD) {

                String AbrtMsg = "Check of biases indicates sensors are in an unpowered state already.\n ABORTING powerOff procedure for REB" + rebnum + " !\n";
                LOG.error(AbrtMsg);
                this.raiseAlert(AbrtMsg, AlertState.NOMINAL);
                outmsg.append(AbrtMsg);
                continue;
            }

            LOG.warn("Powering OFF REB " + rebDevice.getRebNumber());
            outmsg.append("REB ").append(rebDevice.getRebNumber()).append(" : \n");

            // get REB current settings
            REB reb = rebDevice.getREBConfig();
            BiasDACS[] biases = reb.getBiases();
//            LOG.info("Number of biases in current config = " + biases.length);
            DACS dacs = reb.getDacs();
//
// 15.1 Biasing sequence for E2V rafts:
// " Power-up the CCD biases and rails using the nominal operating values in LCA?15131, in the sequence [RM1] 
// (1) RD, (2) OD, (3) GD, (4) OG, (5) SClk_Low, (6) SClk_High, (7) PClk_Low, (8) PClk_High, (9) RG_Low, (10) RG_High."
// we reverse this for powering off ...

            if (raftsCommands.getCcdType().toLowerCase().contains("e2v")) {

                // start turning off channel by channel in reverse order of power On
                // RG
                LOG.warn("turning off RG");
                dacChanOff("RG_HIGH", rebDevice, dacs);
                dacChanOff("RG_LOW", rebDevice, dacs);
                outmsg.append("RG OFF\n");

                // parallel clocks
                LOG.warn("turning off parallels");
                dacChanOff("PCLK_HIGH", rebDevice, dacs);
                dacChanOff("PCLK_LOW", rebDevice, dacs);
                outmsg.append("parallels OFF\n");

                // serial clocks
                LOG.warn("turning off serials");
                dacChanOff("SCLK_HIGH", rebDevice, dacs);
                dacChanOff("SCLK_LOW", rebDevice, dacs);
                outmsg.append("serials OFF\n");

// =----------------------------------------------------------------
                // now for the BiasDACS --------------------------------------------------------
                for (int idx = 0; idx < biases.length; idx++) {
                    BiasDACS bias = biases[idx];

                    outmsg.append("CCD ").append(idx).append(" :\n");

                    // CSGATE
                    LOG.warn("Turning off CSGATE for CCD " + idx);
                    biasDACChanOff("CS_GATE", rebDevice, idx, bias);
                    outmsg.append("CSGATE OFF\n");

                    // OG
                    LOG.warn("Turning off OG for CCD " + idx);
                    biasDACChanOff("OG", rebDevice, idx, bias);
                    outmsg.append("OG OFF\n");
                    // GD
                    LOG.warn("Turning off GD for CCD " + idx);
                    biasDACChanOff("GD", rebDevice, idx, bias);
                    outmsg.append("GD OFF\n");
                    // OD
                    LOG.warn("Turning off OD for CCD " + idx);
                    biasDACChanOff("OD", rebDevice, idx, bias);
                    outmsg.append("OD OFF\n");
                    // RD
                    LOG.warn("Turning off RD for CCD " + idx);
                    biasDACChanOff("RD", rebDevice, idx, bias);
                    outmsg.append("RD OFF\n");
                }
                // ================================= start of ITL front bias powering OFF =====================
// 15.2 Biasing sequence for ITL rafts:
// " Power-up the CCD biases and rails using the nominal operating values in LCA?15131, in the sequence [RM2]
// (1) SClk_Low, (2) SClk_High, (3) Pclk_Low, (4) Pclk_High, (5) RG_Low, (6) RG_High, (7) OG, (8) GD, (9) RD, (10) OD".
// we reverse this for powering off ...
            } else {

// =----------------------------------------------------------------
                for (int idx = 0; idx < biases.length; idx++) {
                    BiasDACS bias = biases[idx];

                    outmsg.append("CCD ").append(idx).append(" :\n");

                    // CSGATE
                    LOG.warn("Turning off CSGATE for CCD " + idx);
                    biasDACChanOff("CS_GATE", rebDevice, idx, bias);
                    outmsg.append("CSGATE OFF\n");

                    // OD
                    LOG.warn("Turning off OD for CCD " + idx);
                    biasDACChanOff("OD", rebDevice, idx, bias);
                    outmsg.append("OD OFF\n");
                    // RD
                    LOG.warn("Turning off RD for CCD " + idx);
                    biasDACChanOff("RD", rebDevice, idx, bias);
                    outmsg.append("RD OFF\n");
                    // OG
                    LOG.warn("Turning off OG for CCD " + idx);
                    biasDACChanOff("OG", rebDevice, idx, bias);
                    outmsg.append("OG OFF\n");
                    // GD
                    LOG.warn("Turning off GD for CCD " + idx);
                    biasDACChanOff("GD", rebDevice, idx, bias);
                    outmsg.append("GD OFF\n");
                }
                // start turning off channel by channel in reverse order of power On
                // RG
                LOG.warn("turning off RG");
                dacChanOff("RG_HIGH", rebDevice, dacs);
                dacChanOff("RG_LOW", rebDevice, dacs);
                outmsg.append("RG OFF\n");

                // parallel clocks
                LOG.warn("turning off parallels");
                dacChanOff("PCLK_HIGH", rebDevice, dacs);
                dacChanOff("PCLK_LOW", rebDevice, dacs);
                outmsg.append("parallels OFF\n");

                // serial clocks
                LOG.warn("turning off serials");
                dacChanOff("SCLK_HIGH", rebDevice, dacs);
                dacChanOff("SCLK_LOW", rebDevice, dacs);
                outmsg.append("serials OFF\n");
            }
        }
        return (outmsg.toString());
    }

    /**
     * power OFF sensors of a REB immediately
     *
     * @return A string representing the state at the end of the power off
     * exercise
     */
    @Command(type = Command.CommandType.ACTION, description = "immediate power off")
    public String powerOffNow(int rebnum) {

        StringBuilder outmsg = new StringBuilder();
        for (REBDevice rebDevice : rebDevices) {
            if (rebDevice.getRebNumber() != rebnum) {
                continue;
            }
            try {
                if (rebDevice.isBackBiasOn()) {
                    LOG.error("REB " + rebDevice.getRebNumber() + ": Attempted to to power off clocks while HV bias switch still ON. ABORTING!");
                    outmsg.append("REB " + rebDevice.getRebNumber() + ": Attempted to to power off clocks while HV bias switch still ON. ABORTING!");
                    continue;
                }
            } catch (Exception ex) {
                LOG.error("REB " + rebDevice.getRebNumber() + ": Unable to determine if HV bias is on ... ABORTING!");
                continue;
            }

            LOG.warn("Powering OFF REB " + rebDevice.getRebNumber());
            try {
                outmsg.append(resetREBCfg(rebDevice));
                outmsg.append("\n#BIAS DACs loaded = " + rebDevice.loadBiasDacs(true)+"\n");
                outmsg.append("#DACS loaded = " + rebDevice.loadDacs(true)+"\n");
            } catch (Exception e) {
                LOG.error(outmsg.toString()+"\nFailed to power OFF REB!\n " + e);
            }

        }
        return (outmsg.toString());
    }

    /**
     * Read a register address on a Reb.
     *
     * @param rebId The rebId: "Reb0", "R11:Reb0"
     * @param address The address of the register to read.
     * @return The value of the register.
     */
    @Command(description = "Read a register address", type = Command.CommandType.QUERY)
    public int readRegister(String rebId, int address) {
        Object reb = rebDevicesMap.get(rebId);
        if (reb == null || !(reb instanceof REBDevice)) {
            throw new IllegalArgumentException("Reb id : " + rebId + " is not valid.");
        }
        try {
            return ((REBDevice) reb).getRegister(address, 1).getValues()[0];
        } catch (Exception x) {
            throw new RuntimeException("Error during register read ", x);
        }
    }

    /**
     * loadSequence with modified geometry
     *
     * @param seq
     * @param prerows
     * @param readrows
     * @param postrows
     * @param precols
     * @param readcols
     * @param postcols
     * @return
     * @throws java.lang.Exception
     */
    @Command(description = "load sequencer and adjust row and col parameters")
    public List<Integer> loadSequencerAndReDimension(
            @Argument(description = "sequencer filename") String seq,
            @Argument(description = "Number of rows to skip before window") int prerows,
            @Argument(description = "Number of rows of the window") int readrows,
            @Argument(description = "Number of rows after window") int postrows,
            @Argument(description = "Number of columns to skip before readout window, including prescan") int precols,
            @Argument(description = "Number of columns to read") int readcols,
            @Argument(description = "Number of columns to discard after window ") int postcols) throws Exception {
        List<Integer> seqrply = raftsCommands.loadSequencer(seq);
        getRaftsCommands().setSequencerParameter("PreRows", prerows);
        getRaftsCommands().setSequencerParameter("ReadRows", readrows);
        getRaftsCommands().setSequencerParameter("PostRows", postrows);
        getRaftsCommands().setSequencerParameter("PreCols", precols);
        getRaftsCommands().setSequencerParameter("ReadCols", readcols);
        getRaftsCommands().setSequencerParameter("PostCols", postcols);

        return (seqrply);
    }

    /**
     * Write a register address on a Reb.
     *
     * @param rebId The id of the Reb: "Reb0", "R11:Reb0"
     * @param address The address to write to.
     * @param value The value to set the address to.
     */
    @Command(description = "Write a register address", type = Command.CommandType.ACTION)
    public void writeRegister(String rebId, int address, int value) {
        Object reb = rebDevicesMap.get(rebId);
        if (reb == null || !(reb instanceof REBDevice)) {
            throw new IllegalArgumentException("Reb id : " + rebId + " is not valid.");
        }
        try {
            ((REBDevice) reb).setRegister(address, new int[]{value});
        } catch (Exception x) {
            throw new RuntimeException("Error during register write ", x);
        }
    }

    /**
     * Set a key for the Primary header of all CCDs.
     *
     * @param headerName The name of the Header to set
     * @param headerValue The corresponding value
     * @param sticky Boolean value to specify if the provided Header keyword
     * value should be used across exposures. The default value is true. If
     * false is provided, the provided Header keyword value will be reset after
     * each exposure.
     */
    @Command(description = "Set a primary header value for all CCDs", type = Command.CommandType.QUERY)
    public void setHeader(String headerName, Object headerValue,
            @Argument(defaultValue = "true") boolean sticky) {
        globalHeaderKeywords.put(headerName, new HeaderKeywordValue(headerValue, sticky));
    }

    /**
     * Set a key for the Primary header of a given CCD identified by its id. The
     * Id could be an enumeration or a string ("S00", "R01:S00").
     *
     * @param ccdId The CCD id
     * @param headerName The name of the Header to set
     * @param headerValue The corresponding value
     * @param sticky Boolean value to specify if the provided Header keyword
     * value should be used across exposures. The default value is true. If
     * false is provided, the provided Header keyword value will be reset after
     * each exposure.
     */
    @Command(description = "Set a primary header value for a given CCD", type = Command.CommandType.QUERY)
    public void setCCDHeader(String ccdId, String headerName, Object headerValue,
            @Argument(defaultValue = "true") boolean sticky) {

        Map<String, HeaderKeywordValue> specificMap = ccdSpecificHeaderKeywords.get(ccdId);
        if (specificMap == null) {
            throw new IllegalArgumentException("CCD with id " + ccdId + " does not exist.");
        }
        specificMap.put(headerName, new HeaderKeywordValue(headerValue, sticky));
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave() throws IOException, InterruptedException, REBException {
        return exposeAcquireAndSave(exposureTimeInMillis);
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave(long exposureTimeInMillis) throws IOException, InterruptedException, REBException {
        return exposeAcquireAndSave(exposureTimeInMillis, openShutter);
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @param openShutter True/false if the shutter should be open
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave(long exposureTimeInMillis, boolean openShutter)
            throws IOException, InterruptedException, REBException {
        return exposeAcquireAndSave(exposureTimeInMillis, openShutter, actuateXED);
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @param openShutter True/false if the shutter should be open
     * @param actuateXED True/false if the FE55 should be involved
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave(long exposureTimeInMillis, boolean openShutter,
            boolean actuateXED) throws IOException, InterruptedException, REBException {
        return exposeAcquireAndSave(exposureTimeInMillis, openShutter, actuateXED, null);
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @param openShutter True/false if the shutter should be open
     * @param actuateXED True/false if the FE55 should be involved
     * @param filePattern The filePattern for the output fits files
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public List<String> exposeAcquireAndSave(long exposureTimeInMillis,
            boolean openShutter, boolean actuateXED,
            String filePattern) throws REBException, IOException, InterruptedException {

        LOG.info("Exposure requested: exptime = " + exposureTimeInMillis + ", shutter = " + openShutter + ", XED = " + actuateXED);
        List<Map<String, Integer>> mainmap = getRaftsCommands().getMainMap();
        boolean hasExposeMain = false;
        boolean hasBiasMain = false;
        boolean hasPocketPumpMain = false;
        boolean hasAcquireMain = false;

        String imageType = "";

        setHeader("TemperatureSetPoint", raftsCommands.getControlTemp(), true);

        try {
            imageType = (String) globalHeaderKeywords.get("ImageType").getValue();
        } catch (Exception e) {
            LOG.info("No image type set.");
        }

        for (REBDevice rebDevice : rebDevices) {
            setRebName(rebDevice.getRebNumber(), rebDevice.getName());
        }

        String mainRoutine = raftsCommands.getSequencerStart();

        for (Map<String, Integer> mymain : mainmap) {
            if (mymain.keySet().contains("PocketPump")) {
                hasPocketPumpMain = true;
            }
            if (mymain.keySet().contains("Expose")) {
                hasExposeMain = true;
            }
            if (mymain.keySet().contains("Acquire")) {
                hasAcquireMain = true;
            }
            if (mymain.keySet().contains("Bias")) {
                hasBiasMain = true;
            }
        }

        boolean runPreAcquisition = exposureTimeInMillis > 0;
        if (hasPocketPumpMain && hasExposeMain && hasAcquireMain && hasBiasMain) {
            if (exposureTimeInMillis > 0) {
                mainRoutine = "Acquire";
            } else {
                LOG.info("selecting bias mains");
                mainRoutine = "Bias";
            }
        } else {
            throw new RuntimeException("Required main routine missing!");
        }
        Predicate<BusMessage<?, ?>> imageStateStatusListenerFilter = BusMessageFilterFactory
                .messageClass(StatusSubsystemData.class).and(BusMessageFilterFactory.messageOrigin(subsys.getName()));
        ImageStatusDataListener imageStateListener = new ImageStatusDataListener(rebDevices.size());
        subsys.getMessagingAccess().addStatusMessageListener(imageStateListener,
                imageStateStatusListenerFilter);

        List<String> fitsFiles = new ArrayList<>();
        try {

//        if (System.getProperty("org.lsst.ccs.run.mode", "").equals("simulation")) {
//            // For now generate an image here. This will be moved somewhere else when the
//            // Sequencer loading works.
//            GeneratedImage generatedImage = PatternGeneratorFactory.generateImageForGeometry(geometry, "image", null);
//            PatternGeneratorFactory.exposeGeometryToGeneratedImage(geometry, generatedImage);
//        }
            //Start Acquisition
            try {
                getRaftsCommands().setSequencerParameter("ExposureTime", (int) (exposureTimeInMillis / 25.0)); // units are 25 ms        
                getRaftsCommands().setExposureTime(exposureTimeInMillis / 1000.); // set exposure time value in the metadata
                LOG.info("ExposureTime " + (int) (exposureTimeInMillis / 25.0) + " x 25ms periods");
            } catch (Exception e) {
                LOG.warn("Could not set exposure time on sequencer");
            }

            List<SequencerProc> sequencersToStartBeforeAcquisition = new ArrayList<>();
            //Open and Close the Shutter
            if (runPreAcquisition) {
                try {
                    if (!imageType.toLowerCase().contains("pump")) {
                        LOG.info("Setting Expose main for all devices ");
                        raftsCommands.setSequencerStart("Expose");
                    } else {
                        LOG.info("Setting PocketPump main for all devices ");
                        raftsCommands.setSequencerStart("PocketPump");
                    }
                } catch (Exception e) {
                    throw new RuntimeException("Error while setting the expose main ", e);
                }

                LOG.info("Exposure main setup: XED REB = " + xedREB + " , Shutter REB = " + shutterREB);
                for (REBDevice rebDevice : rebDevices) {
                    String rebname = rebDevice.getName();
                    try {

                        if ((openShutter && (rebname.toLowerCase().contains(shutterREB.toLowerCase())))
                                || (actuateXED && (rebname.toLowerCase().contains(xedREB.toLowerCase())))) {

                            int ExposureFlushPntr = -1;
                            try {
                                ExposureFlushPntr = rebDevice.getSequencer().getFunctionMap().get("ExposureFlush");
                            } catch (NullPointerException ne) {
                                ExposureFlushPntr = rebDevice.getSequencer().getSubroutineMap().get("ExposureFlush");
                            }

                            LOG.info("ExposureFlushPntr = " + ExposureFlushPntr);
                            rebDevice.getSequencer().setParameter("Exposure", ExposureFlushPntr);
                            LOG.info(rebname + " sequence set to actuate LVDS during exposure");

                        } else {

                            int SerialFlushPntr = -1;
                            try {
                                SerialFlushPntr = rebDevice.getSequencer().getFunctionMap().get("SerialFlush");
                            } catch (NullPointerException ne) {
                                SerialFlushPntr = rebDevice.getSequencer().getSubroutineMap().get("SerialFlush");
                            }

                            LOG.info("SerialFlushPntr = " + SerialFlushPntr);
                            rebDevice.getSequencer().setParameter("Exposure", SerialFlushPntr);
                            LOG.info(rebname + " sequence set to NOT actuate LVDS during exposure");

                        }

                        sequencersToStartBeforeAcquisition.add(rebDevice.getSequencer());

                    } catch (Exception e) {
                        throw new RuntimeException("Error while setting parameters ", e);
                    }
                }
            }

            //Starting the sequencers
            for (SequencerProc sequencerToStart : sequencersToStartBeforeAcquisition) {
                sequencerToStart.startSequencer();
            }

            if (exposureTimeInMillis > 0) {

                //Wait for the sequencers to terminate their pre-acquisition execution
                long waitForShutter = exposureTimeInMillis + 2000;
                if (sequencersToStartBeforeAcquisition.size() > 0) {
                    LOG.info("Waiting " + waitForShutter + " ms for exposure sequencers to terminate");
                    for (SequencerProc sequencerToStart : sequencersToStartBeforeAcquisition) {
                        sequencerToStart.waitDone((int) waitForShutter);
                    }
                }
            }

            //Set the acquisition main routine
            try {
                raftsCommands.setSequencerStart(mainRoutine);
            } catch (Exception e) {
                throw new RuntimeException("Error while setting the " + mainRoutine, e);
            }

            //Go on with the acquisition now.        
            for (REBDevice reb : rebDevices) {
                reb.getSequencer().resetError();
            }
            LOG.info("starting image acquisition at " + System.currentTimeMillis() + " millis");
            globalProc.acquireImage();

            imageStateListener.waitForImages(30000);

            LOG.info("finished image acquisition at " + System.currentTimeMillis() + " millis");

            MetaDataSet commonMetaDataSet = getCommonMedaDataSet();

            try {
                boolean writeFits = true;
                for (REBDevice reb : rebDevices) {
                    if (filePattern != null) {
                        if (!filePattern.isEmpty()) {
                            reb.getImageProc().setFitsFileNamePattern(filePattern);
                        } else {
                            writeFits = false;
                        }
                    }
                    if (writeFits) {
                        List<String> rebFiles = reb.getImageProc().saveFitsImage(null, new TS8FitsHeaderMetadataProvider(this, commonMetaDataSet));
                        for (String rebFile : rebFiles) {
                            fitsFiles.add(rebFile);
                        }
                    }
                }
            } catch (RaftException x) {
                //TODO: understand why a RaftException would be thrown here.
                throw new IOException("Error while writing FITS files", x);
            }
            LOG.info("exposure complete");

        } finally {
            LOG.info("Removing listener " + imageStateListener);
            subsys.getMessagingAccess().removeStatusMessageListener(imageStateListener);
            LOG.info("Clearing non sticky header keywords");
            clearNonStickyHeaderKeywordValues();
        }

        LOG.info("Returning " + fitsFiles);
        return fitsFiles;
    }

    /**
     * Start an acquisition and save the resulting images for the given
     * parameters. This method does not return until all of the files have been
     * written. Events will be published on the buses when files are available
     * (TODO:True?, if so which events).
     *
     * @param exposureTimeInMillis The exposure time in milliseconds
     * @param openShutter True/false if the shutter should be open
     * @return A list of the file written (as Strings)
     * @throws REBException If there is an error interacting with the REBs
     * @throws IOException if there is an error while writing the fits files.
     * @throws InterruptedException If there is an interrupt waiting for the
     * image acquisition to complete
     */
    @Command(description = "Start an acquisition and save the images to fits.", type = Command.CommandType.ACTION)
    public String exposeAcquireAndSaveVerbose(long exposureTimeInMillis,
            boolean openShutter, boolean actuateXED, String filePattern)
            throws IOException, InterruptedException, REBException {
        List<String> msg = exposeAcquireAndSave(exposureTimeInMillis, openShutter, actuateXED, filePattern);
        return (sa.getAllLast().toString());
    }

    /**
     * Set the default exposure time in milliseconds.
     *
     * @param millis Exposure time in milliseconds.
     */
    @Command(description = "Set the default exposure time.", type = Command.CommandType.ACTION)
    public void setDefaultExposureTime(int millis) throws Exception {
        if (millis < 0) {
            throw new IllegalArgumentException("Illegal exposure time: it cannot be negative.");
        }
        this.exposureTimeInMillis = millis;

    }

    /**
     * Set the default exposure time in milliseconds.
     *
     * @param millis Exposure time in milliseconds.
     */
    @Command(description = "Set the sequencer exposure time.", type = Command.CommandType.ACTION)
    public void setSequencerExposureTime(int millis) throws Exception {
        if (millis < 0) {
            throw new IllegalArgumentException("Illegal exposure time: it cannot be negative.");
        }
        getRaftsCommands().setSequencerParameter("ExposureTime", (int) (millis / 25.0)); // units are 25 ms        

    }

    /**
     * Set the default open shutter flag for an exposure.
     *
     * @param open True/False if the shutter should be opened during the
     * exposure.
     */
    @Command(description = "Set the exposure default open shutter flag.", type = Command.CommandType.ACTION)
    public void setDefaultOpenShutter(boolean open
    ) {
        this.openShutter = open;
    }

    /**
     * Set the exposure default Actuated XED.
     *
     * @param actuate True/False for XED actuation.
     */
    @Command(description = "Set the exposure default XED actuation.", type = Command.CommandType.ACTION)
    public void setDefaultActuateXED(boolean actuate
    ) {
        this.actuateXED = actuate;
    }

    /*
     * THE FOLLOWING METHODS ARE NEEDED BY THE GUI
     *
     * TODO: IS THERE A WAY TO GENERALIZE THIS PROCESS SINCE ALL THESE METHODS
     * ARE ALSO DEFINED IN RAFTSMAIN?
     *
     */
    /**
     * Gets the full monitoring state. This is intended to be called by GUIs
     * during initialization
     *
     * @return The full Raft state
     */
    @Command(type = Command.CommandType.QUERY,
            description = "Gets the full Raft module state")
    public RaftFullState getFullState() {
        return new RaftFullState(new RaftState((int) periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis()), subsys.getMonitor().getFullState());
    }

    private double biasDACChanOn(String name, REBDevice rebDevice, int idx, double voltage) {
        REB reb = rebDevice.getREBConfig();
        int dacidx;
        switch (name) {
            case "GD":
                dacidx = BiasDACS.GD;
                break;
            case "OD":
                dacidx = BiasDACS.OD;
                break;
            case "RD":
                dacidx = BiasDACS.RD;
                break;
            case "OG":
                dacidx = BiasDACS.OG;
                break;
            case "CSGATE":
                dacidx = BiasDACS.CS_GATE;
                break;
            case "CS_GATE":
                dacidx = BiasDACS.CS_GATE;
                break;
            default:
                throw new IllegalArgumentException("Illegal Bias DAC name:  " + name);
        }

        reb.getBiases()[idx].getPValues()[dacidx] = voltage;
        LOG.info("setting new config with " + name + " voltage at " + voltage + "V");
        rebDevice.setREBConfig(reb);
        LOG.info(showREBCfg(rebDevice));

        try {
            int ndacs_loaded = rebDevice.loadBiasDacs(true);
            LOG.info("#dacs successfully loaded = " + ndacs_loaded);
            if (!name.contains("OD")) {
                checkCurrents(rebDevice.getRebNumber());
            } else {
                checkCurrents(rebDevice.getRebNumber(), true);
            }
        } catch (Exception e) {
            /* leave this being handled by checkCurrents??
            LOG.error("Channel" + name +" failed shorts test and is being set to 0V");
            reb.getBiases()[idx].getPValues()[dacidx] = 0.;
            rebDevice.setREBConfig(reb);
            try {
                int ndacs_loaded = rebDevice.loadBiasDacs(true);
            } catch (Exception f) {
                LOG.error("Setting channel off failed! " + f);
            }
             */
            throw new RuntimeException("FAILED CHECK AT VOLTAGE : ", e);
        }

        double new_volt = reb.getBiases()[idx].getPValues()[dacidx];

        String msg = rebDevice.getName() + ": " + name + " set to " + new_volt + " V - status OK\n";
        this.raiseAlert(msg, AlertState.NOMINAL);
        LOG.info(msg);

        return new_volt;
    }

    private double dacChanOn(String name, REBDevice rebDevice, DACS dacsOriginal, double mag_test_voltage) {
        REB reb = rebDevice.getREBConfig();

        int dacidx = -1;
        switch (name) {
            case "PCLK_HIGH":
                dacidx = DACS.PCLK_HIGH;
                break;
            case "PCLK_LOW":
                dacidx = DACS.PCLK_LOW;
                break;
            case "SCLK_HIGH":
                dacidx = DACS.SCLK_HIGH;
                break;
            case "SCLK_LOW":
                dacidx = DACS.SCLK_LOW;
                break;
            case "RG_HIGH":
                dacidx = DACS.RG_HIGH;
                break;
            case "RG_LOW":
                dacidx = DACS.RG_LOW;
                break;
            default:
                throw new IllegalArgumentException("Illegal DAC name: " + name);
        }

        double test_voltage = Math.copySign(mag_test_voltage, dacsOriginal.getPValues()[dacidx]);

        try {
            reb.getDacs().getPValues()[dacidx] = test_voltage;
            LOG.info("setting new config with " + name + " test_voltage at " + test_voltage + "V");
            rebDevice.setREBConfig(reb);
            LOG.info(showREBCfg(rebDevice));
            rebDevice.loadDacs(true);
            this.checkCurrents(rebDevice.getRebNumber());
        } catch (Exception e) {
            /* leave this being handled by checkCurrents??
            LOG.error("Channel" + name +" failed shorts test and is being set to 0V");
            reb.getDacs().getPValues()[dacidx] = 0.;
            rebDevice.setREBConfig(reb);
            try {
                int ndacs_loaded = rebDevice.loadDacs(true);
            } catch (Exception f) {
                LOG.error("Setting channel off failed! " + f);
            }
             */
            throw new RuntimeException("FAILED CHECK OF " + name + " AT TEST VOLTAGE\n ALERT: " + this.last_alertStr + "\n : ", e);
        }

        try {
            reb.getDacs().getPValues()[dacidx] = dacsOriginal.getPValues()[dacidx];
            LOG.info("setting new config with " + name + " voltage at " + dacsOriginal.getPValues()[dacidx] + "V");
            rebDevice.setREBConfig(reb);
            LOG.info(showREBCfg(rebDevice));
            rebDevice.loadDacs(true);
//            this.checkCurrents(rebDevice.getRebNumber());
        } catch (Exception e) {
            throw new RuntimeException("FAILED CHECK OF " + name + " AT FULL VOLTAGE\n ALERT: " + this.last_alertStr + "\n : ", e);
        }

        return reb.getDacs().getPValues()[dacidx];
    }

    private void biasDACChanOff(String name, REBDevice rebDevice, int idx, BiasDACS bias) {
        REB reb = rebDevice.getREBConfig();

        int dacidx = 0;
        switch (name) {
            case "GD":
                dacidx = BiasDACS.GD;
                break;
            case "OD":
                dacidx = BiasDACS.OD;
                break;
            case "RD":
                dacidx = BiasDACS.RD;
                break;
            case "OG":
                dacidx = BiasDACS.OG;
                break;
            case "CSGATE":
                dacidx = BiasDACS.CS_GATE;
                break;
            case "CS_GATE":
                dacidx = BiasDACS.CS_GATE;
                break;
            default:
                throw new IllegalArgumentException("Illegal Bias DAC name: " + name);
        }

        try {
            reb.getBiases()[idx].getPValues()[dacidx] = 0.0;

            LOG.info("setting new config with " + name + " DAC set to at 0");
            rebDevice.setREBConfig(reb);
            this.showREBCfg(rebDevice);
            rebDevice.loadBiasDacs(true);
        } catch (Exception e) {
            throw new RuntimeException("FAILED CHECK AT DAC==0 : ", e);
        }

    }

    private void dacChanOff(String name, REBDevice rebDevice, DACS dacs) {
        REB reb = rebDevice.getREBConfig();

        int dacidx = -1;
        switch (name) {
            case "PCLK_HIGH":
                dacidx = DACS.PCLK_HIGH;
                break;
            case "PCLK_LOW":
                dacidx = DACS.PCLK_LOW;
                break;
            case "SCLK_HIGH":
                dacidx = DACS.SCLK_HIGH;
                break;
            case "SCLK_LOW":
                dacidx = DACS.SCLK_LOW;
                break;
            case "RG_HIGH":
                dacidx = DACS.RG_HIGH;
                break;
            case "RG_LOW":
                dacidx = DACS.RG_LOW;
                break;
            default:
                throw new IllegalArgumentException("Illegal DAC name! " + name);
        }

        try {
            reb.getDacs().getPValues()[dacidx] = 0.0;
            LOG.info("setting new config with " + name + " DAC voltage = 0.0");
            rebDevice.setREBConfig(reb);
            this.showREBCfg(rebDevice);
            rebDevice.loadDacs(true);
//            this.checkCurrentsAndTemperatures(rebDevice.getRebNumber());
        } catch (Exception e) {
            throw new RuntimeException("FAILED CHECK AT DAC voltage 0.0 : ", e);
        }

    }

    private class ImageStatusDataListener implements StatusMessageListener {

        private final CountDownLatch latch;

        ImageStatusDataListener(int nImages) {
            latch = new CountDownLatch(nImages);
        }

        @Override
        public void onStatusMessage(StatusMessage msg) {
            StatusSubsystemData statusData = (StatusSubsystemData) msg;
            if (statusData.getDataKey().equals(ImageState.KEY)) {
                latch.countDown();
            }
        }

        public void waitForImages(long timeout) throws InterruptedException {
            boolean done = latch.await(timeout, TimeUnit.MILLISECONDS);
            if (!done) {
                throw new RuntimeException("Exceeded " + timeout + "mS wait period to receive all images. " + latch.getCount() + " images have not been received.");
            }
        }

    }

    //Utility methods
    List<REBDevice> getRebDevices() {
        return rebDevices;
    }

    Geometry getGeometry() {
        return geometry;
    }

    MetaDataSet getCommonMedaDataSet() {
        //Common metadata to all images
        MetaDataSet metaDataSet = new MetaDataSet();
        metaDataSet.addMetaDataMap("StatusAggregator", getStatusAggregator().getAllLast());
        metaDataSet.addMetaDataMap("TS8GlobalKeywords", getGlobalPrimaryHeaderKeywordsMap());
        return metaDataSet;
    }

    private StatusAggregator getStatusAggregator() {
        return sa;
    }

    private Map<String, Object> getGlobalPrimaryHeaderKeywordsMap() {
        return buildMapOfHeaderKeywordValues(globalHeaderKeywords);
    }

    Map<String, Object> getPrimaryHeaderKeywordsMapForCCD(String ccdId) {
        return buildMapOfHeaderKeywordValues(ccdSpecificHeaderKeywords.get(ccdId));
    }

    RaftsCommands getRaftsCommands() {
        return raftsCommands;
    }

    private Map<String, Object> buildMapOfHeaderKeywordValues(Map<String, HeaderKeywordValue> map) {
        Map<String, Object> result = new HashMap<>();
        for (String headerKeyword : map.keySet()) {
            result.put(headerKeyword, map.get(headerKeyword).getValue());
        }
        return result;
    }

    private void fillHeaderKeywordMaps(Geometry geometry) {
        if (geometry instanceof Raft) {
            Raft raft = (Raft) geometry;
            for (Reb reb : raft.getChildrenList()) {
                for (CCD ccd : reb.getChildrenList()) {
                    ccdSpecificHeaderKeywords.put(ccd.getUniqueId(), new HashMap<>());
                }
            }
        } else {
            throw new RuntimeException("This class is currently designed to support a single Raft.");
        }
    }

    protected void clearNonStickyHeaderKeywordValues() {
        clearNonStickyHeaderKeywordValuesFromMap(globalHeaderKeywords);
        for (Map<String, HeaderKeywordValue> map : ccdSpecificHeaderKeywords.values()) {
            clearNonStickyHeaderKeywordValuesFromMap(map);
        }
    }

    private void clearNonStickyHeaderKeywordValuesFromMap(Map<String, HeaderKeywordValue> map) {
        Iterator<Map.Entry<String, HeaderKeywordValue>> iter = map.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, HeaderKeywordValue> entry = iter.next();
            if (!entry.getValue().isSticky()) {
                iter.remove();
            }
        }
    }

    //Private class used to set Header Keyword values.
    private class HeaderKeywordValue {

        private final Object value;
        private final boolean sticky;

        HeaderKeywordValue(Object value, boolean sticky) {
            this.value = value;
            this.sticky = sticky;
        }

        Object getValue() {
            return value;
        }

        boolean isSticky() {
            return sticky;
        }
    }

    class ChanParams {

        private final int type;
        private final int addr;

        ChanParams(int type, int addr) {
            this.type = type;
            this.addr = addr;
        }

    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the LSST assigned CCD Serial Number")
    public void setLsstSerialNumber(String ccdId, String serialNumber) {
        setCCDHeader(ccdId, "CCDSerialLSST", serialNumber, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the Measured CCD Temperature")
    public void setMeasuredCCDTemperature(String ccdId, double ccdtemp) {
        LOG.info("Setting measured CCD temperature to " + ccdtemp);
        setCCDHeader(ccdId, "MeasuredTemperature", ccdtemp, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the Measured CCD HVBias")
    public void setMeasuredCCDBSS(String ccdId, double hvbss) {
        LOG.info("Setting measured CCDBSS to " + hvbss);
        setCCDHeader(ccdId, "MeasuredCCDBSS", hvbss, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the Monochromator Wavelength")
    public void setMonoWavelength(double wl) {
        LOG.info("Setting header MonochromatorWL to " + wl);
        setHeader("MonochromatorWavelength", wl, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Set the Manufacturer assigned CCD Serial Number")
    public void setManufacturerSerialNumber(String ccdId, String serialNumber) {
        setCCDHeader(ccdId, "CCDSerialManufacturer", serialNumber, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Sets the test type property for the output filename")
    public void setTestType(String ttype) {
        setHeader("TestType", ttype, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Sets the name of the test stand station")
    public void setTestStand(String tsname) {
        setHeader("TestStand", tsname, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Sets the image type property for the output filename")
    public void setImageType(String itype) {
        setHeader("ImageType", itype, true);
    }

    @Command(type = Command.CommandType.QUERY,
            description = "Sets the sequence info property for the output filename")
    public void setSeqInfo(Integer seqNum) {
        setHeader("SequenceInfo", String.format("%03d", seqNum), true);
        raftsCommands.setSequenceNumber(seqNum);
    }

    @Command
    public void showPhotoDiodeAnalysis(String pddatfile)
            throws IOException, FitsException {
        fitsUtils.showPhotoDiodeAnalysis(new File(pddatfile));
    }

    @Command
    public void addBinaryTable(String pddatfile, String fitsfile,
            String extnam, String c1name, String c2name, double tstart)
            throws IOException, FitsException {
        fitsUtils.updatePhotoDiodeValues(new File(pddatfile),
                new File(fitsfile), extnam, c1name, c2name, tstart);
    }

    @Command
    public double getFluxStats(String fitsfile) throws FitsException,
            IOException {
        return (fitsUtils.getFluxStats(new File(fitsfile)));
    }

    /**
     ***************************************************************************
     **
     ** Sleep - what a waste
     * **************************************************************************
     */
    public void Sleep(double secs) {
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            LOG.error("Rude awakening!" + ex);
        }
    }

}
