package org.lsst.ccs.subsystem.teststand;

import java.util.Map;
import java.util.HashMap;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.agilent.E3631;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 ***************************************************************************
 **
 ** Program to test the Agilent E3631 device driver * (based on Owen Saxton's
 * Agilent driver tester) * * @author Homer Neal *
 * **************************************************************************
 */
public class AgilentDevice extends Device implements LEDDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("VOLTS", Channel.TYPE_VOLTS);
    }

    private E3631 agips = null;

    private boolean isconnected = false;

    TSConfig cfg = new TSConfig();

    private String outname = "";

    private double[] runBias = new double[cfg.MAXSTATES]; // value overriden by groovy configuration

    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;

    boolean failedToInitialize = false;

    /**
     ***************************************************************************
     **
     ** AgilentDevice constructor - put Agilent in initial subsystem state
     *
     ** @param host device address
     ** @param port port or baud * * @param runBias CCD bias voltage while
     * running *
     * **************************************************************************
     */
    public AgilentDevice(String host, int port)
            throws DriverException {
        isconnected = true;
        try {
            agips = new E3631();

            if (agips != null) {
                agips.openftdi(host, port); // open connection
            }
        } catch (DriverException f) {
            System.out.println("Failed to open connection to Agilent device!");
            isconnected = false;
        }
        if (isconnected) {
            if (agips != null) {
                for(E3631.OutputName outname : E3631.OutputName.values()) 
                    agips.setVoltage(outname.getValue(), 0.0); // set voltage to 0.
            }
            if (agips != null) {
                for(E3631.OutputName outname : E3631.OutputName.values()) 
                    agips.outputOff(outname.getValue());

                kstate = TSState.pwrstates.OFF;
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Constructor *
     * **************************************************************************
     */
    public AgilentDevice() {
        try {
            agips = new E3631();
        } catch (DriverException f) {
            System.out.println("Failed to create to Agilent device instance!");
        }

    }

    /**
     ***************************************************************************
     **
     ** Performs configuration. *
     * **************************************************************************
     */
    @Override
    protected void initDevice() {
        fullName = "Agilent module";
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (agips != null) {
                agips.close();
            }
            kstate = TSState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("Agilent device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the device
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        if (isconnected) {
            log.debug("Agilent device is already initialized and connected to:");
            try {
                log.debug(agips.getID());
            } catch (Exception e) {
                if (!failedToInitialize) {
                    log.error("Agilent device failed to respond to ident request! The initialization has FAILED.");
                    failedToInitialize = true;
                }
                isconnected = false;
            }
            log.debug("Setting device online.");
            setOnline(true);
            kstate = TSState.pwrstates.OK;
        } else {
            if (!failedToInitialize) {
                log.error("Tried to initialize unconnected LED device!");
                failedToInitialize = true;
            }

        }
    }

    /**
     ***************************************************************************
     **
     ** Opens FTDI connection to a device. *
     * **************************************************************************
     */
    @Command(name = "openftdi", description = "Open FTDI connection to device")
    public String openftdi(@Argument(name = "serialname", description = "Serial device name") String host,
            @Argument(name = "baud", description = "baud rate") int port
    ) throws DriverException {
        agips.openftdi(host, port);
        kstate = TSState.pwrstates.OK;
        return agips.getID();
    }

  
    /**
     ***************************************************************************
     **
     ** Sets the voltage range *
     * **************************************************************************
     */
    @Command(name = "setvoltagerange", description = "Set the voltage range. Can be 10V, 50V or 500V")
    public void setVoltageRange(@Argument(name = "value", description = "Voltage Range to set") double value) {
        log.error("function not available!");
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage. *
     * **************************************************************************
     */
    @Command(name = "selectoutput", description = "Select the output for control")
    public void selectOutput(@Argument(name = "value", description = "outname : P6V,P25V,N25V") String value) {
        outname = value;
        try {
            if (agips != null) {
                agips.selectOutput(outname);
            }
        } catch (DriverException f) {
            log.error("Failed to select the Agilent output for control");
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage. *
     * **************************************************************************
     */
    @Command(name = "setvoltage", description = "Set the voltage")
    public void setVoltage(@Argument(name = "value", description = "Voltage to set") double value) {
        try {
            if (kstate.equals(TSState.pwrstates.OK) || kstate.equals(TSState.pwrstates.OFF)) {
                if (agips != null) {
                    agips.setVoltage(outname, value);
                }
            }
        } catch (DriverException f) {
            log.error("Failed to set the Agilent voltage!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the voltage to the value needed for acquisition. *
     * **************************************************************************
     */
    @Command(name = "setvoltageacq", description = "Set the voltage to the acquisition set point")
    public void setVoltageAcq(int cfgstate) {
        double value = getRunBias(cfgstate);
        rampVolts(5.0, value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the bias value for acquisition *
     * **************************************************************************
     */
    @Command(name = "setrunbias", description = "Sets the bias value for acquisition")
    public void setRunBias(double runBias, int cfgstate) {
        this.runBias[cfgstate] = runBias;
        return;
    }

    /**
     ***************************************************************************
     **
     ** Returns the bias value for acquisition *
     * **************************************************************************
     */
    @Command(name = "getrunbias", description = "Returns the bias value for acquisition")
    public double getRunBias(int cfgstate) {
        return (runBias[cfgstate]);
    }


    /**
     ***************************************************************************
     **
     ** Ramps to the desired voltage *
     * **************************************************************************
     */
    @Command(name = "rampvolts", description = "ramp the voltage")
    public void rampVolts(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "value", description = "Voltage to ramp to") double value) {
        log.error("function not available yet!");
        /* -----------------
         try {
         if (agips != null) {
         //                agips.rampVolts(duration, value);
         }
         } catch (DriverException f) {
         log.error("Agilent driver failed to ramp volts!");
         }
         --------------- */
    }

    /**
     ***************************************************************************
     **
     ** Ramps to the desired voltage over a given time and for nsteps
     *
     * @param duration
     * @param value
     * @param nsteps
     * **************************************************************************
     */
    @Command(name = "rampvolts", description = "ramp the voltage")
    public void rampVolts(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "value", description = "Voltage to ramp to") double value,
            @Argument(name = "nsteps", description = "number of steps") int nsteps) {
        log.error("function not available");
        /* ----------------
         try {
         if (agips != null) {
                
         //                agips.rampVolts(duration, value, nsteps);
         }
         } catch (DriverException f) {
         log.error("Agilent driver failed to ramp volts!");
         }
         --------------------*/

    }

    /**
     ***************************************************************************
     **
     ** Shows the set voltage. *
     * **************************************************************************
     */
    @Command(name = "showvoltage", description = "Show the set voltage")
    public String getVoltage() throws DriverException {
        return "Voltage = " + agips.getVoltage();
    }

    /**
     ***************************************************************************
     **
     ** Reads the actual voltage. *
     * **************************************************************************
     */
    @Command(name = "readvoltage", description = "Read the actual voltage")
    public double readVoltage() {
        double volt = 0.0;
        try {
            volt = agips.getVoltage(outname);
        } catch (DriverException e) {
            log.error("Agilent driver failed to read voltage!");
        }
        return (volt);
    }

    /**
     ***************************************************************************
     **
     ** Sets the soft voltage limit. *
     * **************************************************************************
     */
    @Command(name = "setvoltlim", description = "Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name = "value",
            description = "Voltage limit to set") double value) throws DriverException {
        log.error("function not available!");
    }

    /**
     ***************************************************************************
     **
     ** Shows the soft voltage limit. *
     * **************************************************************************
     */
    @Command(name = "showvoltlim", description = "Show the soft voltage limit")
    public String getVoltageLimit() throws DriverException {
        return "function not available";
    }

    /**
     ***************************************************************************
     **
     ** Sets the current range *
     * **************************************************************************
     */
    @Command(name = "setcurrentrange", description = "Set the current range")
    public void setCurrentRange(@Argument(name = "value", description = "Current Range to set") double value) throws DriverException {
        if (agips != null) {
//            agips.setCurrentRange(value);
            log.error("function not available!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Sets the current. *
     * **************************************************************************
     */
    @Command(name = "setcurrent", description = "Set the current")
    public void setCurrent(@Argument(name = "value", description = "Current to set") double value) {
        try {
            if (agips != null) {
                agips.setCurrent(outname, value);
            }
        } catch (DriverException f) {
            log.error("Agilent driver failed to set current!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Zero correct the current *
     * **************************************************************************
     */
    @Command(name = "zerocorrectcurrent", description = "Zero correct the current")
    public void zeroCorrectCurrent() throws DriverException {
        if (agips != null) {
            log.error("function not available!");
//            agips.zeroCorrectCurrent();
        }
    }

    /**
     ***************************************************************************
     **
     ** Shows the set current. *
     * **************************************************************************
     */
    @Command(name = "showcurrent", description = "Show the set current")
    public double getCurrent() {
        double curr = 0.0;
        log.error("can only show the current measurement");
        /* ----------------------------
         try {
         //            curr = agips.getCurrent();
         } catch (DriverException e) {
         log.error("Agilent driver failed to get the current!!");
         }
         ------------------------ */
        return (curr);
    }

    /**
     ***************************************************************************
     **
     ** Reads the actual current. *
     * **************************************************************************
     */
    @Command(type = CommandType.ACTION, name = "readcurrent", description = "Read the actual current")
    public double readCurrent() {
        log.debug("AgilentDevice executing readcurrent operation:");

        double curr = 0.0;
        try {
            curr = agips.getCurrent();
        } catch (DriverException e) {
            log.error("Agilent driver failed to retrieve the current measurement!");
        }

        return (curr);
    }

    /**
     ***************************************************************************
     **
     ** Sets the soft current limit. *
     * **************************************************************************
     */
    @Command(name = "setcurrlim", description = "Set the soft current limit")
    public void setCurrentLimit(@Argument(name = "value",
            description = "Current limit to set") double value) throws DriverException {
        if (agips != null) {
//            agips.setCurrentLimit(value);
            log.error("function not available");
        }
    }

    /**
     ***************************************************************************
     **
     ** Shows the soft current limit. *
     * **************************************************************************
     */
    @Command(name = "showcurrlim", description = "Show the soft current limit")
    public String getCurrentLimit() throws DriverException {
//        return "Current limit = " + agips.getCurrentLimit();
        return "function not available";
    }


    /**
     ***************************************************************************
     **
     ** Turns the output on or off. *
     * **************************************************************************
     */
    @Command(name = "setoutput", description = "Turn output on or off. 1 ==> ON")
    public void setOutput(@Argument(name = "state",
            description = "Output state: on or off") int state) {
        try {
            if (kstate.equals(TSState.pwrstates.OK) || kstate.equals(TSState.pwrstates.OFF)) {
                if (agips != null) {
                    if (state == 1) {
                        agips.outputOn(outname);
                    } else {
                        agips.outputOff(outname);
                    }
                }
            }
        } catch (DriverException e) {
            log.error("Agilent driver failed to set the output state!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Shows the output state. *
     * **************************************************************************
     */
    @Command(name = "showoutput", description = "Show the output state")
    public boolean showOutput() {
        boolean state = (1 == 0);
        log.error("function not available");
        return (state);
    }

    /**
     ***************************************************************************
     **
     ** Converts an array of numbers to a string. *
     * **************************************************************************
     */
    private StringBuilder getString(double[] values) {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }

    /**
     ***************************************************************************
     **
     ** Converts a boolean to on/off. *
     * **************************************************************************
     */
    private String getOnOff(boolean on) {
        return on ? "on" : "off";
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_VOLTS && iType != Channel.TYPE_POWER) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
            if (type == Channel.TYPE_VOLTS) {
                log.debug("setting channel online - chan=" + chan);
                setOnline(true);
            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    @Command(name = "reset", description = "reset trips etc...")
    public void reset() throws DriverException {
        if (agips != null) {
            agips.reset();
        }
        kstate = TSState.pwrstates.OFF;

    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        log.debug("AgilentDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            if (type == Channel.TYPE_VOLTS) {
                value = readVoltage();
            }
        } catch (Exception e) {
            log.debug("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    @Command(name = "setstate", description = "set agilent device status")
    public void setState(int istate) {
        kstate = TSState.pwrstates.values()[istate];
    }

    @Command(name = "getstate", description = "get agilent device status")
    public int getState() {
        return (kstate.ordinal());
    }

}
