package org.lsst.ccs.subsystem.teststand;

//import java.util.HashSet;
//import java.util.Set;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 ***************************************************************************
 **
 ** Program to test the Keithley N6487 device driver * (based on Owen Saxton's
 * Agilent driver tester) * * @author Homer Neal *
 * **************************************************************************
 */
public class KeithleySimDevice extends KeithleyDevice { //implements BiasDevice, PhotoDiodeDevice {

    private boolean isconnected = false;
    private double nplc = 0.; // period in terms of # of pulse line counts for buffered reads
    private int nreads = 1; // #reads for buffered reads

    private String connType;
    private String devName;
    private int baudRate;

    /**
     ***************************************************************************
     **
     ** put Keithley in initial subsystem state * * @param host device address
     *
     * * * @param port port or baud * * @param runBias CCD bias voltage while
     * running *
     * **************************************************************************
     */
    public KeithleySimDevice(String host, int port)
            throws DriverException {
        this.setState(TSState.pwrstates.ON.ordinal());
    }

    /**
     ***************************************************************************
     **
     ** Constructor *
     * **************************************************************************
     */
    public KeithleySimDevice() {
//        scpi = n64;
    }

    /**
     ***************************************************************************
     **
     ** Performs configuration. *
     * **************************************************************************
     */
    @Override
    protected void initDevice() {
        fullName = "Keithley module";
    }

    /**
     ***************************************************************************
     **
     ** Opens connection to a device using all defaults *
     * **************************************************************************
     */
    @Command(name = "open", description = "Open connection to device using defaults")
    public void open() throws DriverException {
//        n64.open();
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
    }

    /**
     ***************************************************************************
     **
     ** Initializes the connection. *
     * **************************************************************************
     */
    @Override
    protected void initialize() {        
        if (! isconnected) {
            setOnline(true);
            isconnected = true;
        } else {
            log.error("Tried to initialize undefined bias device!");

        }
    }

    /**
     ***************************************************************************
     **
     ** Opens connection to a device. *
     * **************************************************************************
     */
    @Command(name = "open", description = "Open connection to device")
    public String open(@Argument(name = "host", description = "Host name") String host
    ) throws DriverException {
        return "Simulated Keithley Device";
    }

    /**
     ***************************************************************************
     **
     ** Opens connection to a device. *
     * **************************************************************************
     */
    @Command(name = "open", description = "Open connection to device")
    public String open(@Argument(name = "host", description = "Host name") String host,
            @Argument(name = "port", description = "Port number") int port
    ) throws DriverException {
        /*
         n64.open(host, port);
         return n64.printdevid();*/
        return "Simulated Keithley Device";

    }

    /**
     ***************************************************************************
     **
     ** Opens FTDI connection to a device. *
     * **************************************************************************
     */
    @Command(name = "openftdi", description = "Open FTDI connection to device")
    public String openftdi(@Argument(name = "serialname", description = "Serial device name") String host,
            @Argument(name = "baud", description = "baud rate") int port
    ) throws DriverException {
        /*
         n64.openftdi(host, port);
         return n64.printdevid();
         */
        return "Simulated Keithley Device";

    }

    /**
     ***************************************************************************
     **
     ** accumBuffer: start the accumulation of readings into the devices
     * internal buffer
     *
     * **************************************************************************
     * @param nreads
     * @param nplc
     */
    @Command(name = "accumBuff", description = "start the buffer accumulation")
    public double accumBuffer(
            @Argument(name = "nreads", description = "#reads (armcount)") int nreads,
            @Argument(name = "nplc", description = "#Pulse Line Counts btwn reads (0.01->60.)") double nplc,
            @Argument(name = "wait", description = "to wait (true) or not to (false)") boolean wait) {
        this.nreads = nreads;
        this.nplc = nplc;
        if (wait) {
            try {
                Thread.sleep((long) (nplc / 60. * nreads * 1000));
            } catch (Exception e) {
                log.error("Sleep exception " + e);
            }
        }
        return (nplc / 60. * nreads);
    }

    /**
     ***************************************************************************
     **
     ** Read buffer *
     * **************************************************************************
     */
    @Command(name = "readBuffstr", description = "read the buffer")
    public String readBufferStr() throws DriverException {
//        return "Buffer = " + getString(n64.readBuffer());
        double data[][] = readBuffer();
        return "Data = " + getString(data[0]) + "\n"
                + " Times = " + getString(data[1]) + "\n";
//        return "Data = 1 2 3 2 1 \n Times = 0.1 0.2 0.3 0.4 0.5";

    }

    /**
     ***************************************************************************
     **
     ** Read buffer *
     * **************************************************************************
     */
    @Command(name = "readBuff", description = "read the buffer")
    public double[][] readBuffer() {
//        return "Buffer = " + getString(n64.readBuffer());
        double[][] vals = new double[2][nreads];

        for (int i = 0; i < nreads; i++) {
            vals[0][i] = nplc / 60. * i;
            vals[1][i] = 0.0002 * Math.random();
        }
        return vals;

    }

    /**
     ***************************************************************************
     **
     ** Shows the set voltage. *
     * **************************************************************************
     */
    @Command(name = "showVoltage", description = "Show the set voltage")
    public double getVoltage() throws DriverException {
        //return "Voltage = " + n64.getVoltage();
        return (-67.89);
    }

    /**
     ***************************************************************************
     **
     ** Reads the actual voltage. *
     * **************************************************************************
     */
    @Command(name = "readVoltage", description = "Read the actual voltage")
    public double readVoltage() {
        double volt = -67.88 + 1.0 * Math.random();
        return (volt);
    }

    /**
     ***************************************************************************
     **
     ** Shows the soft voltage limit. *
     * **************************************************************************
     */
    @Command(name = "showvoltlim", description = "Show the soft voltage limit")
    public String getVoltageLimit() throws DriverException {
//        return "Voltage limit = " + n64.getVoltageLimit();
        return "Voltage limit = -71.0";
    }

    /**
     ***************************************************************************
     **
     ** Shows the set current. *
     * **************************************************************************
     */
//    @Command(name = "showcurrent", description = "Show the set current")
    public double getCurrent() {
        double curr = 1.23E-12;
        /*      try {
         curr = n64.getCurrent();
         }
         catch (DriverException e) {
         log.error("Keithley driver failed to get the current!!");
         }
         */
        return (curr);
    }

    /**
     ***************************************************************************
     **
     ** Reads the actual current. *
     * **************************************************************************
     */
    @Command(type = CommandType.ACTION, name = "readCurrent", description = "Read the actual current")
    public double readCurrent() {
        log.debug("KeithleyDevice executing readcurrent operation:");

        double curr = 1.24E-12 + Math.random() * 2.E-13;
        return (curr);
    }

    /**
     ***************************************************************************
     **
     ** Shows the soft current limit. *
     * **************************************************************************
     */
    @Command(name = "showcurrlim", description = "Show the soft current limit")
    public String getCurrentLimit()  {
//        return "Current limit = " + n64.getCurrentLimit();
        return "Current limit = 0.001";
    }

    /**
     ***************************************************************************
     **
     ** Shows the output state. *
     * **************************************************************************
     */
    @Command(name = "showOutput", description = "Show the output state")
    public boolean showOutput() {
        boolean state = (1 == 0);
        return (state);
    }

}
