package org.lsst.ccs.subsystem.teststand;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.devices.power.distribution.APC7900Device;
import org.lsst.ccs.subsystem.common.devices.turbopump.TwisTorr84Device;
import org.lsst.ccs.subsystem.common.devices.vacuum.VQMDevice;
import org.lsst.ccs.subsystem.teststand.alerts.TS7Alerts;
import org.lsst.ccs.subsystem.teststand.data.TS7Outlets;
import org.lsst.ccs.subsystem.teststand.data.TestStandAgentProperties;
import org.lsst.ccs.subsystem.teststand.limits.TransitionAlgorithm;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Implementation of SLAC TS7
 *
 * @author The LSST CCS Team
 */
public class TS7 extends Subsystem implements HasLifecycle, AlertListener {

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    CryoDevice cryoDevc;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    VQMDevice vqmDevc;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    protected final Map<String, APC7900Device> pduDevicesMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    AP9630UPSDevice upsDevc;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService als;
    
    //The TurboPump device
    @LookupField(strategy = LookupField.Strategy.TREE)
    private TwisTorr84Device turbo;    

    private final Logger LOGGER = Logger.getLogger(getClass().getPackage().getName());

    private TransitionAlgorithm vacuumTransitionAlgorithm;
    private TransitionAlgorithm coldTransitionAlgorithm1;
    private TransitionAlgorithm coldTransitionAlgorithm2;
    private TransitionAlgorithm cryoTransitionAlgorithm;

    private volatile boolean alertResponseEnabled = false;
    private UPSMonitor upsMon;

    /**
     *  Constructor
     */
    public TS7() {
        super("ccfcs", AgentInfo.AgentType.WORKER);
    }

    /**
     * Initialization:
     * - register the raised Alerts
     */
    @Override
    public void init() {
        als.registerAlert(TS7Alerts.UPS_DISCONNECTED.getAlert());
        als.registerAlert(TS7Alerts.UPS_ON_BATTERY.getAlert());
        
        //Add Alert listener
        als.addListener(this);
    }
    
    /**
     * Post-initialization
     */
    @Override
    public void postInit() {
        if (cryoDevc == null) {
            //What do you want to do if cryo is not present?
            LOGGER.warning("Cryo device not defined");
        }

        
        // By setting TESTSTAND_TYPE_AGENT_PROPERTY we signal to consoles that this subsystem is compatible with the teststand subsystm GUI
        getAgentService(AgentPropertiesService.class).setAgentProperty(TestStandAgentProperties.TESTSTAND_TYPE_AGENT_PROPERTY, TS7.class.getCanonicalName());
        
        // Set up timer to handle transitions
        upsMon = new UPSMonitor(this, upsDevc, LOGGER, pts, als);
    }

    @Override
    public void build() {
        pts.scheduleAgentPeriodicTask(new AgentPeriodicTask("ups-timer", () -> {
            upsMon.monitorUPS();
        }).withPeriod(Duration.ofSeconds(1)));
    }

    
    //Listen to Alert and take action
    @Override
    public void onAlert(AlertEvent event) {
        
        Alert raisedAlert = event.getAlert();
        String alertId = raisedAlert.getAlertId();
        
        if ( alertId.equals(TS7Alerts.PRESSURE_TOO_HIGH.getAlert().getAlertId()) ) {
            if ( event.getLevel() == AlertState.ALARM ) {
                //Alarm triggered
                try { //Shut the valve, then fork off the rest of the response
                    turnOutletOff(TS7Outlets.VACUUMVALVE);
                    LOGGER.info("Shutting (Turning off) vacuum valve " + TS7Outlets.VACUUMVALVE.getOutletName());
                } catch (DriverException de) {
                    //How to handle the exception here?
                    //TO-DO Raise another alert?
                    LOGGER.severe("Failed to turn off VACUUMVALVE outlet -- manual intervention needed" + de);
                }                
                
                Thread t = new Thread(new VacuumAlarmResponse());
                t.start();                
            }
        } else if ( alertId.equals(TS7Alerts.PRESSURE_TOO_LOW.getAlert().getAlertId()) ) {
            //This is a vacuum related alarm, no action needed but alerts distributed.
            // The expectation is that the gauge may be having an error and needs attention
        } else if ( alertId.equals(TS7Alerts.COLD_PLATE_TEMPERATURE_TOO_LOW.getAlert().getAlertId()) ) {
            // This could be run-away cooling and potentially bad, but the time-scale
            // is a few hours.  So alert distribution at warning is first defense.
            // At alarm level, we'd better turn off the refrigerators
            if ( event.getLevel() == AlertState.ALARM ) {
                // Launch thread that turns off NF-55-1&2, PT-30 and REBs
                LOGGER.info("COLD_PLATE_TEMPERATURE_TOO_LOW: Turn off Polycold chillers...");
                Thread t = new Thread(new ColdPlateAlarmLowResponse());
                t.start();
            }            
        } else if ( alertId.equals(TS7Alerts.CRYO_PLATE_TEMPERATURE_TOO_LOW.getAlert().getAlertId()) ) {
            //  Response to turn off PT-30.  Cold plate can stay as is unless it triggers its own alert.
            if ( event.getLevel() == AlertState.ALARM ) {
                LOGGER.info("CRYO_PLATE_TEMPERATURE_TOO_LOW: Turn off Polycold chiller...");
                Thread t = new Thread(new CryoPlateAlarmLowResponse());
                t.start();                
            }
        } else if ( alertId.equals(TS7Alerts.COLD_PLATE_TEMPERATURE_TOO_HIGH.getAlert().getAlertId()) ) {
            // This could occur if the transition algorithm does not keep up which is
            // mostly benign but it could also occur in the event of a vacuum problem
            // but not much to do about that, if it wasn't caught in that section.
            if ( event.getLevel() == AlertState.ALARM ) {
                LOGGER.info("COLD_PLATE_TEMPERATURE_TOO_HIGH: Turning off heat sources");
                if (isAlertResponseEnabled()) {
                    try {
                        LOGGER.info("Turning off REB 48V power supply outlet: " + TS7Outlets.REB48VOLTPOWERSUPPLY.getOutletName());
                        turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY);
                    } catch (DriverException e) {
                        LOGGER.error("Error powering off REB power supply: " + e);
                    }
                } else {
                    LOGGER.info("Powering off REB power supply was disabled");
                }
                // Need to verify that OTP circuit is enabled on Cryocon and active
                // value to 0%.  (N.B. this needs reset on startup)
                LOGGER.info("COLD_PLATE_TEMPERATURE_TOO_HIGH: Alert response initiated");
            }
        } else if ( alertId.equals(TS7Alerts.CRYO_PLATE_TEMPERATURE_TOO_HIGH.getAlert().getAlertId()) ) {
            // This could occur if the transition algorithm does not keep up which is
            // mostly benign but it could also occur in the event of a vacuum problem
            // but not much to do about that, if it wasn't caught in that section.
            if ( event.getLevel() == AlertState.ALARM ) {
                LOGGER.info("CRYO_PLATE_TEMPERATURE_TOO_HIGH: Turning off heat sources");
                if (isAlertResponseEnabled()) {
                    try {
                        LOGGER.info("Turning off REB 48V power supply outlet: " + TS7Outlets.REB48VOLTPOWERSUPPLY.getOutletName());
                        turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY);
                    } catch (DriverException e) {
                        LOGGER.error("Error powering off REB power supply: " + e);
                    }
                } else {
                    LOGGER.info("Powering off REB power supply was disabled");
                }
                // Need to verify that OTP circuit is enabled on Cryocon and active
                // value to 0%.  (N.B. this needs reset on startup)
                LOGGER.info("CRYO_PLATE_TEMPERATURE_TOO_HIGH: Turn off REB PS if powerState is not OFF, verify Cryo OTP is enabled");
            }
        } else if ( alertId.equals(TS7Alerts.TURBO_PUMP_FAIL.getAlert().getAlertId()) ) {
            // ALARM:  Turbo pump in Failure state, a severe problem
            if ( event.getLevel() == AlertState.ALARM ) {
                String extraInfo;
                try {
                    extraInfo = "\n " + turbo.readTurboFailCode();
                }
                catch (DriverException e) {
                    extraInfo = "\n Exception reading Turbo error code: " + e;
                }                
                LOGGER.info(extraInfo);
            }
        } else if ( alertId.equals(TS7Alerts.TURBO_PUMP_SLOW.getAlert().getAlertId()) ) {
            // Turbo pump drops to status other than Normal or Fail
            if ( event.getLevel() == AlertState.ALARM ) {
                String extraInfo;
                try {
                    extraInfo = "\n New Turbo status = " + turbo.readTurboStatus();
                } catch (DriverException e) {
                    extraInfo = "\n Exception reading Turbo status: " + e;
                }
                LOGGER.info(extraInfo);
            }
        }
    }
    

    class VacuumAlarmResponse implements Runnable {
        @Override
        public void run() {
            try {
                LOGGER.info("Turning off cold plate refrigerator 1 " + TS7Outlets.COLDPLATEREFRIGERATOR_1.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_1);
                } else {
                    LOGGER.info("The action of turning off the cold plate refrigerator 1 was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off ColdPlateRefrigerator_1 outlet -- manual intervention needed");
            }
            sleep(2000); // TODO evaluate a better way for this
            try {
                LOGGER.info("Turning off cold plate refrigerator 2 " + TS7Outlets.COLDPLATEREFRIGERATOR_2.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_2);
                } else {
                    LOGGER.info("The action of turning off the cold plate refrigerator 2 was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off ColdPlateRefrigerator_2 outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOGGER.info("Turning off cryo plate refrigerator " + TS7Outlets.CRYOPLATEREFRIGERATOR.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.CRYOPLATEREFRIGERATOR);
                } else {
                    LOGGER.info("The action of turning off the cryo plate refrigerator was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off CryoPlateRefrigerator outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOGGER.info("Turning off turbo pump " + TS7Outlets.TURBOPUMP.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.TURBOPUMP); // no way to vent properly to slow down
                } else {
                    LOGGER.info("The action of turning off the turbo pump was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off Turbo Pump outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOGGER.info("Turning off roughing pump " + TS7Outlets.ROUGHINGPUMP.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.ROUGHINGPUMP); // Note that this also closes the second valve
                } else {
                    LOGGER.info("The action of turning off the roughing pump was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off Roughing Pump outlet -- manual intervention needed");
            }
            try {
                LOGGER.info("Turning off reb power supply " + TS7Outlets.REB48VOLTPOWERSUPPLY.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY); // TODO: Revise this later
                } else {
                    LOGGER.info("The action of turning off the reb power supply was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off REB 48V supply outlet -- manual intervention needed");
            }
        }
    }
    
    class ColdPlateAlarmLowResponse implements Runnable {
        @Override
        public void run() {
            try {
                LOGGER.info("Turning off cold plate refrigerator 1 " + TS7Outlets.COLDPLATEREFRIGERATOR_1.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_1);
                } else {
                    LOGGER.info("The action of turning off the cold plate refrigerator 1 was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off ColdPlateRefrigerator_1 outlet -- manual intervention needed");
            }
            sleep(2000); // TODO evaluate a better way for this
            try {
                LOGGER.info("Turning off cold plate refrigerator 2 " + TS7Outlets.COLDPLATEREFRIGERATOR_2.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_2);
                } else {
                    LOGGER.info("The action of turning off the cold plate refrigerator 2 was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off ColdPlateRefrigerator_2 outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOGGER.info("Turning off cryo plate refrigerator " + TS7Outlets.CRYOPLATEREFRIGERATOR.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.CRYOPLATEREFRIGERATOR);
                } else {
                    LOGGER.info("The action of turning off the cryo plate refrigerator was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off CryoPlateRefrigerator outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOGGER.info("Turning off reb power supply " + TS7Outlets.REB48VOLTPOWERSUPPLY.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY); // TODO: Revise this later
                } else {
                    LOGGER.info("The action of turning off the reb power supply was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off REB 48V supply outlet -- manual intervention needed");
            }
        }
    }
    
    class CryoPlateAlarmLowResponse implements Runnable {
        @Override
        public void run() {
            try {
                LOGGER.info("Turning off cryo plate refrigerator " + TS7Outlets.CRYOPLATEREFRIGERATOR.getOutletName());
                if (isAlertResponseEnabled()) {
                    turnOutletOff(TS7Outlets.CRYOPLATEREFRIGERATOR);
                } else {
                    LOGGER.info("The action of turning off the cryo plate refrigerator was disabled");
                }
            } catch (DriverException e) {
                LOGGER.severe("Failed to turn off CryoPlateRefrigerator outlet -- manual intervention needed" + e);
            }
            sleep(2000);
        }
    }
    
    private static void sleep(long msec) {
        try {
            Thread.sleep(msec);
        } catch (InterruptedException ie) {
            throw new RuntimeException("Could not sleep",ie);
        }
    }
    
    
    /**
     * Gets the list of PDU names.
     *
     * @return The list of names
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the list of PDU names")
    public List<String> getPduNames() {
        return new ArrayList<>(pduDevicesMap.keySet());
    }

    /**
     * Gets the list of PDU outlet names.
     *
     * @return The list of names (of the form "pdu/outlet")
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the list of PDU outlet names")
    public List<String> getOutletNames() {
        List names = new ArrayList<>();
        for (String pduName : pduDevicesMap.keySet()) {
            for (String outletName : pduDevicesMap.get(pduName).getOutletNames()) {
                names.add(pduName + "/" + outletName);
            }
        }
        return names;
    }

    /**
     * Gets the map of PDU outlet on states.
     *
     * @return The map of names (of the form "pdu/outlet") to on states
     * (true/false)
     * @throws DriverException
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the map of PDU outlet on states")
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException {
        Map states = new HashMap<>();
        for (String pduName : pduDevicesMap.keySet()) {
            Map<String, Boolean> pduStates = pduDevicesMap.get(pduName).getOutletOnStateMap();
            for (String outletName : pduStates.keySet()) {
                states.put(pduName + "/" + outletName, pduStates.get(outletName));
            }
        }
        return states;
    }

    @Command(type = Command.CommandType.ACTION, description = "Turn Off named outlet")
    public void turnOutletOff(TS7Outlets outlet) throws DriverException {
        changeOutletState(outlet, false);
    }

    @Command(type = Command.CommandType.ACTION, description = "Turn On named outlet")
    public void turnOutletOn(TS7Outlets outlet) throws DriverException {
        changeOutletState(outlet, true);
    }

    private void changeOutletState(TS7Outlets outlet, boolean on) throws DriverException {
        String outletName = outlet.getOutletName();
        for (APC7900Device pdu : pduDevicesMap.values()) {
            if (pdu.getOutletNames().contains(outletName)) {
                if (on) {
                    pdu.forceOutletOn(outletName);
                } else {
                    pdu.forceOutletOff(outletName);
                }
                return;
            }
        }
        throw new IllegalArgumentException("Could not find device to turn off outlet " + outletName);
    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable alert response")
    public void enableAlertResponse(boolean enable) {
        this.alertResponseEnabled = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isAlertResponseEnabled() {
        return alertResponseEnabled;
    }
        
}

