package org.lsst.ccs.subsystem.teststand;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.pfeiffer.TPG261;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 *
 * Device class for the Pfeiffer TPG261 driver
 *
 * @author homer
 */
public class TPG261Device extends Device implements VacuumGaugeDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("TEMP", Channel.TYPE_TEMP);
        typeMap.put("VOLTS", Channel.TYPE_VOLTS);
        typeMap.put("CURR", Channel.TYPE_POWER);
        typeMap.put("PRESSURE", Channel.TYPE_UNKNOWN);
        typeMap.put("WAVELENGTH", Channel.TYPE_SWITCH);
        typeMap.put("PA", Channel.TYPE_UNKNOWN);
    }

    private TPG261 microDev;

    TSConfig cfg = new TSConfig();

    private double[] runVac = new double[cfg.MAXSTATES]; // desired pressure for acquisitions

    TSState.vacstates vstate = TSState.vacstates.NOTCONFIGURED;
    
    boolean isConnected = false;
    boolean failedToInitialize = false;

    double lastPres = 0.0;
    
     /**
     ***************************************************************************
     **
     ** TPG261 Device constructor
     * **************************************************************************
     */
    public TPG261Device() {
    }

     /**
     ***************************************************************************
     **
     ** TPG261Device constructor with connection opening
     * @param itype
     * @param host
     * @param port
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public TPG261Device(int itype, String host, int port) throws DriverException {

        try {
            microDev = new TPG261();
            microDev.openSerial(host, port);
            vstate = TSState.vacstates.OFF;

        } catch (DriverException e) {
            System.out.println("Failed to open connection to TPG261 device!");
        }
    }

    /**
     ***************************************************************************
     **
     ** isVacuumGaugeOn - Is the TPG261 Device on? *
     * **************************************************************************
     */
    @Command(name = "isVacuumGaugeOn", description = "returns whether the TPG261 device is active")
    public boolean isVacuumGaugeOn() {
        boolean state = false;
        try {
            state = microDev.getStatus().contains("1");
            if (!vstate.equals(TSState.vacstates.TRIPPED)) {
                vstate = state ? TSState.vacstates.ON : TSState.vacstates.OFF;
            }
        } catch (DriverException e) {
            log.error("TPG261 device failed to close!!!");
        }
        return (state);
    }

    public void setGaugeOn() {
        log.info("TBD");
    }
    
    public void setGaugeOff() {
        log.info("TBD");
    }
    
    /**
     ***************************************************************************
     **
     ** returns the pressure read from the TPG261 device *
     * **************************************************************************
     */
    @Command(name = "readPressure", description = "returns the pressure read from the TPG261 device")
    public double readPressure() {
        double pressure = 0.0;

        try {
            if (isConnected) {
                pressure = microDev.getPressure();
                lastPres = pressure;
            }
        } catch (DriverException e) {
            log.error("TPG261 device failed to return pressure reading!!!");
        }
        return (pressure);
    }

    @Command(name = "getLastPressure", description = "returns the last successful pressure read from the TPG261 device")
    public double getLastPres() {
        return lastPres;
    }

    
    
    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (microDev != null) {
                microDev.close();
            }
        } catch (DriverException e) {
            log.error("TPG261 device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the connection. *
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        try {
            log.error("State of of the TPG261 device is"
                    + (microDev.getStatus().contains("1") ? "ON" : "OFF"));
            log.error("TPG261 device initialized.");
            isConnected = true;
            vstate = TSState.vacstates.OK;
            setOnline(true);
        } catch (DriverException e) {
            if (!failedToInitialize) log.error("TPG261 device failed to initialize!!!");
            failedToInitialize = true ;
        }
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_UNKNOWN) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        log.debug("TPG261Device readChannel called! chan=" + chan + " type=" + type);
        try {
            value = readPressure();
        } catch (Exception e) {
            log.error("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the vac value for acquisition *
     * **************************************************************************
     */
    @Command(name = "setrunvac", description = "Sets the vac value for acquisition")
    public void setRunVac(double runVac, int cfgstate) {
        this.runVac[cfgstate] = runVac;
        return;
    }

    /**
     ***************************************************************************
     **
     ** Returns the vac value for acquisition *
     * **************************************************************************
     */
    @Command(name = "getrunvac", description = "Returns the vac value for acquisition")
    public double getRunVac(int cfgstate) {
        return (runVac[cfgstate]);
    }

    @Command(name = "setState", description = "set vacuum device status")
    public void setState(int istate) {
        vstate = TSState.vacstates.values()[istate];
    }

    @Command(name = "getState", description = "get vacuum device status")
    public int getState() {
        return (vstate.ordinal());
    }

}
