package org.lsst.ccs.subsystem.teststand;

import java.util.Map;
import java.util.HashMap;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.apcpdu.AP7900;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 ***************************************************************************
 **
 * AP7900 device class for the teststand subsystem
 *
 * * @author: Homer Neal
 * **************************************************************************
 */
public class AP7900Device extends Device implements PDUDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();
    private final Thread readW = new Thread(new Reader());

    private int last_state = -1; // last channel state seen
    private int nRead = 0, nByte = 0, nTimeout = 0;
    private boolean open = false;                // are we logged in and able to talk
    private boolean isconnected = false; // is the device defined

    TSConfig cfg = new TSConfig();

    AP7900 devpdu = null;

    String pdu_host = null;

    int pdu_port = 23;

    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;

    boolean failedToInitialize = false;
    private int major_version;

    /**
     ***************************************************************************
     **
     ** Inner class to implement device reading thread. *
     * **************************************************************************
     */
    private class Reader implements Runnable {

        @Override
        public void run() {
            System.out.println("Reader thread started.");
            while (true) {
                if (open) {
                    try {
                        String reply = devpdu.read();
                        if (reply.indexOf("State") >= 0) {
                            last_state = 0;
                            if (reply.indexOf("ON") >= 0) {
                                last_state = 1;
                            }
                        }
                        System.out.println(reply);
                        nRead++;
                        nByte += reply.length();
                    } catch (DriverTimeoutException e) {
                        nTimeout++;
                    } catch (DriverException e) {
                        if (open) {
                            System.out.println(e);
                            System.exit(0);
                        }
                    }
                } else {
                    try {
                        Thread.sleep(100);
                    } catch (InterruptedException ex) {
                        System.out.println("Sleep interrupted:"+ex);
                    }
                }
               
            }
//            System.out.println("Reader thread ended.");
        }
    }

    /**
     ***************************************************************************
     **
     ** AP7900Device constructor - put AP7900 in initial subsystem state
     *
     ** @param host device address
     ** @param port port or baud * * @param runBias CCD bias voltage while
     * running
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public AP7900Device(String host, int port)
            throws DriverException {
        isconnected = true;
        pdu_host = host;
        pdu_port = port;

        try {
            devpdu = new AP7900();

        } catch (DriverException f) {
            System.out.println("Failed to open connection to AP7900 device!");
            isconnected = false;
        }
        readW.setDaemon(true);
        readW.start();

    }

    /**
     ***************************************************************************
     **
     ** Constructor *
     * **************************************************************************
     */
    public AP7900Device() {
//        isconnected = true;
        try {
            devpdu = new AP7900();
        } catch (DriverException f) {
            System.out.println("Failed to create to AP7900 device instance!");
        }

    }

    /**
     ***************************************************************************
     **
     ** Performs configuration.
     *
     *
     * **************************************************************************
     */
    @Override
    protected void initDevice() {
        fullName = "AP7900 module";
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (devpdu != null) {
                devpdu.close();
            }
            kstate = TSState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("AP7900 device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the device
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        if (isconnected) {
            log.debug("AP7900 device is already initialized");
            /*
             try {
             log.debug(devpdu.getIdent());
             } catch (Exception e) {
             if (!failedToInitialize) {
             log.error("AP7900 device failed to respond to ident request! The initialization has FAILED.");
             failedToInitialize = true;
             }
             isconnected = false;
             }
             */
            log.debug("Setting device online.");
            kstate = TSState.pwrstates.OK;
            setOnline(true);
        } else {
            if (!failedToInitialize) {
                log.error("Tried to initialize unconnected PDU device!");
                failedToInitialize = true;
            }

        }
    }

    /**
     ***************************************************************************
     **
     ** set an outlet state *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.ACTION, name = "setOutletState", description = "set the state of an outlet")
    public void setOutletState(@Argument(name = "outlet", description = "outlet number") int outlet,
            @Argument(name = "otlstate", description = "on(true)/off(false) state") boolean otlstate) {
        if (devpdu == null) {
            return; // not even defined perhaps
        }
        try {
            devpdu.opennet(pdu_host, pdu_port,major_version); // open connection
            open = true;
            devpdu.setTimeout(60.0);
        } catch (DriverException f) {
            System.out.println("Failed to open connection to AP7900 device!");
            isconnected = false;
        }
        try {
            if (open) {
                devpdu.setOutletState(outlet, otlstate);
            }
        } catch (DriverException f) {
            log.error("Failed open/close the AP7900");
        }
        try {
            open = false;
            Thread.sleep(500); // give the other thread a chance to stop before closing the connection
            devpdu.close(); // close connection
        } catch (DriverException f) {
            System.out.println("Failed to close connection to AP7900 device!");
        } catch (InterruptedException e) {
            System.out.println("Sleep interrupted" + e);
        }

    }

    /**
     ***************************************************************************
     **
     ** get an outlet state *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.ACTION, name = "getOutletState", description = "set the state of an outlet")
    public boolean getOutletState(@Argument(name = "outlet", description = "outlet number") int outlet) {
        boolean otlstate = false;
        if (devpdu == null) {
            return (false); // not even defined perhaps
        }
        try {
            devpdu.opennet(pdu_host, pdu_port, major_version); // open connection
            open = true;
            devpdu.setTimeout(60.0);
        } catch (DriverException f) {
            System.out.println("Failed to open connection to AP7900 device!");
            open = false;
        }
        try {
            if (open) {
                devpdu.isOutletOn(outlet);
                last_state = -1; // reset
                int countdown = 100;
                while (countdown-- > 0 && last_state < 0) {
                    System.out.println("Waiting for a response from the PDU");
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException ex) {
                        System.out.println("Sleep attempt interrupted" + ex);
                    }
                }
                if (last_state < 0) {
                    throw new DriverException("PDU never responded to outlet state request");
                }

                otlstate = last_state == 1;
            }
        } catch (DriverException f) {
            log.error("Failed open/close the AP7900");
        }
        try {
            open = false;
            Thread.sleep(500); // give the other thread a chance to stop before closing the connection
            devpdu.close(); // close connection
        } catch (DriverException f) {
            System.out.println("Failed to close connection to AP7900 device!");
        } catch (InterruptedException e) {
            System.out.println("Sleep interrupted" + e);
        }

        return (otlstate);
    }

    /**
     ***************************************************************************
     **
     ** Converts a boolean to on/off. *
     * **************************************************************************
     */
    private String getOnOff(boolean on) {
        return on ? "on" : "off";
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_SWITCH) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
            if (type == Channel.TYPE_SWITCH) {
                log.debug("setting channel online - chan=" + chan);
            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        log.debug("AP7900Device readChannel called! chan=" + chan + " type=" + type);
        try {
            if (type == Channel.TYPE_SWITCH) {
                value = 1.0; // to be replaced with a byte indicating the outlet states
            }
        } catch (Exception e) {
            log.debug("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    @Command(name = "setstate", description = "set AP7900 device status")
    public void setState(int istate) {
        kstate = TSState.pwrstates.values()[istate];
    }

    @Command(name = "getstate", description = "get AP7900 device status")
    public int getState() {
        return (kstate.ordinal());
    }

}
