package org.lsst.ccs.subsystem.teststand;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.netbotz.NetbotzDriver;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;

/**
 *
 * Device class for the Netbotz driver
 *
 * @author homer
 */
public class NetBotzDevice extends Device implements EnviroDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("TEMP", Channel.TYPE_TEMP);
        typeMap.put("HUMIDITY", Channel.TYPE_VOLTS);
        typeMap.put("DEW", Channel.TYPE_POWER);
        typeMap.put("PARTCNT", Channel.TYPE_UNKNOWN);
        typeMap.put("UNKNOWN", Channel.TYPE_UNKNOWN);
    }

    private NetbotzDriver apcnb;

    private String envirofile = null; // Environment file path

    public NetBotzDevice(String ef) {
//        String ef = System.getProperty("lsst.ccs.teststand.envirofile", "");
        apcnb = new NetbotzDriver();
//        System.out.println("Opening connection to the NetBotz device!. File is " + ef);
        this.open(ef);

    }

    @Command(name = "open", description = "Open connection for NetBotz readings retrieval")
    public void open(@Argument(name = "envirofile", description = "environment file") String ef
    ) //throws DriverException
    {
        envirofile = ef;
        System.out.println("Calling netbotz driver to open connection to file - " + envirofile);
        try {
            apcnb.open(envirofile);
        } catch (DriverException ex) {
            System.out.println("Failed to open environment datafile!"+ ex);
        }
    }

    /**
     ***************************************************************************
     **
     ** Get humidity *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getHumidity", description = "read humidity reading from the NetBotz environment log")
    public Double getHumidity() {
        return apcnb.getHumidity();
    }

    /**
     ***************************************************************************
     **
     ** Get temperature *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getTemperature", description = "read temperature reading from the NetBotz environment log")
    public Double getTemperature() {
        Double temp = apcnb.getTemperature();
        log.debug("getTemp returns - " + temp);
        return temp;
    }

    /**
     ***************************************************************************
     **
     ** Get dew point *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getDew", description = "read dew point reading from the NetBotz environment log")
    public Double getDew() {
        return apcnb.getDew();
    }

    /**
     ***************************************************************************
     **
     ** Get particle count *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getpc", description = "read particle count reading from the NetBotz environment log")
    public Double getPartCnt() {
        return apcnb.getPartCnt();
    }

    /**
     ***************************************************************************
     **
     ** Get environment values *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getEnviro", description = "read line of readings from the NetBotz environment log")
    public void getEnviro() {

        try {
            apcnb.GetEnviro();
        } catch (IOException e) {
            System.out.println("Failed to retrieve environment readings " + e);
        }

        /*
         return 
         "Humidity = " + Double.toString(apcnb.getHumidity()) +
         ", Temperature = " + Double.toString(apcnb.getTemperature()) +
         ", Dew = " + Double.toString(apcnb.getDew()) +
         ", Particle Count = " + Double.toString(apcnb.getPartCnt())
         ;
         */
    }

    /**
     * Get environment variables
     * 
     * @param Humidity
     * @param Temperature
     * @param Dew
     * @param PartCnt
     */
    @Command(type = Command.CommandType.QUERY, name = "getEnviro", description = "read line of readings from the NetBotz environment log and return set arg values")
    public void getEnviro(@Argument(name = "humidity", description = "environment file") Double Humidity,
            @Argument(name = "temperature", description = "") Double Temperature,
            @Argument(name = "dewpoint", description = "") Double Dew,
            @Argument(name = "particlecount", description = "") Double PartCnt) {

        try {
            apcnb.GetEnviro(Humidity, Temperature, Dew, PartCnt);
        } catch (IOException e) {
            System.out.println("Failed to retrieve environment readings " + e);
        }
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        apcnb.close();
    }

    /**
     ***************************************************************************
     **
     ** Initializes the connection. *
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        log.debug("NetBotz device initialized.");
        setOnline(true);
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
//        log.debug("NetBotzDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            apcnb.GetEnviro(); // must retrieve readings first
            switch (chan) {
                case 0:
                    value = apcnb.getHumidity();
                    break;
                case 1:
                    value = apcnb.getTemperature();
//                    log.debug("Room temperature value = "+value);
                    break;
                case 2:
                    value = apcnb.getDew();
                    break;
                case 3:
                    value = apcnb.getPartCnt();
                    break;
            }
        } catch (Exception e) {
            log.error("Error reading channel chan " + chan + " type " + type + ": " + e);
        }
        return (value);
    }

}
