package org.lsst.ccs.subsystem.teststand;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.microion.MicroIon;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 *
 * Device class for the MicroIon driver
 *
 * @author homer
 */
public class MicroIonDevice extends Device implements VacuumGaugeDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("TEMP", Channel.TYPE_TEMP);
        typeMap.put("VOLTS", Channel.TYPE_VOLTS);
        typeMap.put("CURR", Channel.TYPE_POWER);
        typeMap.put("PRESSURE", Channel.TYPE_UNKNOWN);
        typeMap.put("WAVELENGTH", Channel.TYPE_SWITCH);
        typeMap.put("PA", Channel.TYPE_UNKNOWN);
    }

    private MicroIon microDev;

    TSConfig cfg = new TSConfig();

    private double[] runVac = new double[cfg.MAXSTATES]; // desired pressure for acquisitions

    TSState.vacstates vstate = TSState.vacstates.NOTCONFIGURED;

    boolean isConnected = false;
    boolean failedToInitialize = false;

    double lastPres = 0.0;
    
    /**
     ***************************************************************************
     **
     ** MicroIonDevice constructor
     * **************************************************************************
     */
    public MicroIonDevice() {
    }

    /**
     ***************************************************************************
     **
     ** MicroIonDevice constructor with connection opening
     *
     * @param itype
     * @param host
     * @param port
     * @param addr
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public MicroIonDevice(int itype, String host, int port, int addr) throws DriverException {

        try {
            microDev = new MicroIon();
            microDev.openSerial(host, port, addr);
            vstate = TSState.vacstates.OFF;
            isConnected = true;
        } catch (DriverException e) {
            System.out.println("Failed to open connection to MicroIon device!");
        }
    }

    /**
     ***************************************************************************
     **
     ** isVacuumGaugeOn - Is the MicroIon Device on? *
     * **************************************************************************
     */
    @Command(name = "isVacuumGaugeOn", description = "returns whether the MicroIon device is active")
    public boolean isVacuumGaugeOn() {
        boolean state = false;
        try {
            state = microDev.getState() == MicroIon.onOff.ON;
            if (!vstate.equals(TSState.vacstates.TRIPPED)) {
                vstate = state ? TSState.vacstates.ON : TSState.vacstates.OFF;
            }
        } catch (DriverException e) {
            log.error("MicroIon device failed to respond to status request!!!");
        }
        return (state);
    }

    
    @Command(name = "setGaugeOn", description = "turn on power to the vacuum gauge filament")
    public void setGaugeOn() {
        try {
            if (microDev!=null) microDev.setState(1);
        } catch (DriverException e) {
            log.error("MicroIon device failed to turn on!!!");
        }
    }
    
    @Command(name = "setGaugeOff", description = "turn off power to the vacuum gauge filament")
    public void setGaugeOff() {
        try {
            if (microDev!=null) microDev.setState(0);
        } catch (DriverException e) {
            log.error("MicroIon device failed to turn off!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** returns the pressure read from the MicroIon device *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "readPressure", description = "returns the pressure read from the MicroIon device")
    public double readPressure() {
        double pressure = 0.0;

        try {
            if (isConnected) {
                pressure = microDev.getPressure();
                if (pressure>0. && pressure<2000.) { // make sure it is sensible
                    lastPres = pressure;
                } else {
                    pressure = 0. ;
                }
            }
        } catch (DriverException e) {
            log.debug("MicroIon device failed to return pressure reading!!!");
            System.out.print("P"); // a little indicator that the pressure reading didn't work
            pressure = lastPres;
        }
        return (pressure);
    }

    @Command(name = "getLastPressure", description = "returns the last successful pressure read from the TPG261 device")
    public double getLastPres() {
        return lastPres;
    }

    
    
    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (microDev != null) {
                microDev.close();
            }
        } catch (DriverException e) {
            log.error("MicroIon device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the connection. *
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        if (isConnected) {
            try {
                log.debug("State of of the MicroIon device is "
                        + (microDev.getState() == MicroIon.onOff.ON ? "ON" : "OFF"));
                log.debug("MicroIon device initialized.");
                vstate = TSState.vacstates.OK;
                setOnline(true);
            } catch (DriverException e) {
                if (!failedToInitialize) {
                    log.error("MicroIon device failed to initialize!!!");
                }
                failedToInitialize = true;
                isConnected = false;
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_UNKNOWN) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = Double.NaN;
//        log.debug("MicroIonDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            value = readPressure();
        } catch (Exception e) {
            log.debug("Error reading channel type " + type + ": " + e);
            System.out.print("P");
        }
        return (value);
    }

    /**
     ***************************************************************************
     **
     ** Sets the vac value for acquisition *
     * **************************************************************************
     */
    @Command(name = "setRunVac", description = "Sets the vac value for acquisition")
    public void setRunVac(double runVac, int cfgstate) {
        this.runVac[cfgstate] = runVac;
        return;
    }

    /**
     ***************************************************************************
     **
     ** Returns the vac value for acquisition *
     * **************************************************************************
     */
    @Command(name = "getRunVac", description = "Returns the vac value for acquisition")
    public double getRunVac(int cfgstate) {
        return (runVac[cfgstate]);
    }

    @Command(name = "setstate", description = "set vacuum device status")
    public void setState(int istate) {
        vstate = TSState.vacstates.values()[istate];
    }

    @Command(name = "getState", description = "get vacuum device status")
    public int getState() {
        return (vstate.ordinal());
    }

}
