package org.lsst.ccs.subsystem.teststand;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverLockedException;
import org.lsst.ccs.drivers.cornerstone.C260;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;

/**
 ***************************************************************************
 **
 ** Provides access to the Cornerstone260 methods for the teststand subsystem
 *
 *
 * * @author Homer Neal *
 * **************************************************************************
 */
public class Cornerstone260Device extends Device implements MonochromatorDevice {

    protected C260 md = null;
//    protected MonochromatorDriver md;

    private double runWave = 0.0; // wl for acquisitions

    @ConfigurationParameter(maxLength = 6, description = "filter labels")
    private volatile String[] filter_label = {"", "", "", "", "", ""};

    @ConfigurationParameter(maxLength = 6, description = "filter lower edges", 
                            units = "nm")
    private volatile double[] filter_edges = new double[6];

    @ConfigurationParameter(isFinal = true, 
                            description = "serial devcName or ftdi serialNo")
    private volatile String host = null;

    @ConfigurationParameter(isFinal = true, 
                            description = "connection type, serial or ftdi")
    private volatile String connType = "ftdi";

    @ConfigurationParameter(isFinal = true, description = "baud rate")
    private volatile int baud = 9600;

    private int waitForLock = NOT_DEFINED;

    private static final int NOT_DEFINED = -123456789;
    
    private static final Map<String, Integer> typeMap = new HashMap<>();

    long prevTime[] = new long[10]; // previous time (chan) the data was acquired
    
    private double lastwl, lastband;
    private int lastfltpos, lastshut, lastslit1, lastslit2, lastgrat,
            laststep;
    
    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    static {
        typeMap.put("WAVELENGTH", Channel.TYPE_UNKNOWN);
        typeMap.put("PA", Channel.TYPE_UNKNOWN);
        typeMap.put("UNKNOWN", Channel.TYPE_UNKNOWN);
    }


    /**
     ***************************************************************************
     **
     ** Opens connection to a device. *
     * **************************************************************************
     */
    @Command(name = "open", description = "Open connection to device")
    public void open(@Argument(name = "host", description = "Host name") String host
    ) throws DriverException {
        md.open(host);
        log.debug("Opened connection to Cornerstone 260 device.");
        return;
    }

    /**
     ***************************************************************************
     **
     ** Closes connection to a device. *
     * **************************************************************************
     */
//    @Command(name="close", description="Close connection to device")
    @Override
    public void close() {
        if (md != null) {
            try {
                md.close();
            } catch (DriverException e) {
                log.error("Failed to close connection to Cornerstone device.");
            }
        }
        log.debug("Closed connection to Cornerstone 260 device.");
        return;
    }

    /**
     ***************************************************************************
     **
     ** Initializes the connection. *
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        try {
            log.info("Try to open connection to Cornerstone260!");
            log.info("Using host = (" + host + ") and baud = (" + baud +") connection type: "+connType);

            md = waitForLock != NOT_DEFINED ? new C260(waitForLock) : new C260();
            if ( connType.equals("serial") ) {
                md.open(host, baud);
            } else {
                md.openftdi(host, baud);
            }
            log.info("Succeeded in opening connection to C260 monochromator.");
            setOnline(true);
        } catch (DriverException f) {
            log.severe("Failed to open connection to Cornerstone260 device!",f);
            md = null;
        }
    }

//[homer@teststand-1 kbin]$ egrep "import|class|public" monotxt | awk '/public/{print "    /**\n    ***************************************************************************\n    **\n    ** "substr($3,1,index($3,"(")-1)":\n    **\n    ***************************************************************************\n    **/\n    @Command(name=\""substr($3,1,index($3,"(")-1)"\", description=\"\")\n"$0"\n    {\n    }\n\n"}' | more
    /**
     ***************************************************************************
     **
     ** abort: *
     * **************************************************************************
     *
     */
    @Command(name = "abort", description = "")
    public void abort() throws DriverException {
        if (md != null) {
            md.abort();
        }
    }

    /**
     ***************************************************************************
     **
     ** getBandpass: *
     * **************************************************************************
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(type = Command.CommandType.QUERY, name = "getBandpass", description = "")
    public double getBandpass() throws DriverException {
        if (md != null) {
            double gbp = md.getBandpass();
            return gbp;
        }
        return -999.;
    }

    /**
     ***************************************************************************
     **
     ** setBandpass: *
     * **************************************************************************
     *
     */
    @Command(name = "setBandpass", description = "")
    public void setBandpass(double bandpass) throws DriverException {
        if (md != null) {
            md.setBandpass(bandpass);
        }
    }

    /**
     ***************************************************************************
     **
     ** setCalibrate: *
     * **************************************************************************
     *
     */
    @Command(name = "setCalibrate", description = "")
    public void setCalibrate(double val) throws DriverException {
        if (md != null) {

        }
    }

    /**
     ***************************************************************************
     **
     ** getError: *
     * **************************************************************************
     *
     */
    @Command(type = CommandType.QUERY, name = "getError", description = "")
    public String getError() throws DriverException {
        C260.storedError err = md.new storedError();
        int errCode = md.getFullError(err);
        // Once we query the error status, we probably should erase the
        // current list of errors:
        md.clearErrors();
        if (errCode == -1) // No Error
        {
            return "No Error";
        } else {
            return err.getStr();
        }
    }

    /**
     ***************************************************************************
     **
     ** getFilter: *
     * **************************************************************************
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(type = Command.CommandType.QUERY, name = "getFilter", description = "")
    public int getFilter() throws DriverException {
        if (md != null) {
            int fln = md.getFilter();
            return fln;
        }
        return -1;
    }

    /**
     ***************************************************************************
     **
     ** setFilter: *
     * **************************************************************************
     *
     */
    @Command(name = "setFilter", description = "")
    public void setFilter(int pos) throws DriverException {
        if (isOnline()) {
            if (md != null) {
                md.setFilter(pos);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** getFilterLabel: *
     * **************************************************************************
     *
     * @param filter
     * @return
     */
    @Command(type = CommandType.QUERY, name = "getFilterLabel", description = "")
    public String getFilterLabel(int filter) {
        String label = null;
        /*
         try {
         if (md != null) {
         label = md.getFilterLabel(filter);
         } else {
         */
        label = filter_label[filter - 1];
        /*                log.debug(filter + ": Retrieved filter label from local cache since device is offline. label = " + filter_label);
         }
         } catch (DriverLockedException ex) {
         System.out.print("F"); // just to indicate that we didn't get it because the driver is currently locked
         } catch (DriverException ex) {
         log.error("C260 failed to get filter label!");
         }
         */
        return (label);
    }

    /** This is needed for Monochromator interace, but not a Command;
     *  and using it in setFilteLabels would not be thread-safe).
     */
    public void setFilterLabel(int filter, String label) {
        filter_label[filter] = label;
    }

    /**
     ***************************************************************************
     **
     ** setFilterLabels: 
     ***************************************************************************
     *
     */
    @ConfigurationParameterChanger(propertyName = "filter_labels")
    public void setFilterLabels(String[] labels) {
        if (labels == null || labels.length != filter_label.length) {
            throw new IllegalArgumentException("Invalid labels array, should have 6 elements");
        } else {
            filter_label = labels;
        }
    }

    /**
     *************************************************************************
     **
     ** setFilterEdges
     * ***********************************************************************
     *
     */
    @ConfigurationParameterChanger(propertyName = "filter_edges")
    public void setFilterEdges(double[] edges) {
        if (edges == null || edges.length != filter_edges.length) {
            throw new IllegalArgumentException("Invalid edges array, should have 6 elements");
        } else {
            filter_edges = edges;
        }
    }

    /**
     ***************************************************************************
     **
     ** getGrating: *
     * **************************************************************************
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(type = Command.CommandType.QUERY, name = "getGrating", description = "")
    public String getGrating() throws DriverException {
        if (md != null) {
            return md.getGrating();
        }
        return null;
    }

    /**
     ***************************************************************************
     **
     ** setGrating: *
     * **************************************************************************
     *
     */
    @Command(name = "setGrating", description = "")
    public void setGrating(int param) throws DriverException {
        if (md != null) {
            md.setGrating(param);
        }
        return;
    }

    /**
     ***************************************************************************
     **
     ** getGratingLabel: *
     * **************************************************************************
     *
     */
    @Command(name = "getGratingLabel", description = "")
    public String getGratingLabel(int grat) throws DriverException {
        if (md != null) {
            return md.getGratingLabel(grat);
        }
        return null;
    }

    /**
     ***************************************************************************
     **
     ** setGratingLabel: *
     * **************************************************************************
     *
     */
    @Command(name = "setGratingLabel", description = "")
    public void setGratingLabel(int param, String label) throws DriverException {
        if (md != null) {
            md.setGratingLabel(param, label);
        }
    }

    /**
     ***************************************************************************
     **
     ** getGratingLines: *
     * **************************************************************************
     *
     */
    @Command(name = "getGratingLines", description = "")
    public int getGratingLines(int grat) throws DriverException {
        if (md != null) {
            return getGratingLines(grat);
        }
        return (0);
    }

    /**
     ***************************************************************************
     **
     ** setGratingLines: *
     * **************************************************************************
     *
     */
    @Command(name = "setGratingLines", description = "")
    public void setGratingLines(int grat, int param) throws DriverException {
    }

    /**
     ***************************************************************************
     **
     ** getGratingFactor: *
     * **************************************************************************
     *
     */
    @Command(name = "getGratingFactor", description = "")
    public double getGratingFactor(int grat) throws DriverException {
        if (md != null) {
            return md.getGratingFactor(grat);
        }
        return (0.0);
    }

    /**
     ***************************************************************************
     **
     ** setGratingFactor: *
     * **************************************************************************
     *
     */
    @Command(name = "setGratingFactor", description = "")
    public void setGratingFactor(int grat, double param) throws DriverException {
    }

    /**
     ***************************************************************************
     **
     ** getGratingOffset: *
     * **************************************************************************
     *
     */
    @Command(name = "getGratingOffset", description = "")
    public double getGratingOffset(int grat) throws DriverException {
        return (0.0);
    }

    /**
     ***************************************************************************
     **
     ** setGratingOffset: *
     * **************************************************************************
     *
     */
    @Command(name = "setGratingOffset", description = "")
    public void setGratingOffset(int grat, double param) throws DriverException {
    }

    /**
     ***************************************************************************
     **
     ** getHandshake: *
     * **************************************************************************
     *
     */
    @Command(name = "getHandshake", description = "")
    public String getHandshake() throws DriverException {
        if (md != null) {
            boolean flag = md.getHandshake();
            if (flag == true) {
                return "true";
            } else {
                return "false";
            }
        }
        return null;
    }

    /**
     ***************************************************************************
     **
     ** setHandshake: *
     * **************************************************************************
     *
     */
    @Command(name = "setHandshake", description = "")
    public void setHandshake(int flag) throws DriverException {
        if (md != null) {
            md.setHandshake(flag);
        }
    }

    /**
     ***************************************************************************
     **
     ** getInfo: returns information about the Monochromator
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    @Command(name = "getInfo", description = "return information about the C260")
    public String getInfo() throws DriverException {
        String info = "";
        if (md != null) {
            info = md.getInfo();
        }
        return info;
    }

    /**
     ***************************************************************************
     **
     ** setOutport: *
     * **************************************************************************
     *
     */
    @Command(name = "setOutport", description = "")
    public void setOutport(int num) throws DriverException {
    }

    /**
     ***************************************************************************
     **
     ** getShutter: *
     * **************************************************************************
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(type = Command.CommandType.QUERY, name = "getShutter", description = "")
    public String getShutter() throws DriverException {
        if (md != null) {
            return md.getShutter();
        }
        return null;
    }

    /**
     ***************************************************************************
     **
     ** openShutter: *
     * **************************************************************************
     *
     */
    @Command(name = "openShutter", description = "")
    public void openShutter() throws DriverException {
        if (md != null) {
            md.openShutter();
        }
    }

    /**
     ***************************************************************************
     **
     ** closeShutter: *
     * **************************************************************************
     *
     */
    @Command(name = "closeShutter", description = "")
    public void closeShutter() throws DriverException {
        if (isOnline()) {
            if (md != null) {
                md.closeShutter();
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** isShutterOpen: *
     * **************************************************************************
     *
     */
    @Command(name = "isShutterOpen", description = "")
    public boolean isShutterOpen() throws DriverException {
        boolean is = false;
        if (isOnline()) {
            is = md.isShutterOpen();
        }
        return (is);
    }

    /**
     ***************************************************************************
     **
     ** setShutter: *
     * **************************************************************************
     *
     */
    @Command(name = "setShutter", description = "")
    public void setShutter(String param) throws DriverException {
        if (isOnline()) {
            if (md != null) {
                md.setShutter(param);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** setSlitSize: *
     * **************************************************************************
     *
     */
    @Command(name = "setSlitSize", description = "")
    public void setSlitSize(int slit, int microns) throws DriverException {
        if (md != null) {
            md.setSlitSize(slit, microns);
        }
    }

    /**
     ***************************************************************************
     **
     ** getSlitSize: *
     * **************************************************************************
     *
     */
    @Command(type = Command.CommandType.QUERY, name = "getSlitSize", description = "")
    public int getSlitSize(int slit) throws DriverException {
        int slitsize = 0;
        if (md != null) {
            slitsize = md.getSlitSize(slit);
        }
        return (slitsize);
    }

    public double getLastwl() {
        return lastwl;
    }

    public double getLastband() {
        return lastband;
    }

    public int getLastfltpos() {
        return lastfltpos;
    }

    public int getLastshut() {
        return lastshut;
    }

    public int getLastSW1() {
        return lastslit1;
    }

    public int getLastSW2() {
        return lastslit2;
    }

    public int getLastgrat() {
        return lastgrat;
    }

    public int getLaststep() {
        return laststep;
    }

    /**
     ***************************************************************************
     **
     ** getErrorCondition: * * -1 denotes No Error
     * **************************************************************************
     *
     * @return
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(type = Command.CommandType.QUERY, name = "getErrorCondition", description = "")
    public int getErrorCondition() throws DriverException {
        C260.storedError err = md.new storedError();
        int errNum = md.getFullError(err);
        return (errNum);
    }

    /**
     ***************************************************************************
     **
     ** getStep: *
     * **************************************************************************
     *
     */
    @Command(name = "getStep", description = "")
    public int getStep() throws DriverException {
        if (md != null) {
            return md.getStep();
        }
        return (0);
    }

    /**
     ***************************************************************************
     **
     ** advanceSteps: *
     * **************************************************************************
     *
     */
    @Command(name = "advanceSteps", description = "")
    public void advanceSteps(int param) throws DriverException {
        if (md != null) {
            md.advanceSteps(param);
        }
    }

    /**
     ***************************************************************************
     **
     ** getUnits: *
     * **************************************************************************
     *
     */
    @Command(name = "getUnits", description = "")
    public String getUnits() throws DriverException {
        if (md != null) {
            return md.getUnits();
        }
        return null;
    }

    /**
     ***************************************************************************
     **
     ** setUnits: *
     * **************************************************************************
     *
     */
    @Command(name = "setUnits", description = "")
    public void setUnits(String param) throws DriverException {
        if (md != null) {
            md.setUnits(param);
        }
    }

    /**
     ***************************************************************************
     **
     ** getWave: *
     * **************************************************************************
     *
     */
    @Command(type = Command.CommandType.QUERY, name = "getWave", description = "")
    public double getWave() throws DriverException {
        if (md != null) {
            lastwl = md.getWave();
            return (lastwl);
        }
        return -1.;
    }

    /**
     ***************************************************************************
     **
     ** setWave: *
     * **************************************************************************
     *
     */
    @Command(name = "setWave", description = "")
    public void setWave(double wv) throws DriverException {
        if (isOnline()) {
            if (md != null) {
                md.setWave(wv);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** setTimeout: - set the read timeout in sec
     *
     * @param tko
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    @Command(name = "setTimeout", description = "set the read timeout")
    public void setTimeout(double tko) throws DriverException {
        if (isOnline()) {
            if (md != null) {
                md.setTimeout(tko);
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** setWave: *
     * **************************************************************************
     *
     */
    @Command(name = "setWaveAndFilter", description = "sets wavelength and the corresponding appropriate filter")
    public double setWaveAndFilter(double wv) throws DriverException {
        int ifilt = 0;
        double rwl = -1.;
        if (isOnline()) {
            if (md != null) {
                log.info("Setting wavelength to "+wv);
                md.setWave(wv);
                for (int i = 0; i < filter_edges.length; i++) {
//                    System.out.println("testing edge " + i);
//                    System.out.println("Checking wl=" + wv + " against edge #" + i + " with value " + filter_edges[i]);
                    if (filter_edges[i] > 0. && wv > filter_edges[i]) {
                        ifilt = i + 1;
                    }
                }
                log.info("Setting filter wheel position to " + ifilt + " for wl = " + wv + " nm");
                md.setFilter(ifilt);
                try {
                    Thread.sleep(4000);
                    try {
                        rwl = md.getWave();
                    } catch (Exception e) {
                        log.error("Failed to retrieve new wavelength setting. Trying again."+e);
                        Thread.sleep(10000);
                        try {
                            rwl = md.getWave();
                        } catch (Exception ee) {
                            log.error("Failed again to retrieve new wavelength setting. Trying again."+ee);
                            Thread.sleep(10000);
                            md.setWave(wv + 200.); // go somewhere else
                            Thread.sleep(10000);
                            md.setWave(wv); // then try coming back
                        }
                    }
                } catch (InterruptedException f) {
                    log.error("sleep interrupted!");
                }
            }
        }
        KeyValueData data = new KeyValueData(getName()+"/filterId",ifilt);
        subsys.publishSubsystemDataOnStatusBus(data);
        return (rwl);
    }

    /**
     ***************************************************************************
     **
     ** Sets the wl value for acquisition *
     * **************************************************************************
     */
    @Command(name = "setrunwave", description = "Sets the wl value for acquisition")
    public void setRunWave(double runWave) {
        this.runWave = runWave;
        return;
    }

    /**
     ***************************************************************************
     **
     ** Returns the wl value for acquisition *
     * **************************************************************************
     */
    @Command(name = "getrunwave", description = "Returns the wl value for acquisition")
    public double getRunWave() {
        return (runWave);
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_UNKNOWN) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;

//            log.info("C260Device readChannel called! ATTEMPT chan=" + chan + " type=" + type + " at " + now + " prev " + prevAnyTime);
//            log.info("C260Device readChannel called! PASSED chan=" + chan + " type=" + type + " at " + now);

        try {
            switch (chan) {
                case 0: // wave
                    value = this.getWave();
                    lastwl = value; // not needed
                    break;
                case -1: // runwave
                    value = this.getRunWave();
                    break;
                case 1: // shutter
                    value = this.isShutterOpen() ? 1 : 0;
                    lastshut = (int) value;
                    break;
                case 2: // filter
                    value = this.getFilter();
                    log.info("filter number is" + value);
                    lastfltpos = (int) value;
                    break;
                case 3: // slit size
                    value = this.getSlitSize(1);
                    lastslit1 = (int) value;
                    break;
                case 4: // slit 2 size
                    value = this.getSlitSize(2);
                    lastslit2 = (int) value;
                    break;
                case 5: // grating 
                    value = Integer.valueOf(this.getGrating().split(",")[0]);
                    lastgrat = (int) value;
                    break;
                case 6: // bandpass
                    value = this.getBandpass();
                    lastband = value;
                    break;
                case 7: // grating step
                    value = Integer.valueOf(this.getGrating().split(",")[1]);
                    laststep = (int) value;
                    //this.getStep();
                    break;
            }
        } catch (DriverLockedException f) {
            System.out.print("M"); // just to indicate that we are waiting
        } catch (DriverException e) {
            log.error("Error reading channel type " + chan + ": " + e);
        }
        return (value);

    }

}
