package org.lsst.ccs.subsystem.teststand;

import java.io.OutputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.thorlabs.ThorlabsFW;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.Agent;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.teststand.states.ThorlabsFWState;
import org.lsst.ccs.bus.data.KeyValueData;

/**
 *  Interfaces with driver for Thorlabs filter wheel (either the
 *  six-position model 102C or the twelve-position model 212C)
 *
 *  Changes in position are published as state transitions.
 *  Initial version identifies filters only by position, not by properties.
 *
 *  @author Al Eisner
 */
public class ThorlabsFWDevice extends Device {

   /**
    *  Data fields.
    */

    private final ThorlabsFW fwheel = new ThorlabsFW();  // Associated driver
    private final int maxTime = 11;               // Driver timeout is 10 sec

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService stateService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;

    private static final Logger LOG = Logger.getLogger(ThorlabsSC10Device.class.getName());

   /**
    *  Private lookup maps for driver read commands.  The default
    *  initial capacity of 16 is reasonable for the expected number
    *  of read commands.  The key is a numeric identifier of a monitor
    *  channel defined in the groovy file.
    *
    *  chanMap identifies the quantity to be read (i.e., the query name).
    *  It is taken from the type parameter in the groovy file.
    */
    private Map<Integer, String> chanMap = new HashMap<>();
    private Integer chanKey;

    /* Configuration parameters related to hardware settings */

    @ConfigurationParameter(isFinal = true, description = "port Id")
    protected volatile String devcId;

    @ConfigurationParameter(isFinal = true, 
                            description = "6 or 12 for filter-wheel size")
    protected volatile Integer fwsize;

    @ConfigurationParameter(isFinal = true, maxLength = 12,
                            description = "List of filter names")
    protected volatile List<String> filterNames;

    @ConfigurationParameter(description = "<false|true> for <low|high> speed of motion")
    protected volatile Boolean speedMode;

    @ConfigurationParameter(description = "<false|true> for sensors <inactive|active> when wheel idle")
    protected volatile Boolean sensorMode;

    /* Hard-code hardware trigger-connector mode to "output" */

    private final int trigModeVal = 1;

    /* Mapping of filter wheel position to filter description */
    // To be added


   /**
    *  Performs basic initialization.  
    *  Verify that all ConfigurationParameters are set.
    */
    @Override
    protected void initDevice() {

        stateService.registerState(ThorlabsFWState.class, "Filter Wheel State", this);

        chanKey = 0;

        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, name, "devcId", "is missing");
        }
        fullName = "ThorlabsFW " + name + " (" + devcId + ")";

        if (fwsize == null) {
            ErrorUtils.reportConfigError(LOG, name, "fwsize", "is missing");
        }

        if (filterNames == null) {
            ErrorUtils.reportConfigError(LOG, name, "filterNames", "is missing");
        }
	if (filterNames.size() != fwsize) {
            ErrorUtils.reportConfigError(LOG, name, "filterNames",
            "has size " + filterNames.size() + ", != fwsize " + fwsize);
	}

        if (speedMode == null) {
            ErrorUtils.reportConfigError(LOG, name, "speedMode", "is missing");
        }

        if (sensorMode == null) {
            ErrorUtils.reportConfigError(LOG, name, "sensorMode", "is missing");
        }

    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize() {

        try {
            fwheel.open(devcId);

            /* Hardware initialization */

            int speedModeVal = speedMode ? 1 : 0;
            int sensorModeVal = sensorMode ? 1 : 0;
            fwheel.setFW(ThorlabsFW.CmndFW.POSITION_COUNT, fwsize);
            fwheel.setFW(ThorlabsFW.CmndFW.SPEED_MODE, speedModeVal);
            fwheel.setFW(ThorlabsFW.CmndFW.SENSOR_MODE, sensorModeVal);
            fwheel.setFW(ThorlabsFW.CmndFW.TRIGGER_MODE, trigModeVal);
            int fwpos = Integer.parseInt(fwheel.queryFW(ThorlabsFW.CmndFW.POSITION));
            publishPosition(fwpos);

            initSensors();
            LOG.info("\n Connected to " + fullName);
            LOG.info("Starting filter wheel position for " + "ThorlabsFW "
                     + name + " = " + fwpos + " of " + fwsize);
            setOnline(true);
        }

        catch (DriverException e) {
            if (!inited) {
                LOG.severe("Error connecting to or initializing " 
                           + fullName + ": " + e);
            }
            close();
        }

        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            fwheel.close();
        }
        catch (DriverException e) {
        }
    }

    /**
     * Filter-wheel commands and settings
     */

   /**
    *  Select speed for filter wheel motion
    *
    *  @param mode  True for high-speed, false for low-speed
    *  @throws  DriverException
    */
    @ConfigurationParameterChanger(propertyName = "speedMode")
    public void setSpeedMode(Boolean mode) throws DriverException
    {
        if (isOnline()) {
            if (mode != null) {
                LOG.info(name + 
                         "config. change request: set SPEED_MODE = "+ mode);
                int speedModeVal = mode ? 1 : 0;
                fwheel.setFW(ThorlabsFW.CmndFW.SPEED_MODE, speedModeVal);
                speedMode = mode;
            } else {
                throw new IllegalArgumentException(name + ": improper value for speedMode, not changed");
            }
        } else {                      // initialization call
            if (mode != null) {speedMode = mode;}
        }
    }

   /**
    *  Select sensor mode when filter wheel is idle
    *
    *  @param mode  True for active, false for inactive
    *  @throws  DriverException
    */
    @ConfigurationParameterChanger(propertyName = "sensorMode")
    public void setSensorMode(Boolean mode) throws DriverException
    {
        if (isOnline()) {
            if (mode != null) {
                LOG.info(name + 
                         "config. change request: set SENSOR_MODE = "+ mode);
                int sensorModeVal = mode ? 1 : 0;
                fwheel.setFW(ThorlabsFW.CmndFW.SENSOR_MODE, sensorModeVal);
                sensorMode = mode;
            } else {
                throw new IllegalArgumentException(name + ": improper value for sensorMode, not changed");
            }
        } else {                      // initialization call
            if (mode != null) {sensorMode = mode;}
        }
    }

   /**
    *  Command to list filter names by position
    *
    *  @return  List of filter names
    */
    @Command(type=Command.CommandType.QUERY, name="listFilters",
             description="List filter names by position")
	public String listFilters() {
        String flist = name + " contains " + fwsize + " filters.  Names:";
        for (int i = 0; i < Math.max(filterNames.size(),fwsize); i++) {
            flist += String.format("\n  %2d  ",i+1) + filterNames.get(i);
        }
        return flist;
    }

   /**
    *  Command to move filter wheel (published as State change)
    *
    *  @param   Position to move to (1 to maximum)
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="setPosition",
             description="Set filter wheel position", timeout=maxTime)
    public void setPosition(@Argument(name="<int>",
     description="New position, 1 to maximum") int value)
     throws DriverException
    {
        fwheel.setFW(ThorlabsFW.CmndFW.POSITION, value);
        publishPosition(value);
    }

   /**
    *  Command to move filter wheel with position selected by name
    *  (published as State change)
    *
    *  @param   Name of selected filter
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="setNamedPosition",
             description="Set filter wheel to named position", timeout=maxTime)
    public void setNamedPosition(@Argument(name="<String>",
     description="Name of new position") String posName)
     throws DriverException
    {
        int fwpos = filterNames.indexOf(posName) + 1;
        LOG.info(name + "Position requested " + posName + " (" + fwpos + ")");
        if (fwpos > 0) {
            setPosition(fwpos);
        } else {
            LOG.severe(name + ":  " + posName + " not a valid filter name");
            throw new IllegalArgumentException(name + ":  " + posName + " not a valid filter name");
        }
    }

   /** 
    *  Publish filter wheel position (as a State) and filter name
    *
    *  @param   fwpos  Filter wheel position
    */
    private void publishPosition(int fwpos)
    {
        stateService.updateAgentComponentState(this, ThorlabsFWState.getEnum(fwpos));
        String fname = filterNames.get(fwpos-1);
        KeyValueData kvd = new KeyValueData(name + "/filterName", fname);
        agent.publishSubsystemDataOnStatusBus(kvd);
	LOG.info(name + " position set to filter " + fwpos + " (" + fname +")");
    }

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  chName   The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String chName, int hwChan, String type, 
                                 String subtype) throws Exception
    {
        try {
            ThorlabsFW.CmndFW.valueOf(type);
        }
        catch (IllegalArgumentException e) {
            ErrorUtils.reportChannelError(LOG, chName, "type", type);
        }
        chanKey++;
        chanMap.put(chanKey,type);
        return new int[]{chanKey, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);   //NaN
        String  chanName = chanMap.get(type);

        try {
            value = (double) Integer.parseInt(fwheel.queryFW(ThorlabsFW.CmndFW.valueOf(chanName)));
        }        
        catch (DriverTimeoutException et) {
            LOG.severe(name + " timeout reading " + chanName + ": " + et);
            setOnline(false);
        }
	catch (DriverException e) {
            LOG.severe(name + " exception reading data " + chanName + ": " + e);
        }
        
        return value;
    }

    /* Read-commands for command-line usage */

   /** 
    *  Read filter-wheel position
    *
    *  @return  Filter position number
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="getPosition", 
             description="Read filter wheel position")
    public int getPosition()  throws DriverException
    {
        return Integer.parseInt(fwheel.queryFW(ThorlabsFW.CmndFW.POSITION));
    }

   /** 
    *  Get filter-wheel position by name
    *
    *  @return  name of current filter
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="getNamedPosition", 
             description="Get name of current filter")
    public String getNamedPosition()  throws DriverException
    {
        return filterNames.get(getPosition() - 1);
    }

   /**
    * Read all settings and data from filter wheel
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all data read and exceptions.
    */
    @Command(type=Command.CommandType.QUERY, name="readAll", 
             description="Read all filter-wheel controller settings and data")
    public String readAll()
    {
        String table = "Read all filter wheel settings and data\n" + "\n";

        ThorlabsFW.CmndFW cmndN[] = ThorlabsFW.CmndFW.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                String respN = fwheel.queryFW(cmndN[i]);
                table += respN;
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";
        return table;
    }

}
