package org.lsst.ccs.subsystem.teststand;

import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.drivers.apcpdu.AP9630UPS;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.teststand.alerts.TS7Alerts;
import org.lsst.ccs.subsystem.teststand.data.TS7Outlets;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Monitor the status of the UPS
 * 
 * @author The LSSST CCS team
 */
public class UPSMonitor {

    private enum UPSLoadState {
        NOMINAL, ON_BATTERY, COLD_1_OFF, COLD_2_OFF, CRYO_OFF, REBS_OFF, ALL_OFF
    }
    private static final int UPS_BATTERY_TIME = 30000,
            UPS_COLD_1_OFF_TIME = UPS_BATTERY_TIME + 2000,
            UPS_COLD_2_OFF_TIME = UPS_COLD_1_OFF_TIME + 2000,
            UPS_REBS_OFF_TIME = UPS_COLD_2_OFF_TIME + 120000,
            UPS_VAC_REM_TIME = 600;

    private final TS7 ts7;
    private final AP9630UPSDevice upsDevc;
    private final Logger log;
    private final AgentPeriodicTaskService pts;
    private final AlertService als;

    private long upsDownTime = 0, upsOfflineTime = 0;
    private UPSLoadState upsState = UPSLoadState.NOMINAL;

    /**
     * Constructor
     * 
     * @param ts7
     * @param upsDevc
     * @param log 
     * @param pts
     * @param als
     */
    public UPSMonitor(TS7 ts7, AP9630UPSDevice upsDevc, Logger log, AgentPeriodicTaskService pts, AlertService als) {
        this.ts7 = ts7;
        this.upsDevc = upsDevc;
        this.log = log;
        this.pts = pts;
        this.als = als;
    }

    /**
     * Method called periodically to monitor the status of the UPS.
     *
     * If the UPS goes onto battery power, a sequence of power shutdown actions
     * takes place over an extended period.
     */
    void monitorUPS() {
        int status = AP9630UPS.ST_ONLINE;
        try {
            status = upsDevc.getStatus();
            if (upsOfflineTime == -1) {
                als.raiseAlert(TS7Alerts.UPS_DISCONNECTED.getAlert(), AlertState.NOMINAL, "");
            }
            upsOfflineTime = 0;
        } catch (DriverException e) {
            if (upsOfflineTime == 0) {
                log.error("Error reading UPS status: " + e);
                upsOfflineTime = System.currentTimeMillis();
                return;
            } else if (upsOfflineTime > 0) {
                if (System.currentTimeMillis() - upsOfflineTime <= 15000) {
                    return;
                }
                upsOfflineTime = -1;
                als.raiseAlert(TS7Alerts.UPS_DISCONNECTED.getAlert(), AlertState.ALARM, "");
            }
        }
        int downTime = upsDownTime == 0 ? 0 : (int) (System.currentTimeMillis() - upsDownTime);

        switch (upsState) {
        case NOMINAL:
            if (status != AP9630UPS.ST_ONLINE) {
                upsDownTime = System.currentTimeMillis();
                upsState = UPSLoadState.ON_BATTERY;
                als.raiseAlert(TS7Alerts.UPS_ON_BATTERY.getAlert(), AlertState.WARNING, "");
            }
            break;

        case ON_BATTERY:
            if (status == AP9630UPS.ST_ONLINE) {
                upsDownTime = 0;
                upsState = UPSLoadState.NOMINAL;
                als.raiseAlert(TS7Alerts.UPS_ON_BATTERY.getAlert(), AlertState.NOMINAL, "");
            } else if (downTime >= UPS_BATTERY_TIME) {
                upsState = UPSLoadState.COLD_1_OFF;
                als.raiseAlert(TS7Alerts.UPS_ON_BATTERY.getAlert(), AlertState.ALARM, "");
                if (ts7.isAlertResponseEnabled()) {
                    try {
                        log.info("Powering off cold plate refrigerator 1");
                        ts7.turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_1);
                    } catch (DriverException e) {
                        log.error("Error powering off cold plate refrigerator 1: " + e);
                    }
                } else {
                    log.info("Powering off cold plate refrigerator 1 was disabled");
                }
            }
            break;

        case COLD_1_OFF:
            if (downTime >= UPS_COLD_1_OFF_TIME) {
                upsState = UPSLoadState.COLD_2_OFF;
                if (ts7.isAlertResponseEnabled()) {
                    try {
                        log.info("Powering off cold plate refrigerator 2");
                        ts7.turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_2);
                    } catch (DriverException e) {
                        log.error("Error powering off cold plate refrigerator 2: " + e);
                    }
                } else {
                    log.info("Powering off cold plate refrigerator 2 was disabled");
                }
            }
            break;

        case COLD_2_OFF:
            if (downTime >= UPS_COLD_2_OFF_TIME) {
                upsState = UPSLoadState.CRYO_OFF;
                if (ts7.isAlertResponseEnabled()) {
                    try {
                        log.info("Powering off cryo plate refrigerator");
                        ts7.turnOutletOff(TS7Outlets.CRYOPLATEREFRIGERATOR);
                    } catch (DriverException e) {
                        log.error("Error powering off cryo plate refrigerator: " + e);
                    }
                } else {
                    log.info("Powering off cryo plate refrigerator was disabled");
                }
            }
            break;

        case CRYO_OFF:
            if (downTime >= UPS_REBS_OFF_TIME) {
                upsState = UPSLoadState.REBS_OFF;
                if (ts7.isAlertResponseEnabled()) {
                    try {
                        log.info("Powering off REB power supply");
                        ts7.turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY);
                    } catch (DriverException e) {
                        log.error("Error powering off REB power supply: " + e);
                    }
                } else {
                    log.info("Powering off REB power supply was disabled");
                }
            }
            break;

        case REBS_OFF:
            if (status == AP9630UPS.ST_ONLINE) {
                upsDownTime = 0;
                upsState = UPSLoadState.NOMINAL;
                als.raiseAlert(TS7Alerts.UPS_ON_BATTERY.getAlert(), AlertState.NOMINAL, "");
            }
            else {
                int remTime;
                try {
                    remTime = upsDevc.getRemTime();
                } catch (DriverException e) {
                    log.error("Error reading remaining UPS time: " + e);
                    break;
                }
                if (remTime < UPS_VAC_REM_TIME) {
                    upsState = UPSLoadState.ALL_OFF;
                    if (ts7.isAlertResponseEnabled()) {
                        try {
                            log.info("Powering off vacuum valve");
                            ts7.turnOutletOff(TS7Outlets.VACUUMVALVE);
                            log.info("Powering off vacuum turbo pump");
                            ts7.turnOutletOff(TS7Outlets.TURBOPUMP);
                            log.info("Powering off vacuum roughing pump");
                            ts7.turnOutletOff(TS7Outlets.ROUGHINGPUMP);
                        } catch (DriverException e) {
                            log.error("Error powering off vacuum components: " + e);
                        }
                    } else {
                        log.info("Powering off vacuum components was disabled");
                    }
                }
            }
            break;

        case ALL_OFF:
            if (status == AP9630UPS.ST_ONLINE) {
                upsDownTime = 0;
                upsState = UPSLoadState.NOMINAL;
                als.raiseAlert(TS7Alerts.UPS_ON_BATTERY.getAlert(), AlertState.NOMINAL, "");
            }
            break;
        }
    }

}
