package org.lsst.ccs.subsystem.teststand;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.thorlabs.ThorlabsSC10;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.Subsystem;
//import org.lsst.ccs.subsystem.teststand.states.ThorlabsSC10State;

/**
 *  Interfaces with driver for Thorlabs SC10 shutter controller 
 *
 *  @author Al Eisner
 */
public class ThorlabsSC10Device extends Device {

   /**
    *  Data fields.
    */

    private final ThorlabsSC10 shutr = new ThorlabsSC10();  // Associated driver

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected AgentStateService stateService;

    private static final Logger LOG = Logger.getLogger(ThorlabsSC10Device.class.getName());

   /**
    *  Private lookup maps for driver read commands.  The default
    *  initial capacity of 16 is reasonable for the expected number
    *  of read commands.  The key is a numeric identifier of a monitor
    *  channel defined in the groovy file.
    *
    *  chanMap identifies the quantity to be read (i.e., the query name).
    *  It is taken from the type parameter in the groovy file.
    */
    private Map<Integer, String> chanMap = new HashMap<>();
    private Integer chanKey;
    private Duration exposureDuration;
    private long startExposure;
    private final Duration intervalCheckClosed = Duration.ofMillis(100);

    /* Configuration parameters related to hardware settings */

    @ConfigurationParameter(isFinal = true, description = "port Id")
    protected volatile String devcId;

    @ConfigurationParameter(isFinal = true,
                            description = "<false|true> for output open/closed line to follow <shutter|controller> ")
    protected volatile Boolean outputMode;

   /**
    *  Performs basic initialization.  
    *  Verify that all ConfigurationParameters are set.
    */
    @Override
    protected void initDevice() {

        chanKey = 0;

        if (devcId == null) {
            MonitorLogUtils.reportConfigError(log, name, "devcId", "is missing");
        }
        fullName = "ThorlabsSC10 " + name + " (" + devcId + ")";

        if (outputMode == null) {
            MonitorLogUtils.reportConfigError(log, name, "outputMode", "is missing");
        }
    }

   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize() {

        try {
            shutr.open(devcId);

            /* Hardware initialization */

            /**
             *  These modes are used as convenient initial states.
             *  (Action Commands should set them appropriately).
             */
            ThorlabsSC10.TrigMode trgMode = ThorlabsSC10.TrigMode.INTERNAL;
            ThorlabsSC10.OpMode opMode = ThorlabsSC10.OpMode.MANUAL;

            int outputModeVal = outputMode ? 1 : 0;
            shutr.setSC(ThorlabsSC10.CmndSC.TRIG_OUT_MODE, outputModeVal);
            shutr.setSC(ThorlabsSC10.CmndSC.TRIGGER_MODE, trgMode.getSetting());
            shutr.setSC(ThorlabsSC10.CmndSC.OPERATING_MODE, opMode.getSetting());
            /* If shutter is open, close it */
            if (!isShutterClosed()) {
                shutr.enable();  // In MANUAL mode, enable toggles shutter
            }

            setOnline(true);
            initSensors();
            LOG.log(Level.INFO, "\n Connected to {0}", fullName);
            //stateService.updateAgentComponentState(this, ThorlabsSC10State.getEnum(fwpos));
        }

        catch (DriverException e) {
            if (!inited) {
                LOG.log(Level.SEVERE, "Error connecting to or initializing {0}: {1}", new Object[]{fullName, e});
            }
            close();
        }

        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            shutr.close();
        }
        catch (DriverException e) {
        }
    }

    /**
     *  Check whether shutter is closed or open
     *
     *  @return  <true|false> for shutter <closed|open>
     *  throws  DriverException
     */
    public boolean isShutterClosed() throws DriverException 
    {
        return ((Integer.parseInt(shutr.querySC(ThorlabsSC10.CmndSC.CLOSED)) == 1) ? true : false);
    }

   /**
    * Waits for the shutter to be in the given state.
    *
    * @param  open     If <true|false>, wait for shutter <open|closed>
    * @param  timeout  Maximum time to wait
    * @return The elapsed time in milliseconds
    * @throws DriverException If there is an error talking to the driver
    * @throws TimeoutException  If timeout expires before reaching requested state.
    */
    @SuppressWarnings("SleepWhileInLoop")
	private long waitForShutterState(boolean open, Duration timeout) throws TimeoutException, DriverException {
        long start = System.currentTimeMillis();
        long timeoutTime = start + timeout.toMillis();
        while (true) {
            if (isShutterClosed() != open) {
                break;
            }
            try {
                Thread.sleep(intervalCheckClosed.toMillis());
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt while waiting for Thorlabs shutter state ",ex);
            }
            if (System.currentTimeMillis() > timeoutTime) {
               throw new TimeoutException(name + " wait for shutter to "+(open ? "open" : "close")+" timed out");
            }
        }
        return System.currentTimeMillis() - start;
    }


    /**
     * Shutter commands and settings
     */

   /**
    *  Select mode for output signal
    *
    *  @param mode  <false|true> for signal to follow <shutter|controller>
    *  @throws  DriverException
    */
    @ConfigurationParameterChanger(propertyName = "outputMode")
    public void setOutputMode(Boolean mode) throws DriverException
    {
        if (isOnline()) {
            if (mode != null) {
                LOG.log(Level.INFO, "{0}config. change request: set TRIG_OUT_MODE = {1}", new Object[]{name, mode});
                int outputModeVal = mode ? 1 : 0;
                shutr.setSC(ThorlabsSC10.CmndSC.TRIG_OUT_MODE, outputModeVal);
                outputMode = mode;
            } else {
                throw new IllegalArgumentException(name + ": improper value for outputMode, not changed");
            }
        } else {                      // initialization call
            if (mode != null) {outputMode = mode;}
        }
    }

   /**
    *  Command to open shutter
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="openShutter",
             description="Open shutter", autoAck=false)
    public void openShutter() throws DriverException
    {
        subsys.helper()
        .precondition(isShutterClosed(), name + "shutter already open")
        .action(() -> {
            shutr.setSC(ThorlabsSC10.CmndSC.TRIGGER_MODE,
                        ThorlabsSC10.TrigMode.INTERNAL.getSetting());
            shutr.setSC(ThorlabsSC10.CmndSC.OPERATING_MODE, 
                        ThorlabsSC10.OpMode.MANUAL.getSetting());
            shutr.enable();
            LOG.log(Level.INFO, "{0} opened", name);
        });
    }

   /**
    *  Command to close shutter
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="closeShutter",
             description="Close shutter", autoAck=false)
    public void closeShutter() throws DriverException
    {
        subsys.helper()
        .precondition(!isShutterClosed(), name + "shutter already closed")
        .action(() -> {
            shutr.setSC(ThorlabsSC10.CmndSC.TRIGGER_MODE,
                        ThorlabsSC10.TrigMode.INTERNAL.getSetting());
            shutr.setSC(ThorlabsSC10.CmndSC.OPERATING_MODE, 
                        ThorlabsSC10.OpMode.MANUAL.getSetting());
            shutr.enable();
            LOG.log(Level.INFO, "{0} closed", name);
        });
    }

   /**
    *  Command to open shutter for a specified time interval
    *
    *  @param   duration  Open-duration as a Duration object
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="exposure",
             description="Open shutter for specified time interval (ms)")
    public void exposure(@Argument(name="<Duration>",
     description="exposure duration") Duration duration)
	throws DriverException, TimeoutException
    {
        long requestExposure = System.currentTimeMillis();
        shutr.setSC(ThorlabsSC10.CmndSC.TRIGGER_MODE,
                    ThorlabsSC10.TrigMode.INTERNAL.getSetting());
        shutr.setSC(ThorlabsSC10.CmndSC.OPERATING_MODE, 
                    ThorlabsSC10.OpMode.SINGLE.getSetting());
        shutr.setSC(ThorlabsSC10.CmndSC.OPEN_DURATION, (int) duration.toMillis());
        shutr.enable();      // This is the actual action command
        long delta = System.currentTimeMillis() - requestExposure;
        // Wait until actually open, log how long it took
        long waitTime = waitForShutterState(true, Duration.ofSeconds(5));
        startExposure = System.currentTimeMillis();
        LOG.log(waitTime < 500. ? Level.INFO : Level.WARNING, "{0} requested exposure duration {1} (after {2} ms to configure device and an additional {3} ms to open)", new Object[]{name, duration, delta, waitTime});
        exposureDuration = duration;
    }

   /**
    *  Command to wait for shutter to close after an exposure.  It assumes
    *  that the exposure() method has waited for shutter to open.
    *  Throws TimeoutException if wait exceeds duration by > 1 second
    *
    *  @throws TimeoutException
    */

    @Command(type=Command.CommandType.ACTION, name="waitForClosed",
             description="Returns when shutter is closed after exposure",
             autoAck=false) 
    public void waitForClosed() throws TimeoutException {
        subsys.helper()
	.precondition(exposureDuration != null, name + ": No exposure in progress")
	.duration((exposureDuration != null) ? exposureDuration.plusMillis(1500) : exposureDuration)
        .action(() -> {
            Duration expectedDuration = exposureDuration;
	    Duration timeout = exposureDuration.plusMillis(1000);  // add leeway
            exposureDuration = null;    // reset until next exposure
            long waitTime = waitForShutterState(false, timeout);
            LOG.log(Level.INFO, "{0} closed after requested duration {1}, actually open for {2} ms", new Object[]{name, expectedDuration, System.currentTimeMillis() - startExposure});            
            startExposure = -1;
	});
    }

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  chName   The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String chName, int hwChan, String type, 
                                 String subtype) throws Exception
    {
        try {
            ThorlabsSC10.CmndSC.valueOf(type);
        }
        catch (IllegalArgumentException e) {
            MonitorLogUtils.reportError(log, chName, "type", type);
        }
        chanKey++;
        chanMap.put(chanKey,type);
        return new int[]{chanKey, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);   //NaN
        String  chanName = chanMap.get(type);

        try {
            value = (double) Integer.parseInt(shutr.querySC(ThorlabsSC10.CmndSC.valueOf(chanName)));
        }        
        catch (DriverTimeoutException et) {
            log.error(name + " timeout reading " + chanName + ": " + et);
            setOnline(false);
        }
	catch (DriverException e) {
            LOG.log(Level.SEVERE, "{0} exception reading data {1}: {2}", new Object[]{name, chanName, e});
        }
        
        return value;
    }

    /* Read-commands for command-line usage */

   /**
    * Read all settings and data from shutter controller
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all data read and exceptions.
    */
    @Command(type=Command.CommandType.QUERY, name="readAll", 
             description="Read all filter-wheel controller settings and data")
    public String readAll()
    {

        String table = "Read all shutter settings and data\n" + "\n";

        ThorlabsSC10.CmndSC cmndN[] = ThorlabsSC10.CmndSC.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                String respN = shutr.querySC(cmndN[i]);
                table += respN;
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";
        return table;
    }

}
