package org.lsst.ccs.subsystem.teststand;

import java.time.Duration;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Predicate;
import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.devices.power.distribution.APC7900Device;
import org.lsst.ccs.subsystem.common.devices.vacuum.VQMDevice;
import org.lsst.ccs.subsystem.power.states.RebPowerState;
import org.lsst.ccs.subsystem.teststand.data.TS7Outlets;
import org.lsst.ccs.subsystem.teststand.data.TestStandAgentProperties;
import org.lsst.ccs.subsystem.teststand.limits.TransitionAlgorithm;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Implementation of SLAC TS7
 *
 * @author The LSST CCS Team
 */
public class TS7 implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    @LookupName
    private String name;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    CryoDevice cryoDevc;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    VQMDevice vqmDevc;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    protected final Map<String, APC7900Device> pduDevicesMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    AP9630UPSDevice upsDevc;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private ConfigurationService configurationService;

    @ConfigurationParameter(isFinal = true)
    private String rebPsSubsystem = "ccs-rebps";

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService pts;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService als;

    private final Logger LOGGER = Logger.getLogger(getClass().getPackage().getName());

    private final Object initializationLock = new Object();
    private boolean initializedRebPowerState = false;
    private RebPowerState currentRebsPowerState = RebPowerState.OFF;
    private TransitionAlgorithm vacuumTransitionAlgorithm;
    private TransitionAlgorithm coldTransitionAlgorithm1;
    private TransitionAlgorithm coldTransitionAlgorithm2;
    private TransitionAlgorithm cryoTransitionAlgorithm;

    private volatile boolean alertResponseEnabled = false;
    private UPSMonitor upsMon;

    /**
     * Post-initialization
     */
    @Override
    public void postInit() {
        if (cryoDevc == null) {
            //What do you want to do if cryo is not present?
            LOGGER.warning("Cryo device not defined");
        }

        
        // By setting TESTSTAND_TYPE_AGENT_PROPERTY we signal to consoles that this subsystem is compatible with the teststand subsystm GUI
        subsys.getAgentService(AgentPropertiesService.class).setAgentProperty(TestStandAgentProperties.TESTSTAND_TYPE_AGENT_PROPERTY, TS7.class.getCanonicalName());
        
        // Set up timer to handle transitions
        upsMon = new UPSMonitor(this, upsDevc, LOGGER, pts, als);
    }

    @Override
    public void build() {
        pts.scheduleAgentPeriodicTask(new AgentPeriodicTask("ups-timer", () -> {
            upsMon.monitorUPS();
        }));
        pts.setPeriodicTaskPeriod("ups-timer", Duration.ofSeconds(1));
    }
    


    /**
     * Post start processing
     */
    @Override
    public void postStart() {
        Predicate<BusMessage<? extends Serializable, ?>> filter
                = BusMessageFilterFactory.messageOrigin(rebPsSubsystem);

        subsys.getMessagingAccess().addStatusMessageListener((msg) -> handlePowerSupplyMessage(msg), filter);
    }

    /**
     * Gets the list of PDU names.
     *
     * @return The list of names
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the list of PDU names")
    public List<String> getPduNames() {
        return new ArrayList<>(pduDevicesMap.keySet());
    }

    /**
     * Gets the list of PDU outlet names.
     *
     * @return The list of names (of the form "pdu/outlet")
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the list of PDU outlet names")
    public List<String> getOutletNames() {
        List names = new ArrayList<>();
        for (String pduName : pduDevicesMap.keySet()) {
            for (String outletName : pduDevicesMap.get(pduName).getOutletNames()) {
                names.add(pduName + "/" + outletName);
            }
        }
        return names;
    }

    /**
     * Gets the map of PDU outlet on states.
     *
     * @return The map of names (of the form "pdu/outlet") to on states
     * (true/false)
     * @throws DriverException
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the map of PDU outlet on states")
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException {
        Map states = new HashMap<>();
        for (String pduName : pduDevicesMap.keySet()) {
            Map<String, Boolean> pduStates = pduDevicesMap.get(pduName).getOutletOnStateMap();
            for (String outletName : pduStates.keySet()) {
                states.put(pduName + "/" + outletName, pduStates.get(outletName));
            }
        }
        return states;
    }

    /**
     * Handles REB PS state messages. NOTICE: This method is executed on the
     * Message dispatch thread. It should be light weight.
     */
    private void handlePowerSupplyMessage(StatusMessage msg) {
        final StateBundle sb;
        synchronized (initializationLock) {
            if (initializedRebPowerState && msg instanceof StatusStateChangeNotification) {
                StatusStateChangeNotification ssn = (StatusStateChangeNotification) msg;
                sb = ssn.getNewState();
            } else if (!initializedRebPowerState) {
                sb = msg.getState();
            } else {
                sb = null;
            }
            if (sb != null) {
                RebPowerState powerState = (RebPowerState) sb.getState(RebPowerState.class);
                if (powerState != null) {
                    initializedRebPowerState = true;
                }
                if (currentRebsPowerState != powerState) {
                    currentRebsPowerState = powerState;
                    // TODO: We should have some support in core for handling firing off actions
                    new Thread(() -> updatePowerState(currentRebsPowerState, sb.getComponentsWithState(RebPowerState.class))).start();
                }
            }

        }

    }

    /**
     * Gets the current REB power state
     * 
     * @return The REB power state
     */
    public RebPowerState getCurrentRebsPowerState() {
        return currentRebsPowerState;
    }

    private void updatePowerState(RebPowerState state, Map<String, RebPowerState> map) {
        String rebStateStr = "";
        for (Entry<String, RebPowerState> e : map.entrySet()) {
            rebStateStr += e.getKey() + "=" + e.getValue() + " ";
        }
        LOGGER.info("Updating power state to " + state + " " + rebStateStr);
        // This no longer does anything, since limits no longer depend on power state
    }

    @Command(type = Command.CommandType.ACTION, description = "Turn Off named outlet")
    public void turnOutletOff(TS7Outlets outlet) throws DriverException {
        changeOutletState(outlet, false);
    }

    @Command(type = Command.CommandType.ACTION, description = "Turn On named outlet")
    public void turnOutletOn(TS7Outlets outlet) throws DriverException {
        changeOutletState(outlet, true);
    }

    private void changeOutletState(TS7Outlets outlet, boolean on) throws DriverException {
        String outletName = outlet.getOutletName();
        for (APC7900Device pdu : pduDevicesMap.values()) {
            if (pdu.getOutletNames().contains(outletName)) {
                if (on) {
                    pdu.forceOutletOn(outletName);
                } else {
                    pdu.forceOutletOff(outletName);
                }
                return;
            }
        }
        throw new IllegalArgumentException("Could not find device to turn off outlet " + outletName);
    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable alert response")
    public void enableAlertResponse(boolean enable) {
        this.alertResponseEnabled = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isAlertResponseEnabled() {
        return alertResponseEnabled;
    }
        
}

