package org.lsst.ccs.subsystem.teststand.limits;

import java.util.List;

class ProportionalLimitAlgorithm extends LimitAlgorithm {

    private double initialAlarmLowFactor, initialWarnLowFactor, initialWarnHighFactor, initialAlarmHighFactor;
    private double targetAlarmLowFactor, targetWarnLowFactor, targetWarnHighFactor, targetAlarmHighFactor;

    @Override
    public void init(List<Double> parameters, double initialValue, LimitsInterface initialLimits, double targetValue, LimitsInterface targetLimits) {
        super.init(parameters, initialValue, initialLimits, targetValue, targetLimits);

        this.initialAlarmLowFactor = Math.abs(initialLimits.getAlarmLow()/initialValue);
        this.initialWarnLowFactor = Math.abs(initialLimits.getWarnLow()/initialValue);
        this.initialWarnHighFactor = Math.abs(initialLimits.getWarnHigh()/initialValue);
        this.initialAlarmHighFactor = Math.abs(initialLimits.getAlarmHigh()/initialValue);
        
        if (initialAlarmLowFactor <=0 || initialAlarmLowFactor > initialWarnLowFactor || initialAlarmHighFactor < initialWarnHighFactor) {
            throw new IllegalArgumentException("Proportional limit algorith values are invalid ["+initialAlarmLowFactor+","+initialWarnLowFactor+","+initialWarnHighFactor+","+initialAlarmHighFactor+"]");
        }
        
        this.targetAlarmLowFactor = Math.abs(targetLimits.getAlarmLow()/targetValue);
        this.targetWarnLowFactor = Math.abs(targetLimits.getWarnLow()/targetValue);
        this.targetWarnHighFactor = Math.abs(targetLimits.getWarnHigh()/targetValue);
        this.targetAlarmHighFactor = Math.abs(targetLimits.getAlarmHigh()/targetValue);
        
        if (targetAlarmLowFactor <=0 || targetAlarmLowFactor > targetWarnLowFactor || targetAlarmHighFactor < targetWarnHighFactor) {
            throw new IllegalArgumentException("Proportional limit algorith values are invalid");
        }
        
    }

    @Override
    public void adjust(double currentValue, LimitsInterface limits) {
        double alpha = Math.abs((currentValue - getTargetValue())/(getInitialValue() - getTargetValue()));
        double alarmLowFactor = alpha*initialAlarmLowFactor + (1.0-alpha)*targetAlarmLowFactor;
        double warnLowFactor = alpha*initialWarnLowFactor + (1.0-alpha)*targetWarnLowFactor;
        double warnHighFactor = alpha*initialWarnHighFactor + (1.0-alpha)*targetWarnHighFactor;
        double alarmHighFactor = alpha*initialAlarmHighFactor + (1.0-alpha)*targetAlarmHighFactor;
        limits.setLimits(currentValue*alarmLowFactor, currentValue*warnLowFactor, currentValue*warnHighFactor, currentValue*alarmHighFactor);
    }
}