package org.lsst.ccs.subsystem.teststand;

import hep.aida.*;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.Date;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSFluxCalib;
import org.lsst.ccs.subsystem.teststand.data.TSState;
import org.lsst.ccs.subsystem.teststand.data.TestStandAgentProperties;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Implements the test stand EO sensor modular subsystem.
 *
 * @author Homer Neal *
 */
public class TSSubSys implements HasLifecycle, Monitor.AlarmHandler {

    boolean notify = false; // distribute messages concerning ALARMS

    TSConfig teststandc = new TSConfig();
    TSConfigCatcher tscc = new TSConfigCatcher(teststandc);
    TSFluxCalib teststandf = new TSFluxCalib();
    String stateCorrelId;

    private ConcurrentMessagingUtils cmu;
    private static final Logger LOGGER = Logger.getLogger("org.lsst.ccs.subsystem.metrology.main");

    /**
     * Constants
     */
    String configName = "test";
    int state = 0;

    boolean last_systemOK = true; // last system check status
    int nwarn = 0; // number of consequtive warnings

    double last_vac = 0.;

    /**
     * Data fields
     */
//    private Toolkit toolkit = Toolkit.getDefaultToolkit();
    private static final String teststand_dest = System.getProperty("lsst.ccs.teststand.tsguidest", "ts2");
    /* define references to the TS devices */
//    SensorControllerDevice ccdDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    BiasDevice biasDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    PhotoDiodeDevice pdDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    CryoDevice cryoDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    VacuumGaugeDevice vacDevc;    
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    LampDevice lmpDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    MonochromatorDevice monoDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    Fe55Device fe55Devc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    ShutterDevice shDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    EnviroDevice envDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    LEDDevice ledDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    PDUDevice pduDevc;
    @LookupField(strategy=LookupField.Strategy.CHILDREN)
    VQMDevice vqmDevc;

    @LookupField(strategy=LookupField.Strategy.TOP)
    Subsystem subsys;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupName
    private String name;
    
    public static final String BROADCAST_TASK = "publish-data";

    private long lastWarningTMillis = 0;
    private long warningInterval = 240000;
    private String lastWarning = "";
    private boolean doTestStandReadyCheck = true;
    private long broadcastMillis = 10000;

    /**
     * Main constructor.
     * 
     * @param configName
     */
    public TSSubSys(String configName) {
        this.configName = configName;
//        this.configName = System.getProperty("lsst.ccs.teststand.configName", "");
//        coldStart = cold.equals("true");
//        initModule(); // put the system into the expected initial state 
    }


    /**
     * Initialization
     */
    @Override
    public void build() {
        
        //Define the Runnable object to be invoked periodically
        Runnable myRunnable = new Runnable() {
            @Override
            public void run() {
                //The code to be executed
                periodicBroadcast();
            }
        };

        //Create an AgentPeriodicTask that will execute the above Runnable instance
        //every second at a fixed rate
        AgentPeriodicTask periodicTask = new AgentPeriodicTask(BROADCAST_TASK, myRunnable).withPeriod(Duration.ofMillis(broadcastMillis));

        //Schedule the periodic task with the AgentPeriodicTaskService
        periodicTaskService.scheduleAgentPeriodicTask(periodicTask);
        
    }
    
    /**
     * Initializes the test stand subsystem. ccdDevc; BiasDevice biasDevc;
     * PhotoDiodeDevice pdDevc; CryoDevice cryoDevc; VacuumGaugeDevice vacDevc;
     * MonochromatorDevice monoDevc; Fe55Device fe55Devc; EnviroDevice envDevc;
     */
    @Override
    public void postInit() {

        // By setting TESTSTAND_TYPE_AGENT_PROPERTY we signal to consoles that this subsystem is compatible with the teststand subsystm GUI
        subsys.setAgentProperty(TestStandAgentProperties.TESTSTAND_TYPE_AGENT_PROPERTY, TSSubSys.class.getCanonicalName());

        /*
         **  Initialize all configuration data
         */
//        teststandc.TSConfigReader(log);
        teststandf.TSFluxCalibReader();
// ----
    }


    /**
     * Main startup
     * 
     * @throws HardwareException 
     */
    @Override
    public void postStart() {
        // put TS subsystem into a ready state
        LOGGER.info("Executing teststand startup procedure");

        cmu = new ConcurrentMessagingUtils(subsys.getMessagingAccess());
        
        try {
            startSubSys();
        } catch (Exception e) {
            LOGGER.error("Error starting Test Stand subsystem: " + e);
        }

        try {
            teststandc.setCfgState(TSConfig.operating_states.READY.ordinal());
            this.setStateFlags(TSConfig.operating_states.READY);
        } catch (Exception ee) {
            LOGGER.error("Exception while setting state flags:", ee);
        }


        publishState();

        state |= TSState.READY;

        //Create analysis factory
        IAnalysisFactory af = IAnalysisFactory.create();

// Create datapoint set factory
//        ITree tree = af.createTreeFactory().create();
        IDataPointSetFactory dataPointSetFactory = af.createDataPointSetFactory(null);
        if (pdDevc != null) {
            pdDevc.setDPSF(dataPointSetFactory);
        }
        if (biasDevc != null) {
            biasDevc.setDPSF(dataPointSetFactory);
        }

        LOGGER.info("DPSF started.");

// Create histogram factory
//        IHistogramFactory histogramFactory = af.createHistogramFactory(null);
    }

    /**
     * Performs periodic trending data broadcast.
     */
    protected void periodicBroadcast() {
        /*
         **  Broadcast the state
         */
        System.out.print(".");

        if ((state & TSState.SHUTDOWN) != 0) {
            System.exit(0); // gone in a tick
        }
        publishState();

        ConfigurationInfo configInfo = subsys.getConfigurationService().getConfigurationInfo();
        tscc.updateConfig(configInfo);

        if (doTestStandReadyCheck) {
            if (pdDevc != null && biasDevc != null) {
                int pdState = pdDevc.getState();
                int biasState = biasDevc.getState();
                LOGGER.info("The PD and Bias states are:" + TSState.pwrstates.values()[pdState] + " , " + TSState.pwrstates.values()[biasState]);
                if (pdState == TSState.pwrstates.NOTCONFIGURED.ordinal()
                        && biasState == TSState.pwrstates.NOTCONFIGURED.ordinal()) {

// make sure it was not a simple misread
                    LOGGER.error("Both PD and BIAS devices are in the NOTCONFIGURED state. Checking for a possible POWER OUTAGE.");
                    boolean pdOK = true;
                    try {
                        System.out.println("Doing a soft reset of PD device");
                        pdDevc.softReset();
                        System.out.println("Trying a current read.");
                        pdDevc.readCurrent();
                        System.out.println("PD state is " + TSState.pwrstates.values()[pdDevc.getState()]);
                        if (pdDevc.getState() == TSState.pwrstates.NOTCONFIGURED.ordinal()) {
                            pdOK = false;
                        }
                    } catch (Exception pp) {
                        pdOK = false;
                    }
                    boolean biasOK = true;
                    try {
                        System.out.println("Doing a soft reset of Bias device");
                        biasDevc.softReset();
                        System.out.println("Trying a current read.");
                        biasDevc.readCurrent();
                        System.out.println("Bias state is " + TSState.pwrstates.values()[biasDevc.getState()]);
                        if (biasDevc.getState() == TSState.pwrstates.NOTCONFIGURED.ordinal()) {
                            biasOK = false;
                        }
                    } catch (Exception pp) {
                        biasOK = false;
                    }
                    System.out.println("PD, Bias state summary = " + pdOK + " , " + biasOK);
                    if (pdOK == false && biasOK == false) { // POWER OUTAGE .... take action
                        powerOutageAction();
                    }
                }
            }
            try {
                // check whether the test stand equipment is within tolerance for the current state
                isTestStandReady();
            } catch (Exception ex) {
                LOGGER.error("There was an error during the tick check of the test stand readiness:", ex);
            }
        }

    }

    /**
     * Put the system in a safe mode when a power outage has been signaled
     */
    @Command(type = CommandType.ACTION, description = "Put system in safe mode")
    public void powerOutageAction() {
        LOGGER.error("A POWER OUTAGE SITUATION HAS BEEN IDENTIFIED ... TAKING ACTIONS TO PROTECT SENSOR");
        LOGGER.error("A POWER OUTAGE SITUATION HAS BEEN IDENTIFIED ... TAKING ACTIONS TO PROTECT SENSOR");
        LOGGER.error("A POWER OUTAGE SITUATION HAS BEEN IDENTIFIED ... TAKING ACTIONS TO PROTECT SENSOR");
// in case the bias device is somehow still alive, disable the output
        try {
            biasDevc.setVoltage(0.0);
            biasDevc.setOutput(0);
        } catch (Exception pp) {
            LOGGER.error("Not surprisingly, we failed to communicate with the bias supply. It is fine that it is down.");
        }

// make sure the power to the PolyCold is off
        try {
            LOGGER.fine("SHUTTING OFF POWER TO THE CRYO PUMP");
            int pd_outlet = Integer.decode(System.getenv("CCS_CRYO_OUTLET"));
            this.pduDevc.setOutletState(pd_outlet, false);
        } catch (Exception e1) {
            LOGGER.error("Failed to power off cryo pump: " + e1);
        }
        // warn other subsystems of the situation
        KeyValueData kd = new KeyValueData("TS_Hazard", "SensorHazard: Power Outage");
        subsys.publishSubsystemDataOnStatusBus(kd);
        try {
            sendSyncTSCommand("abortInterpreter");
        } catch (Exception e3) {
            LOGGER.error("Failed to abort script: " + e3);
        }
        try {
            // start warming
            int iloop = cryoDevc.getCurrent_loop();
            cryoDevc.setHeaterRange(iloop, "LOW");
            cryoDevc.setSetPoint(iloop, 20.);
//                        cryoDevc.rampTemp(7200., 25., 60);
//                        setTSWarm();
        } catch (Exception ex) {
            LOGGER.error(ex);
        }
    }

    /**
     * Raise an alert
     * 
     * @param alertmsg
     * @param severity 
     */
    @Command(type = CommandType.QUERY, description = "raise subsystem alert")
    public void raiseTSAlert(String alertmsg, AlertState severity) {
        LOGGER.info("sending alert message: " + alertmsg);
        LOGGER.info("alert message length = " + alertmsg.length());

        org.lsst.ccs.bus.data.Alert a = new org.lsst.ccs.bus.data.Alert("Teststand " + configName + " Alert", alertmsg.substring(0, Math.min(254, alertmsg.length())));
        alertService.raiseAlert(a, severity, alertmsg);
    }

    @Override
    public boolean processAlarm(int event, int parm, String cause, String alarmName) {
        String alarm_ev = null;
        switch (event) {
            case Alarm.EVENT_TRIP:
                alarm_ev = "TRIP";
                break;
            case Alarm.EVENT_RESET:
                alarm_ev = "RESET";
                break;
            default:
                alarm_ev = "Unknown";
        }

        LOGGER.error("Received alarm for " + alarm_ev + " event, where trip=" + Alarm.EVENT_TRIP + " and reset=" + Alarm.EVENT_RESET);
        LOGGER.error("Alarm is from " + TSConfig.EVENT_ID.values()[parm]);

        String alarm_name = configName + ":CCS alarm for " + alarm_ev + "event, from " + TSConfig.EVENT_ID.values()[parm];
        String alarm_msg = null;

        switch (event) {

            case Alarm.EVENT_TRIP:
//                toolkit.beep();
                if (parm == TSConfig.EVENT_ID.BIAS.ordinal() //&& (state & TSState.EXPERT) == 0
                        ) {
                    alarm_msg = "***** Over VOLTAGE or CURRENT alarm received! For safety, the bias voltage is being set OFF! ";
                    if (biasDevc != null) {
                        alarm_name += " V= " + biasDevc.readVoltage() + ", I= " + biasDevc.readCurrent();
                        biasDevc.setAbort(true);
                        biasDevc.setState(TSState.pwrstates.TRIPPED.ordinal());
                        this.raiseTSAlert(alarm_name, AlertState.ALARM);
                        sleep(5.0);
                        biasDevc.setAbort(false);
                        rampBiasVolts(5.0, 0.0);
                        setBiasOutput(0);
                    }
                    state |= TSState.PWRDEVC_TRIPPED;
                    state &= ~TSState.READY;
                }
                if (parm == TSConfig.EVENT_ID.CRYO.ordinal()) {
                    alarm_msg = "***** Temperature alarm received! Please rectify and reset manually!";
                    try {
                        alarm_name += " T= " + cryoDevc.getTemp(cryoDevc.getCurrent_channel());
                    } catch (DriverException ex) {
                        LOGGER.error("Failed to retrieve temperature for alarm message", ex);
                    }
                    cryoDevc.setState(TSState.cryostates.TRIPPED.ordinal());
                    state |= TSState.CRYODEVC_TRIPPED;
                    state &= ~TSState.READY;
                }
                if (parm == TSConfig.EVENT_ID.VAC.ordinal()) {
                    alarm_msg = "***** Vacuum alarm received! Please rectify and reset manually!";
                    if (vacDevc != null) {
                        vacDevc.setState(TSState.vacstates.TRIPPED.ordinal());
                    }
                    state |= TSState.VACDEVC_TRIPPED;
                    state &= ~TSState.READY;
                }
                if (parm == TSConfig.EVENT_ID.PD.ordinal() //&& (state & TSState.EXPERT) == 0
                        ) {
                    alarm_msg = "***** PD over VOLTAGE or CURRENT alarm received! For safety, the photdiode device voltage is being set OFF! ";
                    if (pdDevc != null) {
                        alarm_name += " I= " + pdDevc.readCurrent();
                        pdDevc.setAbort(true);
                        pdDevc.setState(TSState.pwrstates.TRIPPED.ordinal());
                        sleep(5.0);
                        pdDevc.setAbort(false);
                    }
                    state |= TSState.PWRDEVC_TRIPPED;
                    state &= ~TSState.READY;
                }
                break;

            case Alarm.EVENT_RESET:
                if (parm == TSConfig.EVENT_ID.BIAS.ordinal()) {
                    alarm_name += " V= " + biasDevc.readVoltage() + ", I= " + biasDevc.readCurrent();
                    int ikstate = 0;
                    if (biasDevc != null) {
                        ikstate = biasDevc.getState();
                    }
                    if (ikstate == TSState.pwrstates.TRIPPED.ordinal()) {
                        LOGGER.error("already tripped and a manual RESET is required.");
                    } else {
                        ikstate = TSState.pwrstates.OK.ordinal();
                        if (biasDevc != null) {
                            biasDevc.setState(ikstate);
                        }
                        LOGGER.info("BIAS device status set to " + TSState.pwrstates.values()[ikstate]);
                    }
                }
                if (parm == TSConfig.EVENT_ID.CRYO.ordinal()) {
                    cryoDevc.setState(TSState.cryostates.OK.ordinal());
                }
                if (parm == TSConfig.EVENT_ID.VAC.ordinal()) {
                    if (vacDevc != null) {
                        vacDevc.setState(TSState.vacstates.OK.ordinal());
                    }
                }
                if (parm == TSConfig.EVENT_ID.PD.ordinal()) {
                    alarm_name += " I= " + pdDevc.readCurrent();
                    int ikstate = 0;
                    if (pdDevc != null) {
                        ikstate = pdDevc.getState();
                    }
                    if (ikstate == TSState.pwrstates.TRIPPED.ordinal()) {
                        LOGGER.error("already tripped and a manual RESET is required.");
                    } else {
                        ikstate = TSState.pwrstates.OK.ordinal();
                        if (pdDevc != null) {
                            pdDevc.setState(ikstate);
                        }
                        LOGGER.info("PD device status set to " + TSState.pwrstates.values()[ikstate]);
                    }
                }

                //FIXME! This will clear all alerts, even the ones raised
                //by other components.
                alertService.clearAllAlerts();

                break;

            default:
                notify = false;

        }
        long tmillis = System.currentTimeMillis();
        if (notify && (tmillis - lastWarningTMillis) > warningInterval) {
            this.raiseTSAlert(alarm_name, AlertState.ALARM);
            lastWarningTMillis = tmillis;

            LOGGER.error(alarm_msg);
            lastWarning += alarm_name;
            this.lastWarning = alarm_name;
            this.publishState();
            soundAlarm();
        }
        return true;
    }

    /**
     * Sound an alarm.
     */
    @Command(name = "soundAlarm")
    public void soundAlarm() {
        Runtime r = Runtime.getRuntime();
        try {
            LOGGER.error("Sending warning sound as requested by " + name);
            Process p = r.exec("./make-warning-sound");
        } catch (IOException ex) {
            LOGGER.error("Failed to exec shell command to send warning sound " + ex);
        }
    }

    /**
     * Sound a bleep.
     */
    @Command(name = "soundBleep")
    public void soundBleep() {
        Runtime r = Runtime.getRuntime();
        try {
            LOGGER.severe("Sending bleep sound as requested by " + name);
            Process p = r.exec("./make-bleep-sound");
        } catch (IOException ex) {
            LOGGER.severe("Failed to exec shell command to send bleep sound " + ex);
        }
    }

    /**
     * Reset trips.
     */
    @Command(name = "resetTrip", description = "reset trips etc...")
    public void resetTrip() {
        if ((state & TSState.PWRDEVC_TRIPPED) != 0) {
            biasDevc.setState(TSState.pwrstates.OFF.ordinal());
            state &= ~TSState.PWRDEVC_TRIPPED;
            if ((state & TSState.CRYODEVC_TRIPPED) == 0) {
                state |= TSState.READY;
            }
        }
        if ((state & TSState.CRYODEVC_TRIPPED) != 0) {
            cryoDevc.setState(TSState.cryostates.OFF.ordinal());
            state &= ~TSState.CRYODEVC_TRIPPED;
            if ((state & TSState.PWRDEVC_TRIPPED) == 0) {
                state |= TSState.READY;
            }
        }
        // will add in a trip bit just for the PD
        if ((state & TSState.PWRDEVC_TRIPPED) != 0) {
            pdDevc.setState(TSState.pwrstates.OFF.ordinal());
            state &= ~TSState.PWRDEVC_TRIPPED;
            if ((state & TSState.CRYODEVC_TRIPPED) == 0) {
                state |= TSState.READY;
            }
        }
    }

    /**
     * startSubSys: put TS sub system into the ready state
     *
     * @throws Exception
     */
    @Command(type = CommandType.ACTION, description = "Start the TS sub system in the ready state")
    public void startSubSys() throws Exception {
        state |= TSState.STARTUP;
        LOGGER.info("Starting SubSystem.");

        String param = null;

        ConfigurationInfo configInfo = subsys.getConfigurationService().getConfigurationInfo();
        tscc.initCatcher(configInfo);

        LOGGER.info("====== initializing any BIAS device ====== ");
        if (biasDevc == null) {
            LOGGER.info("Bias device not defined");
        } else {
            biasDevc.setCurrentLimit(0.0025); // set max current to 2.5 mA
            for (TSConfig.operating_states wstate : TSConfig.operating_states.values()) {
                int iwstate = wstate.ordinal();
                teststandc.setCfgState(iwstate);
                biasDevc.setRunBias(teststandc.getBiasVAcq(), iwstate);
            }
            biasDevc.setCfg(teststandc);
            state |= TSState.PWRDEVC_ON;
        }
// initialize PD device
        LOGGER.info("====== initializing any PhotoDiode device ====== ");
        if (pdDevc == null) {
            LOGGER.warning("PhotoDiode device not defined");
        } else {
            for (TSConfig.operating_states wstate : TSConfig.operating_states.values()) {
                int iwstate = wstate.ordinal();
                teststandc.setCfgState(iwstate);
            }
            pdDevc.setCfg(teststandc);
            state |= TSState.PDDEVC_ON;
            pdDevc.softReset();
        }

// initialize LED device
        LOGGER.info("====== initializing any LED device ====== ");
        if (ledDevc == null) {
            LOGGER.warning("LED device not defined");
        } else {
            for (TSConfig.operating_states wstate : TSConfig.operating_states.values()) {
                int iwstate = wstate.ordinal();
                teststandc.setCfgState(iwstate);
            }
            state |= TSState.LEDDEVC_ON;
        }
// initialize PDU device
        LOGGER.info("====== initializing any PDU device ====== ");
        if (pduDevc == null) {
            LOGGER.warning("PDU device not defined");
        } else {
            for (TSConfig.operating_states wstate : TSConfig.operating_states.values()) {
                int iwstate = wstate.ordinal();
                teststandc.setCfgState(iwstate);
            }
        }
// -----
        // cryogenic device for monitoring the temperature
        LOGGER.info("====== initializing any CRYO control device ====== ");
        if (cryoDevc == null) {
            LOGGER.warning("Cryo device not defined");
        } else {
            for (TSConfig.operating_states wstate : TSConfig.operating_states.values()) {
                int iwstate = wstate.ordinal();
                teststandc.setCfgState(iwstate);
                cryoDevc.setRunTemp(teststandc.getCryoTAcq(), iwstate);
                try {
                    for (int i = 1; i <= 4; i++) {
                        cryoDevc.setMaxSetPoint(i, 30.);
                    }
                } catch (DriverException e) {
                    LOGGER.error("Failed attempt to set the cryo device's Max Set Point!");
                }
                try {
                    cryoDevc.setType(cryoDevc.getCurrent_channel(), "PTC100");
                } catch (DriverException e) {
                    LOGGER.error("Failed attempt to set the cryo device' calibration curve type!");
                }
            }
            state |= TSState.CRYODEVC_ON;
        }
// ----
        LOGGER.info("====== initializing any Vacuum Gauge device ====== ");
        if (vacDevc == null) {
            LOGGER.warning("VacuumGauge device not defined");
        } else {
            for (TSConfig.operating_states wstate : TSConfig.operating_states.values()) {
                int iwstate = wstate.ordinal();
                teststandc.setCfgState(iwstate);
                vacDevc.setRunVac(teststandc.getVacPAcq(), iwstate);
            }
            state |= TSState.VACDEVC_ON;
        }
// ----
        LOGGER.info("====== initializing any VQM device ====== ");
//        vqmDevc = (VQMDevice) mon.getDevice("VQMonitor");
        if (vqmDevc == null) {
            LOGGER.info("Vacuum Quality Monitor device not defined");
        } else {
            for (TSConfig.operating_states wstate : TSConfig.operating_states.values()) {
                int iwstate = wstate.ordinal();
                teststandc.setCfgState(iwstate);
                vqmDevc.setRunVac(teststandc.getVacPAcq(), iwstate);
            }
            state |= TSState.VQMDEVC_ON;
        }
// ----
        LOGGER.info("====== initializing any LAMP device ====== ");
        if (lmpDevc == null) {
            LOGGER.warning("Lamp device not defined");
        } else {
            lmpDevc.setType(teststandc.getSrcType());
            state |= TSState.LAMPDEVC_ON;
        }
// ----
        LOGGER.info("====== initializing any MonoChromator device ====== ");
        if (monoDevc == null) {
            LOGGER.warning("Monochromator device not defined");
        } else {
            for (TSConfig.operating_states wstate : TSConfig.operating_states.values()) {
                int iwstate = wstate.ordinal();
                teststandc.setCfgState(iwstate);
                LOGGER.debug("setting mono WL for state" + wstate.toString());
                monoDevc.setRunWave(teststandc.getLambdaAcq());
            }
            LOGGER.debug("setting mono filter labels");
            monoDevc.setFilterLabels(teststandc.getFilters());
            LOGGER.debug("setting mono filter validity ranges");
            monoDevc.setFilterEdges(teststandc.getFilter_edges());
            state |= TSState.MONODEVC_ON;
        }
// ----
        LOGGER.info("====== initializing any XED serial device ====== ");
        if (fe55Devc == null) {
            LOGGER.warning("Fe55 device not defined");
        } else {
            fe55Devc.retractFe55();
        }
// ----
        if (shDevc == null) {
            LOGGER.warning("Shutter device not defined");
        }
// ----
        if (envDevc == null) {
            LOGGER.warning("Enviro device not defined");
        } else {
            state |= TSState.ENVDEVC_ON;
        }
        LOGGER.info("====== Completed device initializations ====== ");

        /*
         **  Initialize the hardware
         */
        // Make sure the CCD backplane bias voltage is off
        LOGGER.info("Initializing bias device configuration.");
        if (biasDevc != null) {
            biasDevc.setCurrentRange(1.0e-3);
            biasDevc.setVoltageRange(100.0);
            //biasDevc.setBuffSize(3000);
            pdDevc.setCurrentRange(1.0e-3);
            pdDevc.setBuffSize(3000);
            publishState();

        }

        if (cryoDevc != null) {
            cryoDevc.setUnit(cryoDevc.getCurrent_channel(), "C");
            publishState();
        }

        state |= TSState.READY;

        publishState();
    }

    /**
     * Tells if the subsystem is ready for acquisition go step.
     * 
     * @throws Exception
     */
    @Command(name = "isTestStandReady", description = "returns 1 if the teststand subsystem is ready for acquisition")
    public int isTestStandReady() throws Exception {
        final double HIGH_PRES_LIMIT = 1.e-3;
        final double HIGH_TEMP_LIMIT = 35.;
        final double LOW_TEMP_LIMIT = -130.;
        final double MAX_TEMP_CHANGE = 2.5;
        final double MAX_PARTICLE_COUNT = 4.0;
        final double BSSILIM = 0.001;

        double vac = 0.0;
        double temp = 0.0;
        double tempstddev = 999.0;
        double tempchange = 0.;
        double lmppwr = 0.0;
        double partcnt = 0.0;
        double bsscur = 0.0;
        double bss = 0.0;
        if (cryoDevc != null) {
            temp = cryoDevc.getTemp(cryoDevc.getCurrent_channel());
            tempstddev = cryoDevc.getTempStdDev();
            tempchange = cryoDevc.getTempChange();
            if (!cryoDevc.isInControl()) { // make sure cryo device is in control mode
                LOGGER.info("Found cryo device not in control mode. Setting control mode ON.");
                cryoDevc.setToControl();
            }
        }
        if (vacDevc != null) {
            vac = vacDevc.readPressure();
        } else if (vqmDevc != null) {
            vac = vqmDevc.getLastPres();
        }
        if (lmpDevc != null) {
            lmppwr = lmpDevc.getLampPower();
        }
        if (envDevc != null) {
            partcnt = envDevc.getPartCnt();
        }
        if (biasDevc != null) {
            bsscur = biasDevc.getLastCurrent();
            bss = biasDevc.getLastVoltage();
            if (biasDevc.showOutput() == false) {
                this.raiseTSAlert("THE BIAS SUPPLY OUTPUT APPEARS TO BE DISABLED!!!! PLEASE CHECK.", AlertState.NOMINAL);
            }
        }

        boolean systemOK = true;
        boolean isAlarm = false;

        LOGGER.debug("lamp power is " + lmppwr + " minimum is " + teststandc.getMinLmpPwr()
                + " | CCD temperature(T=" + temp + " or stddev=" + tempstddev + "normal range is " + teststandc.getCryoTAcq() + "+/-" + teststandc.getCryoTAcqTol()
                + " | Dewar pressure, P = " + vac + " should be above " + teststandc.getVacPAcq());

        String warning = "Warning: " + configName + " - " + new SimpleDateFormat("yyyyMMddHHmmss").format(new Date(
                System.currentTimeMillis())) + " - ";
        state &= ~(TSState.CRYO_DEV_WARNING | TSState.VAC_DEV_WARNING);
        // check for dangerous combination of conditions
        if (bsscur > BSSILIM && Math.abs(bss) > 0.0) { // check for overcurrent on BSS line
            if (biasDevc.readCurrent() > BSSILIM) { // still in overcurrent?
                systemOK = false;
                warning = warning.replace("Warning", "ALARM");
                warning += "overcurrent on the BSS line!\n";
                isAlarm = true;
                this.raiseTSAlert(warning, AlertState.ALARM);
                biasDevc.setOutput(0);
                soundAlarm();
                LOGGER.error("Identified dangerous conditions for the sensor. BSS output disabled!");
            }
        }
        // check for dangerous combination of conditions
        if (vac > HIGH_PRES_LIMIT && last_vac > HIGH_PRES_LIMIT && temp < 5.0 && vac < 2000.) {
            systemOK = false;
            warning = warning.replace("Warning", "ALARM");
            warning += "DANGEROUS T/P CONDITIONS! T=" + temp + " P=" + vac + "\n";
            isAlarm = true;
            this.raiseTSAlert(warning, AlertState.ALARM);
            soundAlarm();
            // raise temperature to protect sensor
            LOGGER.error("Identified dangerous conditions for the sensor.");
//            LOGGER.error("The temperature is being raised to avoid condensation.");
//            cryoDevc.rampTemp(600., 20., 10);
        }
        if (partcnt > MAX_PARTICLE_COUNT) {
            systemOK = false;
            warning += "HIGH PARTICLE COUNT! PCNT=" + partcnt + "\n";
            isAlarm = true;
            this.raiseTSAlert(warning, AlertState.ALARM);
            soundAlarm();
            LOGGER.error("ALERT!!! HIGH PARTICLE COUNT DETECTED.");
        }
        if ((tempchange > MAX_TEMP_CHANGE || temp > HIGH_TEMP_LIMIT || temp < LOW_TEMP_LIMIT) && temp > -274.) {
            systemOK = false;
            warning += "DANGEROUS TEMP RISE T=" + temp + " delta_T=" + tempchange + "\n";
            this.raiseTSAlert(warning, AlertState.ALARM);
            isAlarm = true;
            soundAlarm();
            // raise temperature to protect sensor
            LOGGER.error("Identified dangerous temperature or temperature rise.");
            if (temp > LOW_TEMP_LIMIT && temp < HIGH_TEMP_LIMIT) {
                LOGGER.error("HALTING temperature change.");
                cryoDevc.rampTemp(60., temp, 10);
            }
            if (temp > HIGH_TEMP_LIMIT) {
                LOGGER.error("Setting controller to achieve room temperature AND  POWERING OFF THE CCD!!!!!");
                warning += "POWERING OFF THE CCD";
                cryoDevc.rampTemp(60., 25., 10);
                // warn other subsystems of the situation
                KeyValueData kd = new KeyValueData("TS_Hazard", "SensorHazard: Environment concern.");

                subsys.publishSubsystemDataOnStatusBus(kd);
            }
        }
        if ((temp > (teststandc.getCryoTAcq() + teststandc.getCryoTAcqTol())) || tempstddev > teststandc.getCryoTAcqTol()) {
            systemOK = false;
            state |= TSState.CRYO_DEV_WARNING;
            warning += "CCD temperature(T=" + temp + " or stddev=" + tempstddev + " not in range " + teststandc.getCryoTAcq() + "+/-" + teststandc.getCryoTAcqTol() + "\n";
            this.raiseTSAlert(warning, AlertState.WARNING);
        }
        if (vac > teststandc.getVacPAcq()) {
            systemOK = false;
            state |= TSState.VAC_DEV_WARNING;
            warning += "Dewar pressure too high, P = " + vac + "\n";
        }
        if (lmppwr < teststandc.getMinLmpPwr()) {
//            systemOK = false;
            warning += "The lamp doesn't appear to be at the requisite power:\n"
                    + "lamp power is " + lmppwr + ", minimum is " + teststandc.getMinLmpPwr() + "\n";
        }
        last_vac = vac;
        if (systemOK != last_systemOK) {
            nwarn = 0; // reset count if status has changed
        }
        last_systemOK = systemOK;
        long tmillis = System.currentTimeMillis();
//        if (systemOK) {
//            lastWarning = "";
//        } else 
        if (!systemOK && (tmillis - lastWarningTMillis) > warningInterval) {
            lastWarningTMillis = tmillis;
//            toolkit.beep();
            LOGGER.error(warning);
            if (isAlarm) {
                warning = warning.replace("Warning", "ALARM");
                soundAlarm();
                this.publishState();
                System.out.println("About to distribute alarm message.");
                try {
                    String ccsmail = System.getenv("CCSmail");
                    if (ccsmail == null) {
                        ccsmail = "ssh ccdtest@astroracf mail -s "; // default
                    }
                    String ccsmaillist = System.getenv("CCSmaillist");
                    if (ccsmaillist == null) {
                        if (this.configName.toLowerCase().indexOf("bnl") >= 0 && this.configName.toLowerCase().indexOf("sim") < 0) {
//                ccsmaillist = "homer@slac.stanford.edu,homerneal@yahoo.fr,2034356858@tmomail.net,6312354433@vtext.com,mrumore@bnl.gov"; //default
//                            ccsmaillist = "homer@slac.stanford.edu,homerneal@yahoo.fr,2034356858@tmomail.net,mrumore@bnl.gov,kotov@bnl.gov,jhaupt@bnl.gov"; //default
                            ccsmaillist = "homer@slac.stanford.edu,homerneal@yahoo.fr,2034356858@tmomail.net"; //default
                        } else {
                            ccsmaillist = "homer@slac.stanford.edu,homerneal@yahoo.fr,2034356858@tmomail.net";
                        }
                    }
                    String mailalarm = ccsmail + " \"" + warning + "\" " + ccsmaillist;
                    LOGGER.error("Sending alarm message: " + mailalarm);
                    Runtime r = Runtime.getRuntime();

                    Process p = r.exec(mailalarm);
                    sleep(5.0);
                    if (p.isAlive()) {
                        p.destroy();
                    }

                } catch (IOException ex) {
                    LOGGER.error("Failed to exec shell command to send alarm message " + ex);
                }

            }
            lastWarning += warning;
        }
        // if the system is OK then clear the Alerts
        if (systemOK && (alertService.getRaisedAlertSummary().getAlertState() != AlertState.NOMINAL)) {
            //FIXME: Why clearing all alerts?
            alertService.clearAllAlerts();
        }
        // override determination for test state so that tests may proceed
        if ((state & TSState.TEST) != 0) {
            systemOK = true;
        }
        return (systemOK ? 1 : 0);
    }

    /**
     * set configured bias target
     *
     * @param Vacq
     * @param cfgstate
     * @throws Exception
     */
    @Command(description = "set configured bias target")
    public void setCfgBiasVAcq(double Vacq, int cfgstate) throws Exception {
        teststandc.setBiasVAcq(Vacq, cfgstate);
        biasDevc.setRunBias(Vacq, cfgstate);
    }

    /**
     * set configured cryo T target
     *
     * @param Tacq
     * @param cfgstate
     * @throws java.lang.Exception
     */
    @Command(description = "set configured cryo T target")
    public void setCfgCryoTAcq(double Tacq, int cfgstate) throws Exception {
        teststandc.setCryoTAcq(Tacq, cfgstate);
        cryoDevc.setRunTemp(Tacq, cfgstate);
    }

    /**
     * Standard voltage ramp
     *
     * @throws Exception
     */
    void doStandardVRamp() throws Exception {
        if (biasDevc != null) {

            double actual_bias = biasDevc.readVoltage();
            double target_bias = teststandc.getBiasVAcq();
            double safe_rate = 50. / 60.; // no faster than 50V per minute
            double duration = Math.abs(target_bias - actual_bias) / safe_rate;

            if (biasDevc.showOutput() == false) {
                LOGGER.error("***************************************************************");
                LOGGER.error("***************************************************************");
                LOGGER.error("***************************************************************");
                LOGGER.error("THE BIAS SUPPLY OUTPUT APPEARS TO BE DISABLED!!!! PLEASE CHECK.");
                LOGGER.error("THE BIAS SUPPLY OUTPUT APPEARS TO BE DISABLED!!!! PLEASE CHECK.");
                LOGGER.error("THE BIAS SUPPLY OUTPUT APPEARS TO BE DISABLED!!!! PLEASE CHECK.");
                LOGGER.error("***************************************************************");
                LOGGER.error("***************************************************************");
                LOGGER.error("***************************************************************");
                this.raiseTSAlert("THE BIAS SUPPLY OUTPUT APPEARS TO BE DISABLED!!!! PLEASE CHECK.", AlertState.ALARM);
            }

            System.out.println("The current bias voltage is " + actual_bias);
            System.out.println("The target bias is " + target_bias);
            System.out.println("The duration chosen for a safe ramp is " + duration);
            if (duration > 0.1) { // not worth ramping if the duration needed is less that a tenth of a second
//                biasDevc.setOutput(1);
                biasDevc.rampVolts(duration, target_bias, 10);
            } else { // just do it
                biasDevc.setVoltage(target_bias);
            }
        }
    }

    /**
     * Put system in mode for acquisition
     *
     * @return
     * @throws Exception
     */
    @Command(name = "goTestStand", description = "put in mode for acquisition")
    public int goTestStand() throws Exception {
        LOGGER.error("We are ready to go and BIAS ramping can now procede: state = " + state);
        switch (state & (TSState.READY | TSState.ACQ1 | TSState.WARM | TSState.TEST | TSState.EXPERT)) {
            case TSState.READY:
                LOGGER.error("Ramping BIAS voltage to that set for state READY, V=" + teststandc.getBiasVAcq());
                doStandardVRamp();
                break;
            case TSState.ACQ1:
                LOGGER.error("Ramping BIAS voltage to that set for state ACQ1, V=" + teststandc.getBiasVAcq());
                doStandardVRamp();
                break;
            case TSState.WARM:
                LOGGER.error("Ramping BIAS voltage to that set for state WARM");
                doStandardVRamp();
                break;
            case TSState.TEST:
                LOGGER.error("Ramping BIAS voltage to that set for state TEST");
                doStandardVRamp();
                break;
            case TSState.EXPERT:
                LOGGER.error("Ramping BIAS voltage to that set for state EXPERT");
                doStandardVRamp();
                break;
            default:
                LOGGER.error("Trying to go to acquisition go state without being in acquisition mode!");
        }
        return (1);
    }

    /**
     * Set the subsystem to the idle state
     * 
     * @throws Exception
     */
    @Command(description = "set flags corresponding to the specified operating state")
    void setStateFlags(TSConfig.operating_states ostate) throws Exception {
        teststandc.setCfgState(ostate.ordinal());

        int mask = TSState.EXPERT | TSState.READY | TSState.ACQ1 | TSState.WARM | TSState.TEST | TSState.IDLE;
        state &= ~mask; // wipe it

        switch (ostate) {
            case EXPERT:
                state |= TSState.EXPERT;
                break;
            case READY:
                state |= TSState.READY;
                break;
            case ACQ1:
                state |= TSState.ACQ1;
                break;
            case WARM:
                state |= TSState.WARM;
                break;
            case TEST:
                state |= TSState.TEST;
                break;
            case IDLE:
                state |= TSState.IDLE;
                break;
            default:
                throw new Exception("INVALID ACQUISITION STATE!");
        }

    }

    /**
     * Set the subsystem to the acquisition 1 state
     * 
     * @throws Exception
     */
    void setStandardState() throws Exception {

        // set initial wavelength for the monochromator
        if (monoDevc != null) {
            monoDevc.setWave(teststandc.getLambdaAcq());
        }
        // check the vacuum
        double vac = this.getPressure();
        // set cryo-temp
        if (cryoDevc != null) {
            double target_temp = teststandc.getCryoTAcq();
            String chan = cryoDevc.getCurrent_channel();
            int loop = 1;
            double actual_temp = cryoDevc.getTemp(chan);
            if (actual_temp < -200. || target_temp < -170. || target_temp > 25.) {
                LOGGER.error("BOGUS current temp. Aborting cryo temp ramp request!!!");
            } else if (((vacDevc == null && vqmDevc == null) || vac > teststandc.getVacPAcq()) && target_temp < 0.) {
                LOGGER.error("THE VACUUM IS EITHER NOT BEING MONITORED OR THE VACUUM LEVEL IS TOO HIGH\nWE CAN NOT RAMP DOWN THE TEMPERATURE IN THIS CONDITION!!!!");
            } else if (Math.abs(target_temp - actual_temp) > teststandc.getCryoTAcqTol()) { // ramp only if out of tolerance
                double safe_rate = 60. / 3600.; // no faster than 60C per hour
                double duration = Math.abs(target_temp - actual_temp) / safe_rate;
                System.out.println("The current temperature is " + actual_temp + "C");
                System.out.println("The target temperature is " + target_temp + "C");
                System.out.println("The duration chosen for a safe ramp is " + duration + "s");
                int nsteps = (int) (2.0 * duration / 60.);
                if (nsteps == 0) {
                    nsteps = 1;
                }
                cryoDevc.rampTemp(duration, target_temp, nsteps);

            }
        }

    }

    /**
     * return the last pressure reading from whatever gauge is active
     * 
     * @return
     */
    @Command(name = "getPressure", description = "return the last pressure reading from whatever gauge is active")
    public double getPressure() {
        // check the vacuum
        double vac = 1000.;
        if (vacDevc != null) {
            vac = vacDevc.getLastPres();
        }
        if (vqmDevc != null && (vac == 0. || vac > 900.)) {
            vac = vqmDevc.getLastPres();
        }
        return vac;
    }

    /**
     * Set the subsystem to the named state
     * 
     * @param statename
     * @return
     */
    @Command(name = "setTSStateByName", description = "set TS subsystem to state indicated by the string")
    public int setTSStateByName(String statename) {
        try {
            if (statename.toUpperCase().contains("IDLE")) {
                return (setTSIdle());
            }
            if (statename.toUpperCase().contains("READY")) {
                return (setTSReady());
            }
            if (statename.toUpperCase().contains("WARM")) {
                return (setTSWarm());
            }
            if (statename.toUpperCase().contains("ACQ1")) {
                return (setTSAcq1());
            }
            if (statename.toUpperCase().contains("TEST")) {
                return (setTSTEST());
            }
        } catch (Exception ex) {
            LOGGER.error("Failed to set the requested state: " + ex);
        }
        return (-1);
    }

    /**
     * Set the subsystem to the idle state
     * 
     * @return
     * @throws Exception
     */
    @Command(type = CommandType.QUERY, name = "setTSIdle", description = "set TS subsystem to the idle state")
    public int setTSIdle() throws Exception {
        setStateFlags(TSConfig.operating_states.IDLE);
        setStandardState();
        LOGGER.info("Set state to IDLE.");
        return (1);
    }

    /**
     * Set the subsystem to the ready state.
     * 
     * @return
     * @throws Exception
     */
    @Command(type = CommandType.QUERY, name = "setTSReady", description = "set TS subsystem to the ready state")
    public int setTSReady() throws Exception {
        setStateFlags(TSConfig.operating_states.READY);
        setStandardState();
        LOGGER.info("Set state to READY.");
        return (1);
    }

    /**
     * Set the subsystem to the acquisition 1 state
     * 
     * @return
     * @throws Exception
     */
    @Command(type = CommandType.QUERY, name = "setTSAcq1", description = "set TS subsystem to the ACQ1 state")
    public int setTSAcq1() throws Exception {
        setStateFlags(TSConfig.operating_states.ACQ1);
        setStandardState();
        LOGGER.info("Set state to ACQ1.");
        return (1);
    }

    /**
     * Set the subsystem to the acquisition 2 state.
     * 
     * @return
     * @throws Exception
     */
    @Command(type = CommandType.QUERY, name = "setTSWarm", description = "set TS subsystem to the WARM state")
    public int setTSWarm() throws Exception {
        setStateFlags(TSConfig.operating_states.WARM);
        setStandardState();
        LOGGER.info("Set state to WARM.");
        return (1);
    }

    /**
     * Set the subsystem to the TEST state.
     * 
     * @return
     * @throws Exception
     */
    @Command(type = CommandType.QUERY, name = "setTSTEST", description = "set TS subsystem to the TEST state")
    public int setTSTEST() throws Exception {
        setStateFlags(TSConfig.operating_states.TEST);
        setStandardState();
        LOGGER.info("Set state to TEST.");
        return (1);
    }

    /**
     * Set the subsystem to the EXPERT state
     * 
     * @return
     * @throws Exception
     */
    @Command(type = CommandType.QUERY, name = "setTSEXPERT", description = "EXPERTS ONLY!!")
    public int setTSEXPERT() throws Exception {
        setStateFlags(TSConfig.operating_states.EXPERT);
        setStandardState();
        LOGGER.info("Set state to EXPERT.");
        return (1);
    }

    /**
     * Perform setup actions needed before powering on the CCD
     * 
     * @return
     */
    @Command(type = CommandType.QUERY, name = "eoSetupPreCfg", description = "prepare for eo acquisition")
    public int eoSetupPreCfg() {
        return (eoSetupPreCfg("setTSReady"));
    }

    /**
     * Perform setup actions needed before powering on the CCD
     * 
     * @param state1
     * @return
     */
    @Command(type = CommandType.QUERY, name = "eoSetupPreCfg", description = "prepare for eo acquisition")
    public int eoSetupPreCfg(@Argument(name = "state1", description = "IDLE,WARM,READY,TEST,ACQ1") String state1) {

// Initialization
        LOGGER.info("doing initialization");

        pdDevc.softReset();

// move TS to ready state
        LOGGER.info("setting teststand to state: " + state1);
        this.setTSStateByName(state1);
        try {
            goTestStand();
        } catch (Exception ex) {
            LOGGER.error("Failed to finalize acquisition preparation: " + ex);
        }
        return (1);
    }

    /**
     * Perform setup actions needed after powering on the CCD
     * 
     * @param vac_outlet
     * @return
     * @throws Exception
     */
    @Command(type = CommandType.QUERY, name = "eoSetupPostCfg", description = "prepare for eo acquisition")
    public int eoSetupPostCfg(@Argument(name = "vac_outlet", description = "PDU outlet number for the vacuum gauge") int vac_outlet) throws Exception {
        return (eoSetupPostCfg(vac_outlet, "setTSTEST"));
    }

    /**
     * Perform setup actions needed after powering on the CCD
     * 
     * @param vac_outlet
     * @param state2
     * @return
     * @throws Exception
     */
    @Command(type = CommandType.QUERY, name = "eoSetupPostCfg", description = "prepare for eo acquisition")
    public int eoSetupPostCfg(@Argument(name = "vac_outlet", description = "PDU outlet number for the vacuum gauge") int vac_outlet,
            @Argument(name = "state2", description = "IDLE,WARM,READY,TEST,ACQ1") String state2) throws Exception {
// move to TS acquisition state
        LOGGER.info("setting acquisition state");

        // get the glowing vacuum gauge off
        pduDevc.setOutletState(vac_outlet, false);

        this.setTSStateByName(state2);

//check state of ts devices
        LOGGER.info("wait for ts state to become ready");
        int tsstate = 0;
        long starttim = periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis();
        while (true) {
            LOGGER.info("checking for test stand to be ready for acq");
            try {
                tsstate = isTestStandReady();
            } catch (Exception ex) {
                LOGGER.error("Failed to accomplish readiness check!");
            }
// the following line is just for test situations so that there would be no waiting                                         
//        tsstate=1;
            if ((periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis() - starttim) > 10800) {
                LOGGER.info("Something is wrong ... we will never make it to a runnable state");
                throw new Exception("Not succeeding in reaching target conditions! ABORTING!");
            }
            if (tsstate != 0) {
                LOGGER.info("operating conditions achieved");
                break;
            }
            sleep(5.0);
        }
//put in acquisition state
        LOGGER.info("We are ready to go! Ramping the BP bias voltage now.");
        try {
            goTestStand();
        } catch (Exception ex) {
            LOGGER.error("Failed to finalize acquisition preparation: " + ex);
        }
// it takes time for the glow to fade
        sleep(5.0);

        return (1);
    }

    /**
     * Returns the configured set point for the acquisition CCD bias voltage
     *
     * @return
     */
    @Command(name = "showBiasVAcq", description = "returns the acquisition bias voltage set point")
    public double showBiasVAcq() {
        return (teststandc.getBiasVAcq());
    }

    /**
     * Returns the required slit width for a given wavelength
     * 
     * @param wl
     * @return
     */
    @Command(name = "getReqSlitWidth", description = "returns the req. slit width for a given wl")
    public int getReqSlitWidth(double wl
    ) {
        return (teststandf.getRequiredSlitWidth(wl));
    }

    /**
     * set the source file path for the slit width data
     * 
     * @param fl
     */
    @Command(name = "setSlitWidthFile", description = "sets the slit width data file")
    public void setSlitWidthFile(String fl
    ) {
        teststandf.setTSFluxCalibFile(fl);
    }

    /**
     * Reconnects to the VQM device
     * 
     * @throws DriverException 
     */
    @Command(name = "reconnectVQM", description = "reconnect to the VQM port")
    public void reconnectVQM() throws DriverException {
        vqmDevc.reconnect();
    }

    /**
     * stopSubSys: put TS sub system into the ready state.
     */
    @Command(name = "stopsubsys", type = CommandType.ACTION,
            description = "Stop and set the TS sub system into the ready state")
    public void stopSubSys() {
//        state |= TSState.SHUTDOWN;
        LOGGER.error("Stop Subsystem message received.");
        this.raiseTSAlert("STOP SUBSYSTEM CALLED", AlertState.ALARM);

        state &= ~TSState.READY;
        publishState();

        if (biasDevc != null) {
            LOGGER.info("Ramping down bias voltage.");
            rampBiasVolts(3.0, 0.0, 10);
            setBiasOutput(0);
            LOGGER.error("BIAS output disabled!");
            this.raiseTSAlert("THE BIAS SUPPLY OUTPUT IS DISABLED!!!!", AlertState.ALARM);
            publishState();
        }
        if (cryoDevc != null) {
            LOGGER.info("Stopping any temperature ramp.");
            cryoDevc.setState(TSState.cryostates.TRIPPED.ordinal());
        }

        state |= TSState.IDLE;
//        setTickMillis(0);
        publishState();
    }

    /**
     * Saves the configuration data.
     * 
     * @throws IOException
     */
    @Command(name = "saveconfig", type = CommandType.ACTION, description = "Saves the current configuration")
    public void saveConfiguration() throws IOException {
        try {
            subsys.getConfigurationService().saveAllChanges();
        } catch (Exception ex) {
            LOGGER.warning("Failed to save all changes " + ex);
        }
        //register(configName);
        subsys.getMonitor().clearLimitChanges();
        LOGGER.info("Saved configuration.");
    }

    /**
     * Prints the full state of test stand
     *
     * @return
     */
    @Command(name = "printfullstate", type = CommandType.QUERY, description = "Prints the full test stand state")
    public String printFullState() {
        TSState teststandState = new TSState(state, teststandc.getCfgstate(), (int)periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis());
        getDevStates(teststandState);

        StringBuilder sb = new StringBuilder();
        sb.append("TSStateStatus :  (");
        sb.append(teststandState.toString());
        sb.append(")");

        return (sb.toString());
    }

    /**
     * Gets the operating state word.
     *
     * @return The current value of the operating state
     */
    @Command(name = "getstate", type = CommandType.QUERY, description = "Returns the current value of the TS state")
    public int getState() {
        return state;
    }

    /**
     * Gets TS configuration.
     * 
     * @return 
     */
    public TSConfig getTeststandc() {
        return teststandc;
    }

    /**
     * Sets the TS state word.
     * 
     * @param state
     */
    @Command(name = "setstate", type = CommandType.QUERY,
            description = "sets the current value of the TS state")
    public void setState(int state) {
        this.state = state;
        publishState();
    }

    /**
     * Sleep - what a waste
     * 
     * @param secs
     */
    private void sleep(double secs) {
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            LOGGER.error("Rude awakening!" + ex);
        }
    }

    /**
     * Publishes the state of the test stand.
     * 
     * This is intended to be called whenever any element of the state is changed.
     */
    @Command(name = "publishState", type = CommandType.QUERY,
            description = "publishes the TS state")
    public void publishState() {
        LOGGER.debug("Entering pubishState");

        TSState teststandState = new TSState(state, teststandc.getCfgstate(), (int)periodicTaskService.getPeriodicTaskPeriod(BROADCAST_TASK).toMillis());
        getDevStates(teststandState);

        KeyValueData kd = new KeyValueData(TSState.KEY, teststandState);
        subsys.publishSubsystemDataOnStatusBus(kd);
//        getSubsystem().publishStatus(TSState.KEY, teststandState);
//        publish(TSState.KEY, teststandState);
        LOGGER.debug("published state");
        lastWarning = ""; // now that it has been published, we can clear it
    }

    /**
     * gets the state of all TS devices
     *
     * @param tst
     */
    void getDevStates(TSState tst
    ) {
        try {
            if (biasDevc != null) {
                tst.setVolts(biasDevc.getLastVoltage());
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve BIAS device voltage status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            if (biasDevc != null) {
                tst.setBIAS_curr(biasDevc.getLastCurrent());
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve BIAS device current status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            if (biasDevc != null) {
                tst.setBIASon(biasDevc.showOutput() ? 1 : 0);
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve BIAS device output status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            if (pdDevc != null) {
                tst.setCurr(pdDevc.getLastCurrent());
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve PhotoDiode device current value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            if (cryoDevc != null) {
                tst.setTemp(cryoDevc.getTemp(cryoDevc.getCurrent_channel()));
                tst.setTemp2(cryoDevc.getTemp(cryoDevc.getCurrent_channel2()));
            }
        } catch (Exception e) {
            LOGGER.info("failed to retrieve CRYO device status value for publishing!");
//            toolkit.beep();
//            soundBleep();
        }
        try {
            if (vqmDevc != null) {
                tst.setVac2(vqmDevc.getLastPres());
            }
        } catch (Exception e) {
            LOGGER.info("failed to retrieve VQM pressure value for publishing!");
//            toolkit.beep();
//            soundBleep();
        }
        try {
            if (vacDevc != null) {
                tst.setVac(vacDevc.getLastPres());
            }
        } catch (Exception e) {
            LOGGER.info("failed to retrieve VAC device status value for publishing!");
//            toolkit.beep();
//            soundBleep();
        }
        try {
            if (lmpDevc != null) {
                tst.setLmpPwr(lmpDevc.getLampPower());
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve LAMP I device status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            if (monoDevc != null) {
                tst.setWl(monoDevc.getLastwl());
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve MONO WL device status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            if (monoDevc != null) {
                tst.setSw1(monoDevc.getLastSW1());
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve MONO SW1 device status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            if (monoDevc != null) {
                tst.setSw2(monoDevc.getLastSW2());
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve MONO SW2 device status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        int fltpos = 0;
        try {
            if (monoDevc != null) {
                fltpos = monoDevc.getLastfltpos();
//                tst.setFltpos(fltpos);
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve MONO FLTR device status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            if (fe55Devc != null) {
                tst.setXedpos(fe55Devc.getExtend_retract());
            }
        } catch (Exception e) {
            LOGGER.info("failed to retrieve Fe55 device status value for publishing!");
//            toolkit.beep();
//            soundBleep();
        }
        try {
            if (monoDevc != null) {
                tst.setShutpos(monoDevc.getLastshut());
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve MONO SHUTTER device status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            tst.setSrctype(teststandc.getSrcType());
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve LAMP TYPE device status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            tst.setMonotype(teststandc.getMonoType());
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve MONO TYPE device status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            tst.setPDcnt(teststandc.getPDcnt());
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve PD DEVICE count status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            tst.setPDnplc(teststandc.getPDnplc());
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve PD device nplc status value for publishing!");
//            toolkit.beep(); soundBleep();
        }
        try {
            tst.setPDtype(teststandc.getPDType());
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve PD device type status value for publishing!");
//            toolkit.beep();
//            soundBleep();
        }
        try {
            tst.setBIAStype(teststandc.getBIASType());
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve PD DEVICE device status value for publishing!");
//            toolkit.beep();
//            soundBleep();
        }
        try {
            if (monoDevc != null) {
                tst.setFilter(monoDevc.getFilterLabel(fltpos));
            }
        } catch (Exception e) {
            LOGGER.debug("failed to retrieve FILTER LABEL device status value for publishing!");
 //           toolkit.beep();
//            soundBleep();
        }

        try {
            if (lastWarning.length() > 0) {
                LOGGER.info("Writing warning message to DB: " + lastWarning);
                LOGGER.info("Warning message length = " + lastWarning.length());

                tst.setWarning(lastWarning.substring(0, Math.min(254, lastWarning.length())));
            } else {
                tst.setWarning("");
            }
        } catch (Exception e) {
            LOGGER.debug("failed to set warning!");
        }

    }

    /**
     * Checks whether ready check is being done.
     * 
     * @return 
     */
    @Command(name = "isDoTestStandReadyCheck", type = CommandType.QUERY,
            description = "check whether the test stand ready check is being done")
    public boolean isDoTestStandReadyCheck() {
        return doTestStandReadyCheck;
    }

    /**
     * Puts into ready check state.
     * 
     * @param doTestStandReadyCheck 
     */
    @Command(name = "setDoTestStandReadyCheck", type = CommandType.QUERY,
            description = "set the test stand ready check on(true)/off(false)")
    public void setDoTestStandReadyCheck(boolean doTestStandReadyCheck) {
        this.doTestStandReadyCheck = doTestStandReadyCheck;
    }

    /**
     * Sends a synchronous command.
     * 
     * @param name
     * @param params
     * @return 
     */
    protected Object sendSyncTSCommand(String name, Object... params) {
        try {
            return cmu.sendSynchronousCommand(new CommandRequest(teststand_dest, name, params), Duration.ofMillis(30000));
        } catch (Exception e) {
            LOGGER.warning("Unable to perform jgroup communication with destination + "
                    + teststand_dest + " - Exception " + e);
            return null;
        }
    }

    /**
     * Ramps the bias device voltage
     * 
     * @param time
     * @param value
     */
    private void rampBiasVolts(double time, double value) {
        try {
            biasDevc.rampVolts(time, value);
        } catch (DriverException e) {
            LOGGER.error("Error while ramping voltage", e);
        }
    }

    /**
     * Ramps the bias device voltage
     * 
     * @param time
     * @param value
     * @param nsteps
     */
    private void rampBiasVolts(double time, double value, int nsteps) {
        try {
            biasDevc.rampVolts(time, value, nsteps);
        } catch (DriverException e) {
            LOGGER.error("Error while ramping voltage", e);
        }
    }

    /**
     * Sets the bias output voltage on or off
     * 
     * @param on
     */
    private void setBiasOutput(int on) {
        try {
            biasDevc.setOutput(on);
        } catch (DriverException e) {
            LOGGER.error("Error while setting output voltage state", e);
        }
    }

}
