package org.lsst.ccs.subsystem.teststand;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.drivers.gpio.GPIODriver;
import org.lsst.ccs.subsystem.teststand.data.TestStandAgentProperties;
import org.lsst.ccs.subsystem.teststand.data.TSDeviceStatus;
import org.lsst.ccs.utilities.logging.Logger;

/**
 ******************************************************************************
 **
 ** Implements the BOT illumination-control subsystem
 *
 * @author  Al Eisner
 *
 * *****************************************************************************
 */
public class BOTBench implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    Map<String, ThorlabsFWDevice> filterWheelMap = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    ThorlabsSC10Device shutterDev;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
	List<Fe55Shutter> shuttersFe55 = new ArrayList<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    KeithleyDevice photoDiodeDev;

    @LookupField(strategy = LookupField.Strategy.TOP)
    Subsystem subsys;

    @LookupName
    private String name;

    private final Logger log = Logger.getLogger(getClass().getPackage().getName());

    /**
     * Post-initialization
     */
    @Override
    public void postInit() {

        /** 
         * By setting TESTSTAND_TYPE_AGENT_PROPERTY we signal to consoles 
         * that this subsystem is compatible with the teststand subsystm GUI
         */
        subsys.setAgentProperty(TestStandAgentProperties.TESTSTAND_TYPE_AGENT_PROPERTY, BOTBench.class.getCanonicalName());

    }

    /**
     *  Return list of filter wheels and their connection status
     *
     *  @return  List<TSDeviceStatus>
     */
    @Command(type=Command.CommandType.QUERY, name="listFilterWheels",
             description="list filter wheels and connection status")
    public List<TSDeviceStatus> listFilterWheels() {
        List<TSDeviceStatus> listfw = new ArrayList<>();
        Set<String> fwNames = filterWheelMap.keySet();
        for (String fwName : fwNames) {
            boolean fwOnline = filterWheelMap.get(fwName).isOnline();
            listfw.add(new TSDeviceStatus(fwName, fwOnline));
        }
	return listfw;
    }

    /**
     *  Open all Fe55 shutters
     *
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="fe55OpenShutters",
             description="open all Fe55 shutters")
    public void fe55OpenShutters() throws DriverException {
        if (shuttersFe55.isEmpty()) {
            log.error("No Fe55 shutters are configured");
            return;
        }
        Iterator<Fe55Shutter> iter = shuttersFe55.iterator();
        while (iter.hasNext()) {
	    iter.next().setShutter(true);
        }
    }

    /**
     *  Close all Fe55 shutters
     *
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="fe55CloseShutters",
             description="close all Fe55 shutters")
    public void fe55CloseShutters() throws DriverException {
        if (shuttersFe55.isEmpty()) {
            log.error("No Fe55 shutters are configured");
            return;
        }
        Iterator<Fe55Shutter> iter = shuttersFe55.iterator();
        while (iter.hasNext()) {
	    iter.next().setShutter(false);
        }
    }

      /**
     *  read PhotoDiode Current
     *
     * @return 
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="readPDCurrent",
             description="read photodiode current")
    public double readPDcurrent() throws DriverException {
        return(photoDiodeDev.readCurrent());
    }

      /**
     *  accumulate PD readings in buffer
     *
     * @param nreads
     * @param nplc
     * @return 
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="accumPDBuffer",
             description="accumulate PD readings in buffer")
    public double accumPDBuffer(int nreads, double nplc) throws DriverException {
        return(photoDiodeDev.accumBuffer(nreads,nplc));
    }

      /**
     *  is accumulation of PD readings in progress
     *
     * @return 
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="isPDAccumInProgress",
             description="tell if PD reading accumulation is in progress")
    public boolean isPDAccumInProgress() throws DriverException {
        return(photoDiodeDev.isAccumInProgress());
    }

      /**
     *  return array of PD readings from the buffer
     *
     * @param filename
     * @return 
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="readPDbuffer",
             description="return array of PD readings from the buffer")
    public double[][] readPDBuffer(String filename) throws DriverException {
        return(photoDiodeDev.readBuffer(filename));
    }

    /**
     *  reset PhotoDiode device
     *
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="resetPD",
             description="reset PD device")
    public void resetPD() throws DriverException {
        photoDiodeDev.reset();
    }

    /**
     *  clear PhotoDiode buffer
     *
     *  @throws  DriverException
     */
    @Command(type=Command.CommandType.ACTION, name="clearPDbuff",
             description="clearPDbuff")
    public void clearPDbuff() throws DriverException {
        photoDiodeDev.clearBuffer();
    }

 
    
}
