package org.lsst.ccs.subsystem.teststand;

import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.lakeshore.LS330;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 *
 * Device class for the Cryogenic Temperature Controller LakeShore 325 driver
 *
 * @author Heather Kelly / Homer Neal *
 */
public class Lakeshore325Device extends Device implements CryoDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("TEMP", Channel.TYPE_TEMP);
        typeMap.put("VOLTS", Channel.TYPE_VOLTS);
        typeMap.put("PA", Channel.TYPE_UNKNOWN);
    }

    private final static int EVENT_ID_CRYO = 0;

    TSConfig cfg = new TSConfig();
    private double RUN_TEMP[] = new double[cfg.MAXSTATES];
    private double IDLE_TEMP = 20.;
    List<Double> temperature_data = new ArrayList<>();
    List<Double> temperature_time = new ArrayList<>();

    private TSState.cryostates cstate = TSState.cryostates.NOTCONFIGURED;

    private LS330 tempDev;

    String current_channel = ""; // this device has only one channel
    int current_loop = 1;

    boolean failedToInitialize = false;

    /**
     ***************************************************************************
     **
     ** put Lakeshore device in initial subsystem state
     *
     *
     * @param dummy - unused parameter that exists in the CryoDevice interface's
     * open routine. Used to specify the type of connection which is unnecessary
     * in the Lakeshore driver
     * @param host
     * @param baud
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public Lakeshore325Device(int dummy, String host, int baud) throws DriverException {

        try {
            tempDev = new LS330();
            tempDev.openFtdi(host, baud);
        } catch (DriverException f) {
            log.error("Failed to open connection to Lakeshore device!");
        }

    }

    public Lakeshore325Device() {

    }

    /**
     *
     * Opens connection to a device.
     *
     * @param dummy - unused parameter that exists in the CryoDevice interface's
     * open routine. Used to specify the type of connection which is unnecessary
     * in the Lakeshore driver
     */
    @Command(name = "open", description = "Open connection to Model LakeShore 325")
    public void open(
            @Argument(name = "dummy type", description = "Dummy type") int type,
            @Argument(name = "identity", description = "Device identifier") String ident,
            @Argument(name = "parameter", description = "Device parameter") int param)
            throws DriverException {
        tempDev = new LS330();
        tempDev.openSerial(ident, param);
    }

    /**
     *
     * Displays Temp Control 24C identification.
     *
     */
    @Command(name = "showident", description = "Show Temp Control Lakeshore identification")
    public String showIdentification() throws DriverException {
        String ident = tempDev.getIdent();
        return "Show Lakeshore temp control identification" + ident;
    }

    /**
     *
     * Get Source for Loop returns Channels A through D
     *
     */
    @Command(name = "getMaxSetPoint", description = "Retrieve MaxSetPoint for loop")
    public double getMaxSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        log.info("Function not implemeneted for this device!");
        return -999.;
    }

    /**
     *
     * Set MaxSetPoint for Loop [1-4]
     *
     */
    @Command(name = "setMaxSetPoint", description = "setMaxSetPoint for loop")
    public void setMaxSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "temp", description = "temperature") double temp)
            throws DriverException {
        if (tempDev != null) {
            log.info("Function not implemented for this device!");
        }
        return;
    }

    /**
     *
     *
     * Get Source for Loop returns Channels A through D
     *
     */
    @Command(name = "getSetPoint", description = "Retrieve setPoint for loop")
    public double getSetPoint(@Argument(name = "loop", description = "unused loop") int loop)
            throws DriverException {
        return tempDev.getSetpoint();
    }

    /**
     *
     * Set SetPoint for Loop [1-4]
     *
     */
    @Command(name = "setSetPoint", description = "setSetPoint for loop")
    public void setSetPoint(
            @Argument(name = "loop", description = "unused loop") int loop,
            @Argument(name = "temp", description = "temperature") double temp)
            throws DriverException {
        if (tempDev != null) {
            tempDev.setSetpoint(temp);
        }
        return;
    }

    /**
     *
     * Get Source for Loop returns Channels A through D
     *
     */
    @Command(name = "getLoopSource", description = "Retrieve source channel for loop")
    public char getLoopSource(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return ('A');
    }

    /**
     *
     * Set Source Channel for Loop
     *
     */
    @Command(name = "setLoopSource", description = "Set source channel for loop")
    public void setLoopSource(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "channel", description = "channel [A-D]") String channel)
            throws DriverException {
        if (tempDev != null) {
            tempDev.setControlChannel(channel);
        }
        return;
    }

    /**
     *
     * Set Heater Range for Loop [1-4]
     *
     */
    @Command(name = "setHeaterRange", description = "setHeaterRange for loop")
    public void setHeaterRange(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "range", description = "hi,low,mid") String range)
            throws DriverException {
        if (tempDev != null) {
            log.info("Function not implemented for this device");
        }
        return;
    }

    /**
     *
     *
     * Get heater reading for specified loop
     *
     */
    @Command(name = "getHtrRead", description = "Retrieve heater reading for loop")
    public double getHtrRead(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        log.info("Function not implemented for this device");

        return 1;
    }

    /**
     *
     *
     * Get [P]ID parameter for specified loop
     *
     */
    @Command(name = "getPID_P", description = "Retrieve PID parameter P for loop")
    public double getPID_P(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        log.info("Function not implemented for this device");

        return 1;
    }

    /**
     *
     *
     * Get P[I]D parameter for specified loop
     *
     */
    @Command(name = "getPID_I", description = "Retrieve PID parameter I for loop")
    public double getPID_I(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        log.info("Function not implemented for this device");
        return 1;
    }

    /**
     *
     *
     * Get PI[D] parameter for specified loop
     *
     */
    @Command(name = "getPID_D", description = "Retrieve PID parameter D for loop")
    public double getPID_D(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        log.info("Function not implemented for this device");
        return 1;
    }

    /**
     *
     * Get Temperature Reading For Channels A through D
     *
     */
    @Command(name = "getTemp", description = "Retrieve temperature info")
    public double getTemp(
            @Argument(name = "channel", description = "Temp Channel to read") String channel)
            throws DriverException {
        if (!channel.isEmpty()) {
            char ch = channel.charAt(0);
            tempDev.setControlChannel(channel);
        }
        return tempDev.getControlData();
    }

    /**
     *
     * Return the standard devation of the temperature values
     *
     */
    @Command(type = Command.CommandType.QUERY, description = "Return the standard devation of the temperature values")
    public double getTempStdDev() {
        int ntemp = 0;
        double tempsum = 0.;
        double tempsqsum = 0.;
        for (Iterator<Double> temp = temperature_data.iterator(); temp.hasNext();) {
            double tempval = temp.next();
            tempsum += tempval;
            tempsqsum += tempval * tempval;
            ntemp++;
        }
        double ovrstd = 0.;
        if (ntemp > 0) {
            ovrstd = Math.sqrt(tempsqsum / (double) ntemp - (tempsum * tempsum) / (double) (ntemp * ntemp)) / (double) ntemp;
        }

        return (ovrstd);
    }

    /**
     *
     * Return the rate of change in T/min of the temperature values
     *
     */
    @Command(type = Command.CommandType.QUERY, description = "Return the temperature change over the sample period in degrees/minute")
    public double getTempChange() {
        double first_temp = temperature_data.get(0);
        double first_time = temperature_time.get(0);
        double last_temp = temperature_data.get(temperature_data.size() - 1);
        double last_time = temperature_time.get(temperature_time.size() - 1);

        double slope = 0.;
        double delta_t = last_time - first_time;
        if (delta_t > 0.) {
            slope = (first_temp - last_temp) / delta_t;
        }

        return (slope);
    }

    /**
     *
     * Set the temperature setting for acquisitions
     *
     */
    @Command(name = "setruntemp", description = "Sets the temp value for acquisition")
    public void setRunTemp(double Temp, int cfgstate) {
        RUN_TEMP[cfgstate] = Temp;
        return;
    }

    /**
     *
     * Return the temperature setting for acquisitions
     *
     */
    @Command(name = "getruntemp", description = "Returns the temp value for acquisition")
    public double getRunTemp(int cfgstate) {
        return (RUN_TEMP[cfgstate]);
    }

    /**
     ***************************************************************************
     **
     ** Ramps to the desired temperature over a given time and for nsteps
     *
     * @param duration
     * @param value
     * @param nsteps
     * **************************************************************************
     * @return
     */
    @Command(name = "rampVolts", description = "ramp the voltage")
    public int rampTemp(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "value", description = "Temp to ramp to") double value,
            @Argument(name = "nsteps", description = "number of steps") int nsteps) {
        try {
            if (tempDev != null) {
                double tnow = getTemp(current_channel);
                double tstep = (value - tnow) / (double) nsteps;
                int delta = (int) (1000. * duration / (double) nsteps); // delta in msecs
                System.out.println("ramp T from " + Double.toString(tnow)
                        + " to " + Double.toString(value)
                        + " with tstep = " + Double.toString(tstep)
                        + " with tstep = " + Integer.toString(delta));

                System.out.println("starting T ramp");

                double t = tnow;
                boolean trip0 = isTrip();
//        OKTOTALK = true; // override
                for (int istep = 0; istep < nsteps + 1; istep++) {
                    System.out.println("T = " + t);

                    if (!trip0 && isTrip()) {
                        System.out.println("STOPPING RAMP!!! IT LOOKS LIKE WE CAUSED A TRIP!");
                        break;
                    }
                    this.setSetPoint(1, t);

                    try {
                        Thread.currentThread().sleep(delta);//sleep for delta ms
                    } catch (Exception ie) {
                    }
                    t += tstep;
                }
                System.out.println("T ramp completed");

            }
        } catch (DriverException f) {
            log.error("Cryocon Device failed to ramp temperature!");
        }
        return (1);
    }

    /**
     ***************************************************************************
     **
     ** Ramps to the desired temperature over a given time and for nsteps
     *
     * @param duration
     * @param t_target
     * @param nsteps
     * **************************************************************************
     * @param iwloop
     * @param ioloop
     * @param wchan
     * @param ochan
     * @param delta_warm
     * @return
     */
    @Command(type = Command.CommandType.QUERY, name = "rampTemp", description = "ramp the temperature to the desired point")
    public int rampTemp(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "t_target", description = "Temp to ramp to") double t_target,
            @Argument(name = "nsteps", description = "number of steps") int nsteps,
            @Argument(name = "warmer_loop", description = "number of steps") int iwloop,
            @Argument(name = "other_loop", description = "number of steps") int ioloop,
            @Argument(name = "warm_temp_chan", description = "number of steps") String wchan,
            @Argument(name = "other_temp_chan", description = "number of steps") String ochan,
            @Argument(name = "delta_warm", description = "Temp to ramp to") double delta_warm) {

        try {
            if (tempDev != null) {
                double tnow = getTemp(wchan);
                double tstep = (t_target - tnow) / (double) nsteps;
                int delta = (int) (1000. * duration / (double) nsteps); // delta in msecs
                System.out.println("ramp T from " + Double.toString(tnow)
                        + " C\n to " + Double.toString(t_target)
                        + " C\n with delta Temp. (C) = " + Double.toString(tstep)
                        + "\n and time step (ms) = " + Integer.toString(delta)
                        + "\n Total duration (min) = " + duration / 60.);

                System.out.println("starting temperature ramp ...");
                double t = tnow;
                double t_other = t;
                boolean trip0 = isTrip();
                double cryotol = cfg.getCryoTAcqTol();
//                while ( Math.abs((tnow = getTemp(wchan))-t_target)>cryotol ||  Math.abs((tnow = getTemp(ochan))-t_target)>cryotol ) {
                while ((tstep > 0. && t < (t_target - tstep)) || (tstep < 0. && t > (t_target + tstep))
                        || (tstep > 0. && t_other < (t_target - tstep)) || (tstep < 0. && t_other > (t_target + tstep))) {
                    if ((tstep > 0. && t < (t_target - tstep)) || (tstep < 0. && t > (t_target + tstep))) {
                        t += tstep;

                        double otnow = getTemp(ochan);
                        if (t_other < (t - delta_warm) && otnow < (tnow - delta_warm)) {
                            t_other += tstep;
                        }
                    } else if ((tstep > 0. && t_other < (t_target - tstep)) || (tstep < 0. && t_other > (t_target + tstep))) {
                        t_other += tstep;
                    }

                    if (!trip0 && isTrip()) {
                        log.error("STOPPING RAMP!!! IT LOOKS LIKE WE CAUSED A TRIP!");
                        break;
                    }
                    System.out.println("Setting warm loop temp to " + t + " C");
                    this.setSetPoint(iwloop, t);
                    System.out.println("Setting other loop temp to " + t_other + " C");
                    this.setSetPoint(ioloop, t_other);

                    try {
                        Thread.currentThread().sleep(delta);//sleep for delta ms
                    } catch (Exception ie) {
                    }
                }
                System.out.println("Setting warm loop temp to " + t + " C");
                this.setSetPoint(iwloop, t_target);
                System.out.println("Setting other loop temp to " + t_other + " C");
                this.setSetPoint(ioloop, t_target);
                log.info("temperature ramp completed");

            }
        } catch (DriverException f) {
            log.error("Cryocon Device failed to ramp temperature!");
        }
        return (1);
    }

    /**
     **
     ** Get Temperature Units Channels A through D *
     */
    @Command(name = "getUnit", description = "Retrieve temperature units")
    public char getUnit(
            @Argument(name = "channel", description = "Temp Channel to read") String channel)
            throws DriverException {
        char ch = channel.charAt(0);
        return tempDev.getSampleUnits().charAt(0);
    }

    /**
     *
     * Set Temperature Units for Channels A through D
     *
     */
    @Command(name = "setUnit", description = "Set temperature units")
    public void setUnit(
            @Argument(name = "channel", description = "Temp Channel to set") String channel,
            @Argument(name = "units", description = "Units in [K|C|F|S]") String unit)
            throws DriverException {
        char ch = channel.charAt(0);
        char un = unit.charAt(0);
        if (tempDev != null) {
            tempDev.setSampleChannel(channel);
            tempDev.setSampleUnits(unit);
        }
        return;
    }

    /**
     *
     * Set calibration curve type for Channels A through D
     *
     */
    @Command(name = "setType", description = "Set cal curve type : DIODE|ACR|PTC100|PTC1K|NTC10UA|ACR|TC70|NONE")
    public void setType(
            @Argument(name = "channel", description = "Temp Channel to set") String channel,
            @Argument(name = "type", description = "Calib. curve type [DIODE|ACR|PTC100|PTC1K|NTC10UA|ACR|TC70|NONE]") String typ)
            throws DriverException {
        log.debug("This function is not implmented for this device");
        return;
    }

    /**
     *
     * Put loops in control mode
     *
     */
    @Command(name = "setToControl", description = "put loops in control mode")
    public void setToControl()
            throws DriverException {
        if (tempDev != null) {
            log.error("setToControl is not implemented for this device!");
        }
        return;
    }

    /**
     *
     * Get control mode
     *
     */
    @Command(name = "isInControl", description = "is the Lakeshore in control mode")
    public boolean isInControl()
            throws DriverException {
        boolean inControl = false;
        if (tempDev != null) {
            inControl = tempDev.isInControl();
        }
        return (inControl);
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    public void close() {
        try {
            if (tempDev != null) {
                tempDev.close();
            }
        } catch (DriverException e) {
            log.error("Lakeshore cryo device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the connection. *
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        try {
            if (tempDev != null) {
                log.debug("Lakeshore device connected to:");
                log.debug(showIdentification());
                setOnline(true);
            } else if (!failedToInitialize) {
                log.error("Tried to initialize unconnected cryo device!");
            }
        } catch (DriverException e) {
            if (!failedToInitialize) {
                log.error("Lakeshore failed to initialize!!!");
            }
            failedToInitialize = true;
        }
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_TEMP) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
            if (type == Channel.TYPE_TEMP) {
            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        try {
            if (type == Channel.TYPE_TEMP) {
                log.debug("readChannel in CryoCon performing getTemp");
                if (chan == 0) {
                    value = getTemp(current_channel);
                    temperature_data.add(value);
                    temperature_time.add(System.currentTimeMillis() / 60000.); //time in minutes
                    if (temperature_data.size() > 120) {
                        temperature_data.remove(0);
                        temperature_time.remove(0);
                    }
                } else if (chan == -1) {
                    value = getRunTemp(1);
                } else if (chan == 2) {
                    value = getTempStdDev();
                }
            }
        } catch (Exception e) {
            log.error("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    // THE FOLLOWING METHODS HAVE BEEN SUPERCEDED AND WILL SOON BE REMOVED
    public double get_init_temp(int cfgstate) // soon to be deprecated
    {
        return this.RUN_TEMP[cfgstate];
    }

    public double get_idle_temp() {
        return this.IDLE_TEMP;
    }

    /**
     ***************************************************************************
     **
     ** gets the device state
     * **************************************************************************
     */
    @Command
    public int getState() {
        return (cstate.ordinal());
    }

    /**
     ***************************************************************************
     **
     ** sets the device state
     * **************************************************************************
     */
    @Command
    public void setState(int istate) {
        cstate = TSState.cryostates.values()[istate];
    }

    public String getCurrent_channel() {
        return current_channel;
    }

    public void setCurrent_channel(String current_channel) {
        this.current_channel = current_channel;
    }


    public String getCurrent_channel2() {
        return current_channel;
    }

    public void setCurrent_channel2(String current_channel) {
        this.current_channel = current_channel;
    }

    public int getCurrent_loop() {
        return current_loop;
    }

    public void setCurrent_loop(int current_loop) {
        this.current_loop = current_loop;
    }

    boolean isTrip() {
        if (cstate == TSState.cryostates.TRIPPED) {
            return true;
        }
        return false;
    }

    boolean setTrip() {
        cstate = TSState.cryostates.TRIPPED;
        return false;
    }

}
