package org.lsst.ccs.subsystem.teststand;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 *  APC7900 simulated device class for the teststand subsystem
 *
 *  @author: Owen Saxton
 */
public class APC7900SimDevice extends APC7900Device {

    private int onStates = 0;

    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        outNumMap = new HashMap<>();
        fullName = "Simulated APC7900 PDU";
    }


    /**
     *  Initializes the device.
     */
    @Override
    protected void initialize()
    {
        outNums = new int[outlets.size()];
        initSensors();
        setOnline(true);
        kstate = TSState.pwrstates.OK;
        sLog.info("Connected to " + fullName);
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        kstate = TSState.pwrstates.NOTCONFIGURED;
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  Two-element array containing the encoded type [0] and subtype [1] 
    *  @throws  Exception  If parameters are invalid
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        int[] typeSubtype = super.checkChannel(name, hwChan, type, subtype);
        if ((typeSubtype[0] & 0xffff) == TYPE_OUTLET) {
            outNumMap.put(subtype, outlets.size());
        }
        return typeSubtype;
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = 0;
        switch (type & 0xffff) {
        case TYPE_OUTLET:
            int outNum = outNums[type >> 16];
            value = (onStates >> outNum) & 1;
            break;

        case TYPE_POWER:
            value = hwChan == CHAN_POWER ? 1650.0 : 15.0;
        }
        return value;
    }


    /**
     *  Gets the map of outlet on states.
     *
     *  @return  The map of outlet names to on states (true or false)
     *  @throws  DriverException
     */
    @Override
    @Command(name="getOutletOnStateMap", description="Get the map of outlet on states")
    public Map<String, Boolean> getOutletOnStateMap() throws DriverException
    {
        Map<String, Boolean> onStateMap = new HashMap<>();
        for (String oName : outNumMap.keySet()) {
            onStateMap.put(oName, ((1 << outNumMap.get(oName)) & onStates) != 0);
        }
        return onStateMap;
    }


    /**
     *  Turns an outlet on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Override
    @Command(name="outletOn", description="Turn outlet on")
    public void outletOn(@Argument(name="name", description="Outlet name")
                         String name) throws DriverException
    {
        onStates |= (1 << getOutletNumber(name));
    }


    /**
     *  Turns an outlet off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Override
    @Command(name="outletOff", description="Turn outlet off")
    public void outletOff(@Argument(name="name", description="Outlet name")
                          String name) throws DriverException
    {
        onStates &= ~(1 << getOutletNumber(name));
    }


    /**
     *  Forces an outlet to turn on.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Override
    @Command(name="forceOutletOn", description="Force outlet to turn on")
    public void forceOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        outletOn(name);
    }


    /**
     *  Forces an outlet to turn off.
     *
     *  @param  name  The outlet name
     *  @throws  DriverException
     */
    @Override
    @Command(name="forceOutletOff", description="Force outlet to turn off")
    public void forceOutletOff(@Argument(name="name", description="Outlet name")
                               String name) throws DriverException
    {
        outletOff(name);
    }


    /**
     *  Gets whether an outlet is on.
     *
     *  @param  name  The outlet name
     *  @return  Whether outlet is turned on
     *  @throws  DriverException
     */
    @Override
    @Command(name="isOutletOn", description="Get whether outlet is on")
    public boolean isOutletOn(@Argument(name="name", description="Outlet name")
                              String name) throws DriverException
    {
        return (onStates & (1 << getOutletNumber(name))) != 0;
    }

}
