package org.lsst.ccs.subsystem.teststand;

import java.util.Map;
import java.util.HashMap;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.tripplitepdu.TrippLitePDU;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 ***************************************************************************
 **
 * TrippLitePDU device class for the teststand subsystem
 *
 * * @author: Homer Neal
 * **************************************************************************
 */
public class TrippLitePDUDevice extends Device implements PDUDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    private boolean isconnected = false; // is the device defined

    TSConfig cfg = new TSConfig();

    TrippLitePDU devpdu = null;

    String pdu_host = null;

    int pdu_port = 23;

    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;

    boolean open = false;
    boolean failedToInitialize = false;
    private int major_version;


    /**
     ***************************************************************************
     **
     ** TrippLitePDUDevice constructor - put TrippLitePDU in initial subsystem state
     *
     ** @param host device address
     ** @param port port or baud * * @param runBias CCD bias voltage while
     * running
     *
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public TrippLitePDUDevice(String host, int port)
            throws DriverException {
        isconnected = true;
        pdu_host = host;
        pdu_port = port;

        try {
            devpdu = new TrippLitePDU();

        } catch (DriverException f) {
            System.out.println("Failed to open connection to TrippLitePDU device!");
            isconnected = false;
        }

    }

    /**
     ***************************************************************************
     **
     ** Constructor *
     * **************************************************************************
     */
    public TrippLitePDUDevice() {
//        isconnected = true;
        try {
            devpdu = new TrippLitePDU();
        } catch (DriverException f) {
            System.out.println("Failed to create to TrippLitePDU device instance!");
        }

    }

    /**
     ***************************************************************************
     **
     ** Performs configuration.
     *
     *
     * **************************************************************************
     */
    @Override
    protected void initDevice() {
        fullName = "TrippLitePDU module";
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (devpdu != null) {
                devpdu.close();
            }
            kstate = TSState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("TrippLitePDU device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the device
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        if (isconnected) {
            log.debug("TrippLitePDU device is already initialized");
            /*
             try {
             log.debug(devpdu.getIdent());
             } catch (Exception e) {
             if (!failedToInitialize) {
             log.error("TrippLitePDU device failed to respond to ident request! The initialization has FAILED.");
             failedToInitialize = true;
             }
             isconnected = false;
             }
             */
            log.debug("Setting device online.");
            setOnline(true);
            kstate = TSState.pwrstates.OK;
        } else {
            if (!failedToInitialize) {
                log.error("Tried to initialize unconnected PDU device!");
                failedToInitialize = true;
            }

        }
    }

    /**
     ***************************************************************************
     **
     ** Opens FTDI connection to a device. *
     * **************************************************************************
     * @param host
     * @param port
     * @param major_version
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command(name = "openftdi", description = "Open FTDI connection to device")
    public void openftdi(@Argument(name = "serialname", description = "Serial device name") String host,
            @Argument(name = "baud", description = "baud rate") int port,
            @Argument(name = "major_version", description = "major version of PDU firmware") int major_version
    ) throws DriverException {
        devpdu.openftdi(host, port,major_version);
        this.major_version = major_version ;
        kstate = TSState.pwrstates.OK;
    }

    /**
     ***************************************************************************
     **
     ** set an outlet state *
     * **************************************************************************
     * @param outlet
     * @param otlstate
     */
    @Command(type = Command.CommandType.ACTION, name = "setOutletState", description = "set the state of an outlet")
    @Override
    public void setOutletState(@Argument(name = "outlet", description = "outlet number") int outlet,
            @Argument(name = "otlstate", description = "on(true)/off(false) state") boolean otlstate) {
        if (devpdu == null) {
            return; // not even defined perhaps
        }
        try {
            devpdu.opennet(pdu_host, pdu_port,major_version); // open connection
            open = true;
//            devpdu.setTimeout(60.0);
        } catch (DriverException f) {
            System.out.println("Failed to open connection to TrippLitePDU device!");
            isconnected = false;
        }
        try {
            if (open) {
                devpdu.setOutletState(outlet, otlstate);
            }
        } catch (DriverException f) {
            log.error("Failed open/close the TrippLitePDU");
        }
        try {
            open = false;
            devpdu.close(); // close connection
        } catch (DriverException f) {
            System.out.println("Failed to close connection to TrippLitePDU device!");
        }

    }

    /**
     ***************************************************************************
     **
     ** get an outlet state *
     * **************************************************************************
     * @param outlet
     * @return 
     */
    @Command(type = Command.CommandType.ACTION, name = "getOutletState", description = "set the state of an outlet")
    @Override
    public boolean getOutletState(@Argument(name = "outlet", description = "outlet number") int outlet) {
        boolean otlstate = false;
        if (devpdu == null) {
            return (false); // not even defined perhaps
        }
        try {
            devpdu.opennet(pdu_host, pdu_port, major_version); // open connection
            open = true;
//            devpdu.setTimeout(60.0);
        } catch (DriverException f) {
            System.out.println("Failed to open connection to TrippLitePDU device!");
            open = false;
        }
        try {
            if (open) {
                Sleep(1.0);
                otlstate = devpdu.isOutletOn(outlet);
            }
        } catch (DriverException f) {
            log.error("Failed open/close the TrippLitePDU");
        }
        try {
            devpdu.close(); // close connection
        } catch (DriverException f) {
            System.out.println("Failed to close connection to TrippLitePDU device!");
        }

        return (otlstate);
    }

       /**
     ***************************************************************************
     **
     ** Sleep - what a waste
     * **************************************************************************
     */
    public void Sleep(double secs) {
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            System.out.println("Rude awakening!" + ex);
        }
    }

    /**
     ***************************************************************************
     **
     ** Converts a boolean to on/off. *
     * **************************************************************************
     */
    private String getOnOff(boolean on) {
        return on ? "on" : "off";
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return 
     * @throws java.lang.Exception 
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_SWITCH) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     * @param chan
     * @param type
     * @param subtype
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
            if (type == Channel.TYPE_SWITCH) {
                log.debug("setting channel online - chan=" + chan);
                setOnline(true);
            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     * @param chan
     * @param type
     * @return 
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        log.debug("TrippLitePDUDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            if (type == Channel.TYPE_SWITCH) {
                value = 1.0; // to be replaced with a byte indicating the outlet states
            }
        } catch (Exception e) {
            log.debug("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    @Command(name = "setstate", description = "set TrippLitePDU device status")
    public void setState(int istate) {
        kstate = TSState.pwrstates.values()[istate];
    }

    @Command(name = "getstate", description = "get TrippLitePDU device status")
    public int getState() {
        return (kstate.ordinal());
    }

}
