package org.lsst.ccs.subsystem.teststand;

import java.util.Random;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;

/**
 *
 * Device class for the NewportLamp driver
 *
 * @author homer
 */
public class NewportLampSimDevice extends Device implements LampDevice {

    private String lampType = "undef";
    TSConfig cfg = new TSConfig();
    private double[] runlampcurr = new double[cfg.MAXSTATES];

    private int itype = Ascii.CONN_TYPE_SERIAL, port;
    private String host;

    boolean isConnected = false;
    boolean failedToInitialize = false;

    private int powerPreset;
    
    private boolean isPowerOn = false;
    
    private Random r = new Random();

    public NewportLampSimDevice(String host, int port) throws DriverException {
        this(Ascii.CONN_TYPE_SERIAL, host, port);
    }

    public NewportLampSimDevice(int itype, String host, int baud) throws DriverException {
        this.itype = itype;
        this.host = host;
        this.port = port;
    }

    /**
     ***************************************************************************
     **
     ** isLampPowerOn - Is the NewportLamp Device on? *
     * **************************************************************************
     */
    @Command(name = "isLampPowerOn", description = "returns whether the NewportLamp device is active")
    public boolean isLampPowerOn() {
        return isPowerOn;
    }

    /**
     ***************************************************************************
     **
     ** enables/disable the power to the lamp *
     * **************************************************************************
     */
    @Command(name = "setLampPowerEnable", description = "enables/disables the power to the lamp from the NewportLamp device")
    public void setLampPowerEnable(
            @Argument(name = "enable", description = "enables/disables lamp power") boolean on
    ) {
        isPowerOn = true;
    }

    /**
     ***************************************************************************
     *
     * returns the current read from the NewportLamp device
     *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getLampCurrent", description = "returns the #amps read from the NewportLamp device")
    public double getLampCurrent() {
        return isPowerOn ? 4.0 + r.nextGaussian() : 0.0;
    }

    /**
     ***************************************************************************
     **
     ** sets the current for the NewportLamp device *
     * **************************************************************************
     */
    @Command(name = "setLampCurrent", description = "(not implemented) sets the current for the NewportLamp device")
    public void setLampCurrent(double current) {
        log.error("Purposely not implemented");
        return;
    }

    /**
     ***************************************************************************
     *
     * returns the current read from the NewportLamp device
     *
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getLampPower", description = "returns the #amps read from the NewportLamp device")
    public double getLampPower() {
        return isPowerOn ? 7.0 + r.nextGaussian() : 0.0;
    }

    /**
     ***************************************************************************
     **
     ** returns the current read from the NewportLamp device *
     * **************************************************************************
     */
    @Command(name = "setLampPower", description = "(not implimented) sets the power for the NewportLamp device")
    public void setLampPower(double current) {
        log.error("Purposely not implemented!!!");
        return;
    }

    /**
     ***************************************************************************
     **
     ** defines the lamp type *
     * **************************************************************************
     */
    @Command(name = "setType", description = "define lamp type")
    public void setType(String type) {
        this.lampType = type;
    }

    /**
     ***************************************************************************
     **
     ** returns the lamp type as defined by the subsys or user *
     * **************************************************************************
     */
    @Command(name = "getType", description = "returns the lamp type")
    public String getType() {
        return (lampType);
    }

    /**
     ***************************************************************************
     **
     ** returns the identity read from the NewportLamp device *
     * **************************************************************************
     */
    @Command(name = "getLampIdent", description = "returns the identity read from the NewportLamp device")
    public String getLampIdent() {
        String ident = "?";
        return (ident);
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        isConnected = false;
    }

    /**
     ***************************************************************************
     **
     ** Initializes the connection. *
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        isConnected = true;
        setOnline(true);
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        log.debug("NewportLampDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            if (chan == 1) {
                value = getLampCurrent();
            } else {
                value = getLampPower();
            }
        } catch (Exception e) {
            log.error("Error reading channel type " + type + " channel " + chan, e);
        }
        return (value);
    }

    public void setRunlampcurr(double runlampcurr, int iwstate) {
        this.runlampcurr[iwstate] = runlampcurr;
    }

    public double getRunlampcurr(int iwstate) {
        return runlampcurr[iwstate];
    }

    @Command(name = "setPowerPreset", description = "Set the power preset", type = Command.CommandType.ACTION)
    public void setPowerPreset(@Argument(name = "powerPreset", description = "desired power") int powerPreset) throws DriverException {
        this.powerPreset = powerPreset;
    }

    @Command(name = "getStatus", description = "Read Status", type = Command.CommandType.QUERY)
    public String getStatus() throws DriverException {
        return "?";
    }

    @Command(name = "getPowerPreset", description = "Read Power Preset", type = Command.CommandType.QUERY)
    public int getPowerPreset() throws DriverException {
        return powerPreset;
    }

    @Command(name = "getVolts", description = "Read Volts", type = Command.CommandType.QUERY)
    public double getVolts() throws DriverException {
        return isPowerOn ? 45.0 + r.nextGaussian() : 0.0;
    }

    @Command(name = "getLampHrs", description = "Read Lamp Hrs", type = Command.CommandType.QUERY)
    public String getLampHrs() throws DriverException {
        return "13";
    }

    @Command(name = "genericRead", description = "Generic read", type = Command.CommandType.QUERY)
    public String genericRead(@Argument(name = "command", description = "generic read command") String command) throws DriverException {
        return "?";
    }

}
