package org.lsst.ccs.subsystem.teststand;

import java.util.HashMap;
import java.util.Map;
import java.util.Date;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.twistorr.TwisTorr84;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  Interfaces with driver for Agilent TwisTorr84 turbo pump.
 *
 *  @author Al Eisner
 */
public class TwisTorr84Device extends Device {

   /**
    *  Data fields.
    */

    private final TwisTorr84 turbo = new TwisTorr84();  // Associated driver

   /**
    *  Private lookup maps for driver read commands.  The default
    *  initial capacity of 16 is reasonable for the expected number
    *  of read commands.  The key is a numeric identifier of a monitor
    *  channel defined in the groovy file.
    *
    *  typeMap identifies whether a quantity to be read is of type
    *  logical ("Bool"), integer ("Numeric") or alphanumeric ("Alpha").
    *  It is taken from the type parameter in the groovy file.
    *
    *  itemMap identifies the specific quantity ro be read.  It is
    *  taken from the subtype parameter in the groovy file.
    */
    private Map<Integer, String> typeMap = new HashMap<>();
    private Map<Integer, String> itemMap = new HashMap<>();
    private Integer chanKey;

    /* Configuration parameters related to hardware settings */

    @ConfigurationParameter
    protected String devcId;       // Device connection identification

    @ConfigurationParameter(isFinal = true,
                            description = "true/false for low/normal speed ")
    protected Boolean lowSpeedMode;

    @ConfigurationParameter(isFinal = true, 
                            description = "true/false for water/air cooling")
    protected Boolean waterCooling; 

    @ConfigurationParameter(isFinal = true,
                            description = "true/false for manual/auto")
    protected Boolean ventValveByCmnd;  

    @ConfigurationParameter(isFinal = true,
                            description = "true/false for continuous/impulse")
    protected Boolean interlockType;   // 

    @ConfigurationParameter
    protected Boolean softStartMode;   // settable only if pump stopped

    @ConfigurationParameter
    protected Boolean activeStopMode;  // settable only if pump stopped

    /* Hard-coded hardware-initialization parameters */

    private final int rotfreq_set = 1350;   // normal rotation frequncy (Hz) 
    private final int rotfreq_low = 1100;   // slow rotation frequncy (Hz) 
    private final boolean serialType = false;  // true/false for RS-485/RS-232
    private final boolean gasType = false;     // true/false for Argon/Nitrogen

   /**
    *  Performs basic initialization.  
    *  Verify that all ConfigurationParameters are set.
    */
    @Override
    protected void initDevice() {

        chanKey = 0;

        if (devcId == null) {
            MonitorLogUtils.reportConfigError(log, name, "devcId", "is missing");
        }
        fullName = "TwisTorr84 (" + devcId + ")";

        if (lowSpeedMode == null) {
            MonitorLogUtils.reportConfigError(log, name, "lowSpeedMode", "is missing");
        }
        if (waterCooling == null) {
            MonitorLogUtils.reportConfigError(log, name, "waterCooling", "is missing");
        }
        if (ventValveByCmnd == null) {
            MonitorLogUtils.reportConfigError(log, name, "ventValveByCmnd", "is missing");
        }
        if (interlockType == null) {
            MonitorLogUtils.reportConfigError(log, name, "interlockType", "is missing");
        }
        if (softStartMode == null) {
            MonitorLogUtils.reportConfigError(log, name, "softStartMode", "is missing");
        }
        if (activeStopMode == null) {
            MonitorLogUtils.reportConfigError(log, name, "activeStopMode", "is missing");
        }
    }


   /**
    *  Performs full initialization.
    */
    @Override
	protected void initialize() {

        try {
            turbo.open(devcId);
            setOnline(true);
            initSensors();
            log.info("Connected to " + fullName);

            /* Hardware initialization */

            turbo.writeBool(TwisTorr84.CmndBool.LOW_SPEED, lowSpeedMode);
            turbo.writeBool(TwisTorr84.CmndBool.VENTVALVE_BY_CMND, ventValveByCmnd);
            turbo.writeBool(TwisTorr84.CmndBool.INTERLOCK_TYPE, interlockType);
            turbo.writeBool(TwisTorr84.CmndBool.SOFT_START, softStartMode);
            turbo.writeBool(TwisTorr84.CmndBool.ACTIVE_STOP, activeStopMode);
            turbo.writeBool(TwisTorr84.CmndBool.SERIAL_TYPE, serialType);
            turbo.writeBool(TwisTorr84.CmndBool.GAS_TYPE_ARGON, gasType);
            turbo.writeNumeric(TwisTorr84.CmndNumeric.ROTFREQ_SET ,rotfreq_set);
            turbo.writeNumeric(TwisTorr84.CmndNumeric.ROTFREQ_LOW ,rotfreq_low);

            /* Settings requiring that pump be in stopped state */
            int status = turbo.readNumeric(TwisTorr84.CmndNumeric.STATUS);
            if (TwisTorr84.PumpStatus.decode(status) == "STOP") {
                turbo.writeBool(TwisTorr84.CmndBool.WATER_COOLING, waterCooling);
            }
            log.info("Hardware initialized for " + fullName);
        }

        catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to or initializaing " 
                          + fullName + ": " + e);
            }
            close();
            return;
        }

        inited = true;
    }


   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            turbo.close();
        }
        catch (DriverException e) {
        }
    }

    /**
     * Pump commands and settings
     */

   /**
    *  Select soft start vs. normal start
    *
    *  @param value  True for soft-start mode, false for normal mode.
    *  @throws  DriverException
    */
    @ConfigurationParameterChanger(propertyName = "softStartMode")
    protected void setSoftStart(@Argument(name="<true|false>",description=
     "Soft-start mode true or false") boolean value) throws DriverException
    {
        turbo.writeBool(TwisTorr84.CmndBool.SOFT_START, value);
    }

   /**
    *  Select active stop (with braking) vs. normal stop mode
    *
    *  @param value  True for active-stop mode, false for normal-stop mode.
    *  @throws  DriverException
    */
    @ConfigurationParameterChanger(propertyName = "activeStopMode")
    protected void setActiveStop(@Argument(name="<true|false>",description=
     "Active-stop mode true or false") boolean value) throws DriverException
    {
        turbo.writeBool(TwisTorr84.CmndBool.ACTIVE_STOP, value);
    }

   /**
    *  Start pump
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="startTurboPump", 
             description="Start turbo pump")
    public void startTurboPump() throws DriverException
    {
        turbo.writeBool(TwisTorr84.CmndBool.START_STOP, true);
    }

   /**
    *  Stop pump
    *
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.ACTION, name="stopTurboPump", 
             description="Stop turbo pump")
    public void stopTurboPump() throws DriverException 
    {
        turbo.writeBool(TwisTorr84.CmndBool.START_STOP, false);
    }

   /**
    *  Open or close the vent valve
    */
   /*  To be added later (currently there is no controllable vent valve).

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, 
                                 String subtype) throws Exception
    {
        try {
            switch (type) {
                case "Bool":
                    TwisTorr84.CmndBool.valueOf(subtype);
                    break;
                case "Numeric":
                    TwisTorr84.CmndNumeric.valueOf(subtype);
                    break;
                case "Alpha":
                    TwisTorr84.CmndAlpha.valueOf(subtype);
                    break;
                default:
                    MonitorLogUtils.reportError(log, name, "type", type);
            }
        }
        catch (Exception e) {
            MonitorLogUtils.reportError(log, name, type + " subtype ",
					subtype);
        }
        chanKey++;
        typeMap.put(chanKey,type);
        itemMap.put(chanKey,subtype);
        return new int[]{chanKey, 0};
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return  The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);   //NaN
        String itemType;
        String item;

        try {
            itemType = typeMap.get(type);
        }
        catch (Exception e) {
            log.error("Error reading typeMap, TurboPump channel " +type+" "+e);
            return value;
        }
        try {
            item = itemMap.get(type);
        }
        catch (Exception e) {
            log.error("Error reading itemMap, TurboPump channel " +type+" "+e);
            return value; 
        }
        try {
	    switch (itemType) {
                case "Bool":
                    if (turbo.readBool(TwisTorr84.CmndBool.valueOf(item))) {
                        value = 1.0;
			    } else {
                        value = 0.0;
                    }
                    break;
                case "Numeric":
                    value = turbo.readNumeric(TwisTorr84.CmndNumeric.valueOf(item));
                    break;
                case "Alpha":
                    turbo.readAlpha(TwisTorr84.CmndAlpha.valueOf(item));
                    // Not implemented, relevant hardware not installed
                    break;
	    }
        }        
        catch (DriverTimeoutException et) {
            log.error("TurboPump timeout reading data " + item + ": " + et);
            setOnline(false);
        }
	catch (DriverException e) {
            log.error("TurboPump exception reading data " + item + ": " + e);
        }
        
        return value;
    }

    /* Read-commands for command-line usage */

   /** 
    *  Read and decode pump status
    *
    *  @return  String indicating pump state or failure
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="readTurboStatus", 
             description="Get pump status, including operating modes and failure")
    public String readTurboStatus()  throws DriverException
    {
        int status = turbo.readNumeric(TwisTorr84.CmndNumeric.STATUS);
        return TwisTorr84.PumpStatus.decode(status);
    }

   /** 
    *  Read error code (bits show reason for pump failure)
    *
    *  @return  String giving error code in hexadecimal and decoding it
    *  @throws  DriverException
    */
    @Command(type=Command.CommandType.QUERY, name="readTurboFailCode", 
             description="Fin reason(s) for failure")
    public String readTurboFailCode()  throws DriverException
    {
        int errcode = turbo.readNumeric(TwisTorr84.CmndNumeric.ERRCODE);
        return Integer.toString(errcode,16) + ", "+ TwisTorr84.decodeError(errcode);
    }

   /**
    * Read all settings and data from pump controller.
    *
    * Loops over all read commands and returns them in a table format.
    * All DriverExceptions are caught; if one occurs, the data field
    * is replaced by the text (String) associated with the exception.
    *  
    * @return  String reporting all data read and exceptions.
    */
    @Command(type=Command.CommandType.QUERY, name="readAll", 
             description="Read all TurboPump controller settings and data")
    public String readAll()
    {
        String table = "Read all turbo-pump settings and data\n" +"\n";

        /* Boolean commands */

        TwisTorr84.CmndBool cmndB[] = TwisTorr84.CmndBool.values();
        int nB = cmndB.length;
        for (int i = 0; i < nB; i++) {
            table += String.format("\n   %-22s", cmndB[i]);
            try {
                boolean respB = turbo.readBool(cmndB[i]);
                table += Boolean.toString(respB);
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";

        /* Numeric commands */

        TwisTorr84.CmndNumeric cmndN[] = TwisTorr84.CmndNumeric.values();
        int nN = cmndN.length;
        for (int i = 0; i < nN; i++) {
            table += String.format("\n   %-22s", cmndN[i]);
            try {
                int respN = turbo.readNumeric(cmndN[i]);
                table += Integer.toString(respN);
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }
        table += "\n";

        TwisTorr84.CmndAlpha cmndA[] = TwisTorr84.CmndAlpha.values();
        int nA = cmndA.length;
        for (int i = 0; i < nA; i++) {
            table += String.format("\n   %-22s", cmndA[i]);
            try {
                String respA = turbo.readAlpha(cmndA[i]);
                table += respA;
            } catch (DriverException ex) {
                table += ex.getMessage();
            }
        }

        return table;
    }

    public String timestamp() {
        Date now = new Date();
        return now.toString();
    }
}