package org.lsst.ccs.subsystem.teststand;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.apcpdu.AP9630UPS;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *  AP9630UPS device class for the teststand subsystem
 *
 *  @author: Owen Saxton
 */
public class AP9630UPSDevice extends Device {

    public final static int
        CHAN_OUT_VOLTAGE = 0,
        CHAN_OUT_CURRENT = 1,
        CHAN_OUT_POWER = 2,
        CHAN_BATT_TEMP = 3,
        CHAN_BATT_VOLTAGE = 4,
        CHAN_BATT_CHARGE = 5,
        CHAN_STATUS = 6,
        CHAN_REM_TIME = 7,
        NUM_CHANNELS = 8;

    @ConfigurationParameter(isFinal = true, description = "telnet node")
    String node;
    String user = "apc";
    String passwd = "apc";

    final Logger sLog = Logger.getLogger(getClass().getPackage().getName());
    private final AP9630UPS ups = new AP9630UPS();
    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            MonitorLogUtils.reportConfigError(sLog, name, "node", "is not specified");
        }
        fullName = "AP9630 UPS (" + node + ")";
    }


    /**
     *  Initializes the device.
     */
    @Override
    protected void initialize()
    {
        try {
            ups.open(AP9630UPS.ConnType.TELNET, node, user, passwd);
            initSensors();
            setOnline(true);
            kstate = TSState.pwrstates.OK;
            sLog.info("Connected to " + fullName);
        }
        catch (DriverException e) {
            if (!inited) {
                sLog.error("Error connecting to " + fullName + ": " + e);
            }
            close();
        }
        inited = true;
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            ups.close();
        }
        catch (DriverException e) {
        }
        kstate = TSState.pwrstates.NOTCONFIGURED;
    }


   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  Two-element array containing the encoded type [0] and subtype [1] 
    *  @throws  Exception  If parameters are invalid
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= NUM_CHANNELS) {
            MonitorLogUtils.reportError(sLog, name, "hardware channel", hwChan);
        }
        
        return new int[]{0, 0};
    }


    /**
     *  Initializes a channel.
     *
     *  @param  name     The channel name
     *  @param  id       The channel ID
     *  @param  hwChan   The hardware channel
     *  @param  type     The encoded channel type
     *  @param  subtype  The encoded channel subtype
     */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan  The hardware channel number
     *  @param  type    The encoded channel type
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = 0;
        try {
            switch (hwChan) {
            case CHAN_OUT_VOLTAGE:
                return ups.getOutputVoltage();
            case CHAN_OUT_CURRENT:
                return ups.getOutputCurrent();
            case CHAN_OUT_POWER:
                return ups.getOutputPower();
            case CHAN_BATT_VOLTAGE:
                return ups.getBatteryVoltage();
            case CHAN_BATT_CHARGE:
                return ups.getBatteryCharge();
            case CHAN_BATT_TEMP:
                return ups.getTemperature();
            case CHAN_STATUS:
                return ups.getStatus();
            case CHAN_REM_TIME:
                return ups.getRunTime();
            }
        }
        catch (DriverException e) {
            sLog.error("Error reading UPS channel: " + e);
            setOnline(false);
        }
        return value;
    }


    /**
     *  Gets the UPS status.
     *
     *  @return The status
     *  @throws DriverException
     */
    @Command(name="getStatus", description="Get the UPS status")
    public int getStatus() throws DriverException
    {
        try {
            return ups.getStatus();
        } catch (DriverException e) {
            log.error("Error getting UPS status: " + e);
            throw e;
        }
    }


    /**
     *  Gets the UPS battery time remaining.
     *
     *  @return The remaining time (secs)
     *  @throws DriverException
     */
    @Command(name="getRemTime", description="Get the UPS remaining time")
    public int getRemTime() throws DriverException
    {
        try {
            return ups.getRunTime();
        } catch (DriverException e) {
            log.error("Error getting UPS remaining time: " + e);
            throw e;
        }
    }


    /**
     *  Disables the UPS.
     *
     *  Needed to relinquish control to the web or another telnet client
     */
    @Override
    @Command(name="disable", description="Disable the connection to the UPS")
    public void disable()
    {
        super.disable();
        setOnline(false);
    }


    /**
     *  Enables the UPS.
     */
    @Override
    @Command(name="enable", description="Enable the connection to the UPS")
    public void enable()
    {
        super.enable();
    }

}
