package org.lsst.ccs.subsystem.teststand;

import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.cryocon.M24C;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * Device class for the Cryogenic Temperature Controller 24C driver
 *
 * @author Heather Kelly / Homer Neal *
 */
public class CryoCon24cDevice extends Device implements CryoDevice {

    public static final int
        CHAN_CURR_TEMP = 0,
        CHAN_RUN_TEMP = 1,
        CHAN_TEMP_STDDEV = 2,
        CHAN_HTR_1 = 3,
        CHAN_HTR_2 = 4,
        CHAN_TEMP_A = 5,
        CHAN_TEMP_B = 6,
        CHAN_TEMP_C = 7,
        CHAN_TEMP_D = 8,
        CHAN_PID_P_1 = 9,
        CHAN_PID_I_1 = 10,
        CHAN_PID_D_1 = 11,
        CHAN_PID_P_2 = 12,
        CHAN_PID_I_2 = 13,
        CHAN_PID_D_2 = 14,
        CHAN_TEMP_CHNG = 15,
        CHAN_SETPOINT_1 = 16,
        CHAN_SETPOINT_2 = 17,
        NUM_CHANS = 18;

    private static final Set<String> channels = new HashSet<>();
    static {
        channels.add("A");
        channels.add("B");
        channels.add("C");
        channels.add("D");
    }

    private final static int EVENT_ID_CRYO = 0;

    TSConfig cfg = new TSConfig();
    private double RUN_TEMP[] = new double[TSConfig.MAXSTATES];

    private TSState.cryostates cstate;

    private final M24C m24c = new M24C();

    public String current_channel = "B"; // channel to be operated on 
    public String current_channel2 = "C"; // other channel to be monitored
    public int current_loop = 1; // loop to be operated on 

    List<Double> temperature_data_A = new ArrayList<>();
    List<Double> temperature_time_A = new ArrayList<>();
    List<Double> temperature_data_B = new ArrayList<>();
    List<Double> temperature_time_B = new ArrayList<>();
    List<Double> temperature_data_C = new ArrayList<>();
    List<Double> temperature_time_C = new ArrayList<>();
    List<Double> temperature_data_D = new ArrayList<>();
    List<Double> temperature_time_D = new ArrayList<>();
    Map<String, List<Double>> tempDataMap = new HashMap<>();
    Map<String, List<Double>> tempTimeMap = new HashMap<>();

    private final Logger log = Logger.getLogger(getClass().getPackage().getName());

    @ConfigurationParameter
    protected double[] maxSetPoints;
    
    //Can this be a String[]? i.e. are the channels ordered as the maxSetPoints?
    //The Map of types for all the channels.
    //Can bot the channels and the types be Enums?
    @ConfigurationParameter
    protected Map<String,String> channelTypes;
    
    //Same questions as above.
    @ConfigurationParameter
    protected Map<String,String> channelUnits;

    M24C.ConnType connType = M24C.ConnType.NET;
    String host;
    int port = 5000;

    /**
     * Constructor: put CryoCon device in initial subsystem state
     *
     * @param itype
     * @param host
     * @param port
     * @throws DriverException
     //
    public CryoCon24cDevice(int itype, String host, int port) throws DriverException {
        this();
        try {
            tempDev = new M24C(itype, host, port);            
        } catch (DriverException f) {
            log.error("Failed to open connection to CryoCon device: " + f);
        }

    }
    */

    /**
     * No-argument constructor.
     */
    public CryoCon24cDevice() {
        this.cstate = TSState.cryostates.NOTCONFIGURED;
        tempDataMap.put("A", temperature_data_A);
        tempTimeMap.put("A", temperature_time_A);
        tempDataMap.put("B", temperature_data_B);
        tempTimeMap.put("B", temperature_time_B);
        tempDataMap.put("C", temperature_data_C);
        tempTimeMap.put("C", temperature_time_C);
        tempDataMap.put("D", temperature_data_D);
        tempTimeMap.put("D", temperature_time_D);
    }

    /**
     *
     * Sets maxSetPoints array
     *
     * @param maxSetPoints  The 4-element array of set points
     * @throws DriverException
     */
    @ConfigurationParameterChanger(propertyName = "maxSetPoints")
    protected void setMaxSetPoints(double[] maxSetPoints) throws DriverException {
        if ( maxSetPoints == null || maxSetPoints.length != M24C.NUM_LOOPS ) {
            throw new IllegalArgumentException("Invalid maxSetPoints array provided. It must contain four elements.");
        }
        this.maxSetPoints = maxSetPoints;
    }

    /**
     * Sets channelTypes map
     *
     * @param channelTypes The map of channels to units 
     * @throws DriverException
     */
    @ConfigurationParameterChanger(propertyName = "channelTypes")
    protected void setChannelTypes(Map<String,String> channelTypes) throws DriverException {
        if (channelTypes == null || !channelTypes.keySet().equals(channels)) {
            throw new IllegalArgumentException("Invalid channelTypes map provided. It must contain entries for all channels.");
        }
        this.channelTypes = channelTypes;
    }

     /**
     * Sets channelUnits map
     *
     * @param channelUnits The map of channels to units 
     * @throws DriverException
     */
    @ConfigurationParameterChanger(propertyName = "channelUnits")
    protected void setChannelUnits(Map<String,String> channelUnits) throws DriverException {
        if (channelUnits == null || !channelUnits.keySet().equals(channels)) {
            throw new IllegalArgumentException("Invalid channelUnits map provided. It must contain entries for all channels.");
        }
        this.channelUnits = channelUnits;
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice() {
        if (host == null) {
            MonitorLogUtils.reportConfigError(log, name, "host", "is not specified");
        }
        if (maxSetPoints == null) {
            MonitorLogUtils.reportConfigError(log, name, "maxSetPoints", "is not specified");
        }
        if (maxSetPoints.length != M24C.NUM_LOOPS) {
            MonitorLogUtils.reportConfigError(log, name, "maxSetPoints", "has wrong length (must be 4)");
        }
        if (channelTypes == null) {
            MonitorLogUtils.reportConfigError(log, name, "channelTypes", "is not specified");
        }
        if (!channelTypes.keySet().equals(channels)) {
            MonitorLogUtils.reportConfigError(log, name, "channelTypes", "has invalid keyset");
        }
        if (channelUnits == null) {
            MonitorLogUtils.reportConfigError(log, name, "channelUnits", "is not specified");
        }
        if (!channelUnits.keySet().equals(channels)) {
            MonitorLogUtils.reportConfigError(log, name, "channelUnits", "has invalid keyset");
        }
        
        fullName = "Cryocon M24C (" + host + ")";
    }

    /**
     * Initializes the connection.
     */
    @Override
    protected void initialize() {
        try {
            m24c.open(connType, host, port);
            //initSensors();
            for (String channel : channelTypes.keySet()) {
                m24c.setType(channel.charAt(0), channelTypes.get(channel));
            }
            for (String channel : channelUnits.keySet()) {
                m24c.setUnit(channel.charAt(0), channelUnits.get(channel).charAt(0));
            }
            for (int l = 0; l < maxSetPoints.length; l++) {
                m24c.setMaxSetPoint(l + 1, maxSetPoints[l]);
            }
            setOnline(true);
            log.info("Connected to " + fullName);
        } catch (DriverException e) {
            if (!inited) {
                log.error("Error connecting to " + fullName + ": " + e);
            }
        }
        inited = true;
    }

    /**
     * Closes the connection.
     */
    @Override
    public void close() {
        try {
            m24c.close();
        } catch (DriverException e) {
            log.error("Error disconnecting from " + fullName + ": " + e);
        }
    }

    /**
     * Checks a channel's parameters for validity.
     *
     * @param name
     * @param hwChan
     * @param type
     * @param subtype
     * @return 
     * @throws Exception 
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            MonitorLogUtils.reportError(log, name, "hardware channel", hwChan);
        }
        return new int[]{0, 0};
    }

    /**
     * Initializes a channel.
     *
     * Not needed in this case.
     *
     * @param chan
     * @param type
     * @param subtype
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
    }

    /**
     * Reads a channel.
     *
     * @param chan
     * @param type
     * @return 
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        try {
            switch (chan) {
            case CHAN_CURR_TEMP:
                value = getTemp(current_channel); break;
            case CHAN_RUN_TEMP:
                value = getRunTemp(1); break;
            case CHAN_TEMP_STDDEV:
                value = getTempStdDev(); break;
            case CHAN_HTR_1:
                value = m24c.getHtrRead(1); break;
            case CHAN_HTR_2:
                value = m24c.getHtrRead(2); break;
            case CHAN_TEMP_A:
                value = getTemp("A"); break;
            case CHAN_TEMP_B:
                value = getTemp("B"); break;
            case CHAN_TEMP_C:
                value = getTemp("C"); break;
            case CHAN_TEMP_D:
                value = getTemp("D"); break;
            case CHAN_PID_P_1:
                value = m24c.getPID_P(1); break;
            case CHAN_PID_I_1:
                value = m24c.getPID_I(1); break;
            case CHAN_PID_D_1:
                value = m24c.getPID_D(1); break;
            case CHAN_PID_P_2:
                value = m24c.getPID_P(2); break;
            case CHAN_PID_I_2:
                value = m24c.getPID_I(2); break;
            case CHAN_PID_D_2:
                value = m24c.getPID_D(2); break;
            case CHAN_TEMP_CHNG:
                value = getTempChange(); break;
            case CHAN_SETPOINT_1:
                value = m24c.getSetPoint(1); break;
            case CHAN_SETPOINT_2:
                value = m24c.getSetPoint(2); break;
            }
        } catch (DriverException e) {
            log.error("Error reading channel " + chan + ": " + e);
            setOnline(false);
        }
        return (value);
    }

    /**
     * Handles alarm events.
     *
     * @param event The event type
     * @param parm The event parameter
     */
//    @Override
    public void processAlarm(int event, int parm) {
        log.error("Received alarm for event " + event + " where trip=" + Alarm.EVENT_TRIP + " and reset=" + Alarm.EVENT_RESET);
        switch (event) {

            case Alarm.EVENT_TRIP:
                if (parm == EVENT_ID_CRYO) {
                    log.error("Temperature alarm received! Please rectify and reset manually!");
                    cstate = TSState.cryostates.TRIPPED;
                }
                break;

            case Alarm.EVENT_RESET:
                if (parm == EVENT_ID_CRYO) {
                    cstate = TSState.cryostates.OK;
                }
                break;

            default:

        }
    }

   /**
     * Opens connection to a device.
     *
     * @param type
     * @param ident
     * @param param
     * @throws DriverException
     */
    @Deprecated
    @Override
    @Command(name = "open", description = "Open connection to Model 24C")
    public void open(
            @Argument(name = "type", description = "Communications type") int type,
            @Argument(name = "identity", description = "Device identifier") String ident,
            @Argument(name = "parameter", description = "Device parameter") int param)
            throws DriverException {
        m24c.open(type, ident, param);
    }

    /**
     * Displays Temp Control 24C identification.
     *
     * @return
     * @throws DriverException
     */
    @Command(name = "showident", description = "Show Temp Control 24C identification")
    public String showIdentification() throws DriverException {
        String[] ident = m24c.getIdentification();
        return   "Manufacturer:  " + ident[0] + "\n"
               + "Model name:    " + ident[1] + "\n"
               + "Serial number: " + ident[2] + "\n"
               + "F/W version:   " + ident[3];
    }

    /**
     * Get maximum set point for loop
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Command(type = Command.CommandType.QUERY, name = "getMaxSetPoint", description = "Retrieve MaxSetPoint for loop")
    @Override
    public double getMaxSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getMaxSetPoint(loop);
    }

    /**
     * Set MaxSetPoint for Loop [1-4]
     *
     * @param loop
     * @param temp
     * @throws DriverException
     */
    @Command(name = "setMaxSetPoint", description = "setMaxSetPoint for loop")
    @Override
    public void setMaxSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "temp", description = "temperature") double temp)
            throws DriverException {
        m24c.setMaxSetPoint(loop, temp);
    }

    /**
     * Get Source for Loop returns Channels A through D
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Command(name = "getSetPoint", description = "Retrieve setPoint for loop")
    @Override
    public double getSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getSetPoint(loop);
    }

    /**
     * Set SetPoint for Loop [1-4]
     *
     * @param loop
     * @param temp
     * @throws DriverException
     */
    @Command(name = "setSetPoint", description = "setSetPoint for loop")
    @Override
    public void setSetPoint(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "temp", description = "temperature") double temp)
            throws DriverException {
        m24c.setSetPoint(loop, temp);
    }

    /**
     * Get heater reading for specified loop
     *
     * @param  loop  The loop number
     * @return The heater reading
     * @throws DriverException
     */
    @Override
    @Command(name = "getHtrRead", description = "Retrieve heater reading for loop")
    public double getHtrRead(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getHtrRead(loop);
    }

    /**
     * Get [P]ID parameter for specified loop
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Override
    @Command(name = "getPID_P", description = "Retrieve PID parameter P for loop")
    public double getPID_P(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getPID_P(loop);
    }

    /**
     * Get P[I]D parameter for specified loop
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Override
    @Command(name = "getPID_I", description = "Retrieve PID parameter I for loop")
    public double getPID_I(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getPID_I(loop);
    }

    /**
     * Get PI[D] parameter for specified loop
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Override
    @Command(name = "getPID_D", description = "Retrieve PID parameter D for loop")
    public double getPID_D(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getPID_D(loop);
    }

    /**
     * Get Source for Loop returns Channels A through D
     *
     * @param loop
     * @return 
     * @throws DriverException 
     */
    @Override
    @Command(type = Command.CommandType.QUERY, name = "getLoopSource", description = "Retrieve source channel for loop")
    public char getLoopSource(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getSource(loop);
    }

    /**
     * Set Source Channel for Loop
     *
     * @param  loop  The loop number
     * @param  channel  The temperature channel (A-D)
     * @throws DriverException
     */
    @Override
    @Command(name = "setLoopSource", description = "Set source channel for loop")
    public void setLoopSource(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "channel", description = "channel [A-D]") String channel)
            throws DriverException {
        m24c.setSource(loop, channel.charAt(0));
    }

    /**
     * Get Heater Range for Loop [1-4]
     *
     * @param  loop  The loop number
     * @return The range string
     * @throws DriverException
     */
    @Command(name = "getHeaterRange", description = "Get heater range for loop")
    public String getHeaterRange(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getHeaterRange(loop);
    }

    /**
     * Set Heater Range for Loop [1-4]
     *
     * @param  loop  The loop number
     * @param  range  The range
     * @throws DriverException
     */
    @Override
    @Command(name = "setHeaterRange", description = "setHeaterRange for loop")
    public void setHeaterRange(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "range", description = "hi,low,mid") String range)
            throws DriverException {
        m24c.setHeaterRange(loop, range);
    }

    /**
     * Get maximum heater power for Loop [1-4]
     *
     * @param  loop  The loop number
     * @return The maximum heater power (%)
     * @throws DriverException
     */
    @Command(name = "getHeaterMax", description = "Get maximum heater power for loop")
    public double getHeaterMax(
            @Argument(name = "loop", description = "loop number [1-4]") int loop)
            throws DriverException {
        return m24c.getHeaterMax(loop);
    }

    /**
     * Set maximum heater power for Loop [1-4]
     *
     * @param  loop  The loop number
     * @param  power  The maximum heater power (%)
     * @throws DriverException
     */
    @Command(name = "setHeaterMax", description = "Set maximum heater power for loop")
    public void setHeaterMax(
            @Argument(name = "loop", description = "loop number [1-4]") int loop,
            @Argument(name = "power", description = "heater power (%)") double power)
            throws DriverException {
        m24c.setHeaterMax(loop, power);
    }

    /**
     * Get temperature reading for channels A through D
     *
     * @param channel The channel (A-D)
     * @return  The temperature
     * @throws DriverException
     */
    @Override
    @Command(type = Command.CommandType.QUERY, name = "getTemp", description = "Retrieve temperature info")
    public double getTemp(
            @Argument(name = "channel", description = "Temp Channel to read") String channel)
            throws DriverException {
        double value = m24c.getTemp(channel.charAt(0));
        if (value > -274.) { // don't include gurad values
            List<Double> tempData = tempDataMap.get(channel);
            List<Double> tempTime = tempTimeMap.get(channel);
            tempData.add(value);
            tempTime.add(System.currentTimeMillis() / 60000.0); //time in minutes
            if (tempData.size() > 120) {
                tempData.remove(0);
                tempTime.remove(0);
            }
        }

        return value;
    }

    /**
     * Set the temperature setting for acquisitions
     *
     * @param temp The temperature
     * @param cfgstate The configuration index
     */
    @Override
    @Command(name = "setRunTemp", description = "Sets the temp value for acquisition")
    public void setRunTemp(double temp, int cfgstate) {
        RUN_TEMP[cfgstate] = temp;
        cfg.setCryoTAcq(temp, cfgstate);
    }

    /**
     * Return the temperature setting for acquisitions
     *
     * @param cfgstate The configuration index
     * @return temp The temperature
     */
    @Override
    @Command(type = Command.CommandType.QUERY, name = "getRunTemp", description = "Returns the temp value for acquisition")
    public double getRunTemp(int cfgstate) {
        return (RUN_TEMP[cfgstate]);
    }

    /**
     * Return the standard deviation of the temperature values
     *
     * @return The standard deviation
     */
    @Command(type = Command.CommandType.QUERY, description = "Return the standard deviation of the temperature values")
    @Override
    public double getTempStdDev() {
        int ntemp = 0;
        double tempsum = 0., tempsqsum = 0.;
        for (double tempval : tempDataMap.get(current_channel)) {
            tempsum += tempval;
            tempsqsum += tempval * tempval;
            ntemp++;
        }
        double ovrstd = 0.;
        if (ntemp > 0) {
            ovrstd = Math.sqrt(tempsqsum / ntemp - (tempsum * tempsum) / (ntemp * ntemp));
        }

        return ovrstd;
    }

    /**
     * Return the rate of change in T/min of the temperature values
     *
     * @return The rate of change
     */
    @Command(type = Command.CommandType.QUERY, description = "Return the temperature change over the sample period in degrees/minute")
    @Override
    public double getTempChange() {
        double slope = 0.;
        List<Double> tempData = tempDataMap.get(current_channel);
        List<Double> tempTime = tempTimeMap.get(current_channel);
        if (tempData.size() > 1) {
            double deltaTime = tempTime.get(tempTime.size() - 1) - tempTime.get(0);
            if (deltaTime > 0.) {
                slope = (tempData.get(tempData.size() - 1) - tempData.get(0)) / deltaTime;
            }
        }
        return (slope);
    }

    /**
     * Ramps to the desired temperature over a given time and for nsteps
     *
     * @param duration
     * @param value
     * @param nsteps
     * @return
     */
    @Command(type = Command.CommandType.QUERY, name = "rampTemp", description = "ramp the temperature to the desired point")
    @Override
    public int rampTemp(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "value", description = "Temp to ramp to") double value,
            @Argument(name = "nsteps", description = "number of steps") int nsteps) {
        try {
            double tnow = getTemp(current_channel);
            double tstep = (value - tnow) / nsteps;
            int delta = (int) (1000. * duration / nsteps); // delta in msecs
            System.out.println("ramp T from " + Double.toString(tnow)
                    + " C\n to " + Double.toString(value)
                    + " C\n with delta Temp. (C) = " + Double.toString(tstep)
                    + "\n and time step (ms) = " + Integer.toString(delta)
                    + "\n Total duration (min) = " + duration / 60.);

            System.out.println("starting temperature ramp ...");

            double t = tnow;
            boolean trip0 = isTrip();
//        OKTOTALK = true; // override
            for (int istep = 0; istep < nsteps; istep++) {
                t += tstep;

                System.out.println("Setting T = " + t);

                if (!trip0 && isTrip()) {
                    log.error("STOPPING RAMP!!! IT LOOKS LIKE WE CAUSED A TRIP!");
                    clrTrip(); // we've done what was needed not lets get operational
                    break;
                }
                this.setSetPoint(current_loop, t);

                try {
                    Thread.currentThread().sleep(delta);//sleep for delta ms
                } catch (Exception ie) {
                }
            }
            log.info("temperature ramp completed");
        } catch (DriverException f) {
            log.error("Cryocon Device failed to ramp temperature!");
        }
        return 1;  // Why?
    }

    /**
     ** Ramps to the desired temperature over a given time and for nsteps
     *
     * @param duration
     * @param t_target
     * @param nsteps
     * @param iwloop
     * @param ioloop
     * @param wchan
     * @param ochan
     * @param delta_warm
     * @return
     */
    @Override
    @Command(type = Command.CommandType.QUERY, name = "rampTemp", description = "ramp the temperature to the desired point")
    public int rampTemp(@Argument(name = "duration", description = "number of seconds to ramp from current voltage to desired voltage") double duration,
            @Argument(name = "t_target", description = "Temp to ramp to") double t_target,
            @Argument(name = "nsteps", description = "number of steps") int nsteps,
            @Argument(name = "warmer_loop", description = "number of steps") int iwloop,
            @Argument(name = "other_loop", description = "number of steps") int ioloop,
            @Argument(name = "warm_temp_chan", description = "number of steps") String wchan,
            @Argument(name = "other_temp_chan", description = "number of steps") String ochan,
            @Argument(name = "delta_warm", description = "Temp to ramp to") double delta_warm) {

        try {
            double tnow = getTemp(wchan);
            double tstep = (t_target - tnow) / (double) nsteps;
            int delta = (int) (1000. * duration / (double) nsteps); // delta in msecs
            System.out.println("ramp T from " + Double.toString(tnow)
                    + " C\n to " + Double.toString(t_target)
                    + " C\n with delta Temp. (C) = " + Double.toString(tstep)
                    + "\n and time step (ms) = " + Integer.toString(delta)
                    + "\n Total duration (min) = " + duration / 60.);

            System.out.println("starting temperature ramp ...");
            double t = tnow;
            double t_other = t;
            boolean trip0 = isTrip();
            double cryotol = cfg.getCryoTAcqTol();
//                while ( Math.abs((tnow = getTemp(wchan))-t_target)>cryotol ||  Math.abs((tnow = getTemp(ochan))-t_target)>cryotol ) {
            while ((tstep > 0. && t < (t_target - tstep)) || (tstep < 0. && t > (t_target + tstep))
                    || (tstep > 0. && t_other < (t_target - tstep)) || (tstep < 0. && t_other > (t_target + tstep))) {
                if ((tstep > 0. && t < (t_target - tstep)) || (tstep < 0. && t > (t_target + tstep))) {
                    t += tstep;

                    double otnow = getTemp(ochan);
                    if (t_other < (t - delta_warm) && otnow < (tnow - delta_warm)) {
                        t_other += tstep;
                    }
                } else if ((tstep > 0. && t_other < (t_target - tstep)) || (tstep < 0. && t_other > (t_target + tstep))) {
                    t_other += tstep;
                }

                if (!trip0 && isTrip()) {
                    log.error("STOPPING RAMP!!! IT LOOKS LIKE WE CAUSED A TRIP!");
                    clrTrip(); // we've done what was needed now lets get operational
                    break;
                }
                System.out.println("Setting warm loop temp to " + t + " C");
                this.setSetPoint(iwloop, t);
                System.out.println("Setting other loop temp to " + t_other + " C");
                this.setSetPoint(ioloop, t_other);

                try {
                    Thread.currentThread().sleep(delta);//sleep for delta ms
                } catch (Exception ie) {
                }
            }
            System.out.println("Setting warm loop temp to " + t + " C");
            this.setSetPoint(iwloop, t_target);
            System.out.println("Setting other loop temp to " + t_other + " C");
            this.setSetPoint(ioloop, t_target);
            log.info("temperature ramp completed");
        } catch (DriverException f) {
            log.error("Cryocon Device failed to ramp temperature!");
        }
        return (1);
    }

    /**
     * Get Temperature Units Channels A through D *
     * @param channel
     * @return 
     * @throws DriverException 
     */
    @Command(type = Command.CommandType.QUERY, name = "getUnit", description = "Retrieve temperature units")
    @Override
    public char getUnit(
            @Argument(name = "channel", description = "Temp Channel to read") String channel)
            throws DriverException {
        return m24c.getUnit(channel.charAt(0));
    }

    /**
     * Set Temperature Units for Channels A through D
     *
     * @param channel The temperature channel (A-D)
     * @param unit The units string
     * @throws DriverException
     */
    @Override
    @Command(name = "setUnit", description = "Set temperature units")
    public void setUnit(
            @Argument(name = "channel", description = "Temp Channel to set") String channel,
            @Argument(name = "units", description = "Units in [K|C|F|S]") String unit)
            throws DriverException {
        m24c.setUnit(channel.charAt(0), unit.charAt(0));
    }
    
    /**
     * Get sensor type for a temperature channel
     *
     * @param channel The temperature channel (A-D)
     * @return The sensor type string
     * @throws DriverException
     */
    @Command(name = "getType", description = "Get sensor type")
    public String getType(
            @Argument(name = "channel", description = "Temp Channel to set") String channel)
            throws DriverException {
        return m24c.getType(channel.charAt(0));
    }

    /**
     * Set sensor type for a temperature channel
     *
     * @param channel The temperature channel (A-D)
     * @param typ The sensor type
     * @throws DriverException
     */
    @Override
    @Command(name = "setType", description = "Set sensor type : DIODE|ACR|PTC100|PTC1K|NTC10UA|TC70|NONE")
    public void setType(
            @Argument(name = "channel", description = "Temp Channel to set") String channel,
            @Argument(name = "type", description = "Calib. curve type [DIODE|ACR|PTC100|PTC1K|NTC10UA|TC70|NONE]") String typ)
            throws DriverException {
        m24c.setType(channel.charAt(0), typ);
    }

    /**
     * Gets whether OTD is enabled
     *
     * @return Whether OTD is enabled
     * @throws DriverException
     */
    @Command(name = "isOtdEnabled", description = "Get whether OTD system enabled")
    public boolean isOtdEnabled() throws DriverException {
        return m24c.isOtdEnabled();
    }

    /**
     * Sets OTD enabled state
     *
     * @param enab Whether to enable OTD system
     * @throws DriverException
     */
    @Command(name = "enableOtd", description = "Set OTD system enabled state")
    public void enableOtd(@Argument(name = "enab", description = "Whether to enable OTD system") boolean enab)
                          throws DriverException {
        m24c.enableOtd(enab);
    }

    /**
     * Gets OTD source channel
     *
     * @return The source channel (A-D)
     * @throws DriverException
     */
    @Command(name = "getOtdSource", description = "Get source channel for the OTD system")
    public char getOtdSource() throws DriverException {
        return m24c.getOtdSource();
    }

    /**
     * Sets OTD source channel
     *
     * @param channel The OTD source channel (A-D)
     * @throws DriverException
     */
    @Command(name = "setOtdSource", description = "Set source channel for the OTD system")
    public void setOtdSource(@Argument(name = "channel", description = "channel [A-D]") String channel)
                             throws DriverException {
        m24c.setOtdSource(channel.charAt(0));
    }

    /**
     * Gets OTD temperature limit
     *
     * @return The temperature limit
     * @throws DriverException
     */
    @Command(name = "getOtdTemp", description = "Get temperature limit for the OTD system")
    public double getOtdTemp() throws DriverException {
        return m24c.getOtdTemp();
    }

    /**
     * Sets OTD temperature limit
     *
     * @param  temp The temperature limit
     * @throws DriverException
     */
    @Command(name = "setOtdTemp", description = "Set temperature limit for the OTD system")
    public void setOtdTemp(@Argument(name = "temp", description = "Temperature") double temp)
                           throws DriverException {
        m24c.setOtdTemp(temp);
    }

    /**
     * Put loops in control mode
     *
     * @throws DriverException
     */
    @Override
    @Command(name = "setToControl", description = "put loops in control mode")
    public void setToControl() throws DriverException {
        m24c.setToControl();
    }

    /**
     * Take loops out of control mode
     *
     * @throws DriverException
     */
    @Command(name = "stopControl", description = "Take loops out of control mode")
    public void stopControl() throws DriverException {
        m24c.stopControl();
    }

    /**
     * Get control mode
     *
     * @return Whether in control mode
     * @throws DriverException
     */
    @Override
    @Command(name = "isInControl", description = "is the CryoCon24c in control mode")
    public boolean isInControl() throws DriverException {
        return m24c.isInControl();
    }

    /**
     * Gets the device state
     *
     * @return The device state
     */
    @Override
    @Command
    public int getState() {
        return cstate.ordinal();
    }

    /**
     * Sets the device state
     *
     * @param istate The state to set
     */
    @Override
    @Command
    public void setState(int istate) {
        cstate = TSState.cryostates.values()[istate];
    }

    @Override
    @Command
    public String getCurrent_channel() {
        return current_channel;
    }

    @Override
    @Command
    public String getCurrent_channel2() {
        return current_channel2;
    }

    @Override
    @Command
    public void setCurrent_channel(String current_channel) {
        this.current_channel = current_channel;
    }

    @Override
    @Command
    public void setCurrent_channel2(String current_channel2) {
        this.current_channel2 = current_channel2;
    }

    @Override
    @Command
    public int getCurrent_loop() {
        return current_loop;
    }

    @Override
    @Command
    public void setCurrent_loop(int current_loop) {
        this.current_loop = current_loop;
    }

    boolean isTrip() {
        return cstate == TSState.cryostates.TRIPPED;
    }

    boolean setTrip() {
        cstate = TSState.cryostates.TRIPPED;
        return false;
    }

    boolean clrTrip() {
        cstate = TSState.cryostates.OK;
        return false;
    }

    // THE FOLLOWING METHODS HAVE BEEN SUPERCEDED AND WILL SOON BE REMOVED
    @Override
    public double get_init_temp(int cfgstate) // soon to be deprecated
    {
        return this.RUN_TEMP[cfgstate];
    }

}
