package org.lsst.ccs.subsystem.teststand;

import org.lsst.ccs.subsystem.teststand.limits.AlgorithmFactory;
import org.lsst.ccs.subsystem.teststand.limits.LimitAlgorithm;
import java.util.List;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.subsystem.teststand.limits.ChannelLimits;
import org.lsst.ccs.subsystem.teststand.limits.Limits;
import org.lsst.ccs.subsystem.teststand.limits.LimitsInterface;
import org.lsst.ccs.subsystem.teststand.limits.TransitionAlgorithm;

/**
 * A class which encapsulates the vacuum configuration. Typically a different
 * configuration will be loaded when the vacuum target changes.
 *
 * @author tonyj
 */
public class VacuumConfiguration {

    /**
     * Cold set point
     */
    @ConfigurationParameter(category = "vacuum")
    double pressureSetPoint;

    /**
     * Cold set point limits
     */
    @ConfigurationParameter(category = "vacuum")
    List<Double> pressureSetPointLimits;

    /**
     * Algorithm to be used to compute limits, typically based on set point.
     */
    @ConfigurationParameter(category = "vacuum")
    String pressureMonitoringLimitAlgorithm;

    /**
     * Parameters needed for limit algorithms.
     */
    @ConfigurationParameter(category = "vacuum")
    List<Double> pressureMonitoringLimitAlgorithmParameters;

    /**
     * Algorithm to be used to handle transition.
     */
    @ConfigurationParameter(category = "vacuum")
    String pressureMonitoringTransitionAlgorithm;

    /**
     * Parameters needed for transition algorithms.
     */
    @ConfigurationParameter(category = "vacuum")
    List<Double> pressureMonitoringTransitionAlgorithmParameters;

    LimitAlgorithm createVacuumLimitsAlgorithm(double nominalInitialValue, Limits initialLimits) {
        LimitAlgorithm la = AlgorithmFactory.createLimitAlgorithm(pressureMonitoringLimitAlgorithm);
        Limits setPointLimits = new Limits(pressureSetPointLimits);
        la.init(pressureMonitoringLimitAlgorithmParameters, nominalInitialValue, initialLimits, pressureSetPoint, setPointLimits);
        return la;
    }

    TransitionAlgorithm createVacuumTransitionAlgorithm(LimitAlgorithm limitsAlgorithm, double initialPressure, ChannelLimits limits, long startTime) {
        if (pressureMonitoringTransitionAlgorithm != null && !pressureMonitoringTransitionAlgorithm.isEmpty()) {
            TransitionAlgorithm ta = AlgorithmFactory.createTransitionAlgorithm(pressureMonitoringTransitionAlgorithm);
            ta.init(pressureMonitoringTransitionAlgorithmParameters, limitsAlgorithm, initialPressure, limits, startTime);
            return ta;
        } else { 
            return null;
        }
    }

    @Override
    public String toString() {
        return "VacuumConfiguration{" + "pressureSetPoint=" + pressureSetPoint + ", pressureMonitoringLimitAlgorithm=" + pressureMonitoringLimitAlgorithm + ", pressureMonitoringLimitAlgorithmParameters=" + pressureMonitoringLimitAlgorithmParameters + ", pressureMonitoringTransitionAlgorithm=" + pressureMonitoringTransitionAlgorithm + ", pressureMonitoringTransitionAlgorithmParameters=" + pressureMonitoringTransitionAlgorithmParameters + '}';
    }
}
