package org.lsst.ccs.subsystem.teststand;

import org.lsst.ccs.AlertService;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.subsystem.teststand.alerts.TS7Alerts;
import org.lsst.ccs.subsystem.teststand.data.TS7Outlets;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.subsystem.power.states.RebPowerState;

/**
 * A class with information about TS7 alarms
 *
 * @author The LSST CCS Team
 */
public class TS7AlertHandler implements HasLifecycle, Monitor.AlarmHandler, ClearAlertHandler {

    private static final Logger LOG = Logger.getLogger("org.lsst.ccs.subsystem.ts7.alarms");

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    //The TS7 subsystem
    @LookupField(strategy = LookupField.Strategy.TREE)
    private TS7 ts7;
    
        
    @Override
    public boolean processAlarm(int event, int parm, String cause, String alarmName) {
        
        //parm defines the type of alarm as defined in groovy, event is the type of limit boundary crossing
    	
        if ( parm == TS7Alerts.PRESSURE_TOO_HIGH.ordinal() ) {
            alertService.raiseAlert(TS7Alerts.PRESSURE_TOO_HIGH.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
        	switch (event) {
                case Alarm.EVENT_TRIP:                    
                    //Alarm triggered
                    try { //Shut the valve, then fork off the rest of the response
                        ts7.turnOutletOff(TS7Outlets.VACUUMVALVE);
                        LOG.info("Shutting (Turning off) vacuum valve "+TS7Outlets.VACUUMVALVE.getOutletName());
                    } catch (DriverException de) {
                        //How to handle the exception here?
                        //TO-DO Raise another alert?
                    	LOG.severe("Failed to turn off VACUUMVALVE outlet -- manual intervention needed" + de);
                    }
                    Thread t = new Thread(new VacuumAlarmResponse());
                    t.start();
                    break;
                case Alarm.EVENT_LIMBO_FROM_ERROR:
                    //In Warning band from Alarm
                    break;
                case Alarm.EVENT_LIMBO_FROM_GOOD:
                    //In Warning band from Nominal
                    break;
                case Alarm.EVENT_RESET:
                    //Back to Nominal
                    break;
            }
            return true;
        } else if ( parm == TS7Alerts.PRESSURE_TOO_LOW.ordinal() ) {
            //This is a vacuum related alarm, no action needed but alerts distributed.
        	// The expectation is that the gauge may be having an error and needs attention
            alertService.raiseAlert(TS7Alerts.PRESSURE_TOO_LOW.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
            switch (event) {
                //The cases we don't care can be removed
                case Alarm.EVENT_TRIP:                    
                    //Warning --> Alarm
                    break;
                case Alarm.EVENT_LIMBO_FROM_ERROR:
                    //Alarm --> Warning
                    break;
                case Alarm.EVENT_LIMBO_FROM_GOOD:
                    //Nominal --> Warning
                    break;
                case Alarm.EVENT_RESET:
                    // * --> Nominal
                    break;
            }
            return true;
        } else if ( parm == TS7Alerts.COLD_PLATE_TEMPERATURE_TOO_LOW.ordinal() ) {
            // This could be run-away cooling and potentially bad, but the time-scale
        	// is a few hours.  So alert distribution at warning is first defense.
        	// At alarm level, we'd better turn off the refrigerators
            alertService.raiseAlert(TS7Alerts.COLD_PLATE_TEMPERATURE_TOO_LOW.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
            switch (event) {
                case Alarm.EVENT_TRIP:                    
                    //Warning --> Alarm
                	// Launch thread that turns off NF-55-1&2, PT-30 and REBs
                	LOG.info("COLD_PLATE_TEMPERATURE_TOO_LOW: Turn off Polycold chillers...");
                	Thread t = new Thread(new ColdPlateAlarmLowResponse());
                    t.start();
                    break;
                case Alarm.EVENT_LIMBO_FROM_ERROR:
                    //Alarm --> Warning
                    break;
                case Alarm.EVENT_LIMBO_FROM_GOOD:
                    //Nominal --> Warning
                    break;
                case Alarm.EVENT_RESET:
                    // * --> Nominal
                    break;
            }
            return true;
        } else if ( parm == TS7Alerts.CRYO_PLATE_TEMPERATURE_TOO_LOW.ordinal() ) {
            //  Response to turn off PT-30.  Cold plate can stay as is unless it triggers its own alert.
            alertService.raiseAlert(TS7Alerts.CRYO_PLATE_TEMPERATURE_TOO_LOW.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
            switch (event) {
                case Alarm.EVENT_TRIP:                    
                    //Warning --> Alarm
                	LOG.info("CRYO_PLATE_TEMPERATURE_TOO_LOW: Turn off Polycold chiller...");
                	Thread t = new Thread(new CryoPlateAlarmLowResponse());
                    t.start();
                	break;
                case Alarm.EVENT_LIMBO_FROM_ERROR:
                    //Alarm --> Warning
                    break;
                case Alarm.EVENT_LIMBO_FROM_GOOD:
                    //Nominal --> Warning
                    break;
                case Alarm.EVENT_RESET:
                    // * --> Nominal
                    break;
            }
            return true;
        } else if ( parm == TS7Alerts.COLD_PLATE_TEMPERATURE_TOO_HIGH.ordinal() ) {
            // This could occur if the transition algorithm does not keep up which is
        	// mostly benign but it could also occur in the event of a vacuum problem
        	// but not much to do about that, if it wasn't caught in that section.
            alertService.raiseAlert(TS7Alerts.COLD_PLATE_TEMPERATURE_TOO_HIGH.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
            switch (event) {
                case Alarm.EVENT_TRIP:                    
                    //Warning --> Alarm
                	LOG.info("COLD_PLATE_TEMPERATURE_TOO_HIGH: Turning off heat sources");
                	if (ts7.getCurrentRebsPowerState() == RebPowerState.OFF) {
                        LOG.info("REB PowerState is OFF -- turn off outlet anyway");
                    }
                    if (ts7.isAlertResponseEnabled()) {
                            try {
                                LOG.info("Turning off REB 48V power supply outlet: " + TS7Outlets.REB48VOLTPOWERSUPPLY.getOutletName());
                                ts7.turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY);
                            } catch (DriverException e) {
                                LOG.error("Error powering off REB power supply: " + e);
                            }
                        } else {
                            LOG.info("Powering off REB power supply was disabled");
                        }
                	// Need to verify that OTP circuit is enabled on Cryocon and active
                	// value to 0%.  (N.B. this needs reset on startup)
                	LOG.info("COLD_PLATE_TEMPERATURE_TOO_HIGH: Alert response initiated");
                    break;
                case Alarm.EVENT_LIMBO_FROM_ERROR:
                    //Alarm --> Warning
                    break;
                case Alarm.EVENT_LIMBO_FROM_GOOD:
                    //Nominal --> Warning
                    break;
                case Alarm.EVENT_RESET:
                    // * --> Nominal
                    break;
            }
            return true;
        } else if ( parm == TS7Alerts.CRYO_PLATE_TEMPERATURE_TOO_HIGH.ordinal() ) {
        	// This could occur if the transition algorithm does not keep up which is
        	// mostly benign but it could also occur in the event of a vacuum problem
        	// but not much to do about that, if it wasn't caught in that section.
            alertService.raiseAlert(TS7Alerts.CRYO_PLATE_TEMPERATURE_TOO_HIGH.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
            switch (event) {
                case Alarm.EVENT_TRIP: 
                	//Warning --> Alarm
                	LOG.info("CRYO_PLATE_TEMPERATURE_TOO_HIGH: Turning off heat sources");
                	if (ts7.getCurrentRebsPowerState() == RebPowerState.OFF) {
                        LOG.info("REB PowerState is OFF -- turn off outlet anyway");
                    }
                    if (ts7.isAlertResponseEnabled()) {
                            try {
                                LOG.info("Turning off REB 48V power supply outlet: " + TS7Outlets.REB48VOLTPOWERSUPPLY.getOutletName());
                                ts7.turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY);
                            } catch (DriverException e) {
                                LOG.error("Error powering off REB power supply: " + e);
                            }
                        } else {
                            LOG.info("Powering off REB power supply was disabled");
                        }
                	// Need to verify that OTP circuit is enabled on Cryocon and active
                	// value to 0%.  (N.B. this needs reset on startup)
                	LOG.info("CRYO_PLATE_TEMPERATURE_TOO_HIGH: Turn off REB PS if powerState is not OFF, verify Cryo OTP is enabled");
                    break;
                case Alarm.EVENT_LIMBO_FROM_ERROR:
                    //Alarm --> Warning
                    break;
                case Alarm.EVENT_LIMBO_FROM_GOOD:
                    //Nominal --> Warning
                    break;
                case Alarm.EVENT_RESET:
                    // * --> Nominal
                    break;
            }
            return true;
        }
        return false;
    } 
    
    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert) {
        for ( TS7Alerts ts7Alert : TS7Alerts.values() ) {
            if ( ts7Alert.getAlertId().equals(alert.getAlertId() ) ) {
                Alarm a = getAlarmById(ts7Alert.ordinal());
                if ( a != null ) {
                    a.clearState();
                    return ClearAlertCode.CLEAR_ALERT;
                }
            }
        }
        return ClearAlertCode.UNKWNOWN_ALERT;
    }
    
    private static AlertState getAlarmStateFromEvent(int event) {
        switch (event) {
            //The High Alarm treshold has been crossed.
            case Alarm.EVENT_TRIP:
                return AlertState.ALARM;
            case Alarm.EVENT_LIMBO_FROM_ERROR:
                return AlertState.WARNING;
            case Alarm.EVENT_LIMBO_FROM_GOOD:
                return AlertState.WARNING;
            case Alarm.EVENT_RESET:
                return AlertState.NOMINAL;
        }
        throw new IllegalArgumentException("Unknown event: id " + event);
    }
 
    class VacuumAlarmResponse implements Runnable {
        @Override
        public void run() {
            try {
                LOG.info("Turning off cold plate refrigerator 1 " + TS7Outlets.COLDPLATEREFRIGERATOR_1.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_1);
                } else {
                    LOG.info("The action of turning off the cold plate refrigerator 1 was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off ColdPlateRefrigerator_1 outlet -- manual intervention needed");
            }
            sleep(2000); // TODO evaluate a better way for this
            try {
                LOG.info("Turning off cold plate refrigerator 2 " + TS7Outlets.COLDPLATEREFRIGERATOR_2.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_2);
                } else {
                    LOG.info("The action of turning off the cold plate refrigerator 2 was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off ColdPlateRefrigerator_2 outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOG.info("Turning off cryo plate refrigerator " + TS7Outlets.CRYOPLATEREFRIGERATOR.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.CRYOPLATEREFRIGERATOR);
                } else {
                    LOG.info("The action of turning off the cryo plate refrigerator was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off CryoPlateRefrigerator outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOG.info("Turning off turbo pump " + TS7Outlets.TURBOPUMP.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.TURBOPUMP); // no way to vent properly to slow down
                } else {
                    LOG.info("The action of turning off the turbo pump was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off Turbo Pump outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOG.info("Turning off roughing pump " + TS7Outlets.ROUGHINGPUMP.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.ROUGHINGPUMP); // Note that this also closes the second valve
                } else {
                    LOG.info("The action of turning off the roughing pump was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off Roughing Pump outlet -- manual intervention needed");
            }
            try {
                LOG.info("Turning off reb power supply " + TS7Outlets.REB48VOLTPOWERSUPPLY.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY); // TODO: Revise this later
                } else {
                    LOG.info("The action of turning off the reb power supply was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off REB 48V supply outlet -- manual intervention needed");
            }
        }
    }
    
    class ColdPlateAlarmLowResponse implements Runnable {
        @Override
        public void run() {
            try {
                LOG.info("Turning off cold plate refrigerator 1 " + TS7Outlets.COLDPLATEREFRIGERATOR_1.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_1);
                } else {
                    LOG.info("The action of turning off the cold plate refrigerator 1 was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off ColdPlateRefrigerator_1 outlet -- manual intervention needed");
            }
            sleep(2000); // TODO evaluate a better way for this
            try {
                LOG.info("Turning off cold plate refrigerator 2 " + TS7Outlets.COLDPLATEREFRIGERATOR_2.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.COLDPLATEREFRIGERATOR_2);
                } else {
                    LOG.info("The action of turning off the cold plate refrigerator 2 was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off ColdPlateRefrigerator_2 outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOG.info("Turning off cryo plate refrigerator " + TS7Outlets.CRYOPLATEREFRIGERATOR.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.CRYOPLATEREFRIGERATOR);
                } else {
                    LOG.info("The action of turning off the cryo plate refrigerator was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off CryoPlateRefrigerator outlet -- manual intervention needed");
            }
            sleep(2000);
            try {
                LOG.info("Turning off reb power supply " + TS7Outlets.REB48VOLTPOWERSUPPLY.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.REB48VOLTPOWERSUPPLY); // TODO: Revise this later
                } else {
                    LOG.info("The action of turning off the reb power supply was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off REB 48V supply outlet -- manual intervention needed");
            }
        }
    }
    
    class CryoPlateAlarmLowResponse implements Runnable {
        @Override
        public void run() {
            try {
                LOG.info("Turning off cryo plate refrigerator " + TS7Outlets.CRYOPLATEREFRIGERATOR.getOutletName());
                if (ts7.isAlertResponseEnabled()) {
                    ts7.turnOutletOff(TS7Outlets.CRYOPLATEREFRIGERATOR);
                } else {
                    LOG.info("The action of turning off the cryo plate refrigerator was disabled");
                }
            } catch (DriverException e) {
                LOG.severe("Failed to turn off CryoPlateRefrigerator outlet -- manual intervention needed" + e);
            }
            sleep(2000);
        }
    }
    
    private static void sleep(long msec) {
        try {
            Thread.sleep(msec);
        } catch (InterruptedException ie) {
            throw new RuntimeException("Could not sleep",ie);
        }
    }
    
}
