package org.lsst.ccs.subsystem.teststand;

import java.util.Map;
import java.util.HashMap;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.orielshutter.OrielShutter;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;
import org.lsst.ccs.subsystem.teststand.data.TSState;

/**
 ***************************************************************************
 **
 * Oriel Shutter device class for the teststand subsystem
 * 
 * * @author: Homer Neal
 * **************************************************************************
 */
public class OrielShutterDevice extends Device implements ShutterDevice {

    private final static Map<String, Integer> typeMap = new HashMap<>();

    static {
        typeMap.put("OPENSHUT", Channel.TYPE_SWITCH);
    }

    private OrielShutter oshut = null;

    private boolean isconnected = false;
    
    private int openshut = -1 ; // is the shutter open or shut

    TSConfig cfg = new TSConfig();

    private String outname = "";

    TSState.pwrstates kstate = TSState.pwrstates.NOTCONFIGURED;

    boolean failedToInitialize = false;

    /**
     ***************************************************************************
     **
     ** OrielShutterDevice constructor - put Oriel Shutter in initial subsystem state
     *
     ** @param host device address
     ** @param port port or baud * * @param runBias CCD bias voltage while
     * running *
     * @param addr
     * @throws org.lsst.ccs.drivers.commons.DriverException
     * **************************************************************************
     */
    public OrielShutterDevice(String host, int port, int addr)
            throws DriverException {
        isconnected = true;
        try {
            oshut = new OrielShutter();

            if (oshut != null) {
                oshut.openftdi(host, port, addr); // open connection
            }
        } catch (DriverException f) {
            System.out.println("Failed to open connection to Oriel Shutter device!");
            isconnected = false;
        }
    }

    /**
     ***************************************************************************
     **
     ** Constructor *
     * **************************************************************************
     */
    public OrielShutterDevice() {
        try {
            oshut = new OrielShutter();
        } catch (DriverException f) {
            System.out.println("Failed to create to Oriel Shutter device instance!");
        }

    }

    /**
     ***************************************************************************
     **
     ** Performs configuration. *
     * **************************************************************************
     */
    @Override
    protected void initDevice() {
        fullName = "Oriel Shutter module";
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            if (oshut != null) {
                oshut.close();
            }
            kstate = TSState.pwrstates.NOTCONFIGURED;
        } catch (DriverException e) {
            log.error("Oriel Shutter device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the device
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        if (isconnected) {
            log.debug("Oriel Shutter device is already initialized and connected to:");
            try {
                log.debug(oshut.getIdent());
            } catch (Exception e) {
                if (!failedToInitialize) {
                    log.error("Oriel Shutter device failed to respond to ident request! The initialization has FAILED.");
                    failedToInitialize = true;
                }
                isconnected = false;
            }
            log.debug("Setting device online.");
            setOnline(true);
            kstate = TSState.pwrstates.OK;
        } else {
            if (!failedToInitialize) {
                log.error("Tried to initialize unconnected LED device!");
                failedToInitialize = true;
            }

        }
    }

    /**
     ***************************************************************************
     **
     ** Opens FTDI connection to a device. *
     * **************************************************************************
     */
    @Command(name = "openftdi", description = "Open FTDI connection to device")
    public String openftdi(@Argument(name = "serialname", description = "Serial device name") String host,
            @Argument(name = "baud", description = "baud rate") int port,
            @Argument(name = "addr", description = "address offset") int addr
    ) throws DriverException {
        oshut.openftdi(host, port,addr);
        kstate = TSState.pwrstates.OK;
        return oshut.getIdent();
    }

  
    /**
     ***************************************************************************
     **
     ** opens the shutter *
     * **************************************************************************
     */
    @Command(name = "openShutter", description = "open shutter")
    public void openShutter() {
        try {
            if (oshut != null) {
                oshut.openShutter();
                openshut = 1;
            }
        } catch (DriverException f) {
            log.error("Failed open/close the Oriel Shutter");
        }
    }

    /**
     ***************************************************************************
     **
     ** closes the shutter *
     * **************************************************************************
     */
    @Command(name = "closeShutter", description = "close shutter")
    public void closeShutter() {
        try {
            if (oshut != null) {
                oshut.closeShutter();
                openshut = 0;
            }
        } catch (DriverException f) {
            log.error("Failed open/close the Oriel Shutter");
        }
    }


    /**
     ***************************************************************************
     **
     ** Converts a boolean to on/off. *
     * **************************************************************************
     */
    private String getOnOff(boolean on) {
        return on ? "on" : "off";
    }

    /**
     ***************************************************************************
     **
     ** Checks a channel's parameters for validity. *
     * **************************************************************************
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type,
            String subtype)
            throws Exception {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            MonitorLogUtils.reportError(log, name, "type", type);
        } else if (iType != Channel.TYPE_SWITCH) {
            MonitorLogUtils.reportError(log, name, "Wrong channel type specified! type = ", type);
            Exception e;
        }

        return new int[]{iType, 0};
    }

    /**
     ***************************************************************************
     **
     ** Initializes a channel. *
     * **************************************************************************
     */
    @Override
    protected void initChannel(int chan, int type, int subtype) {
        try {
            if (type == Channel.TYPE_SWITCH) {
                log.debug("setting channel online - chan=" + chan);
                setOnline(true);
            }
        } catch (Exception e) {
            log.error("Error configuring channel type " + type + ": " + e);
        }
    }

    @Command(name = "reset", description = "reset trips etc...")
    public void reset() throws DriverException {
        if (oshut != null) {
            oshut.reset();
        }
        kstate = TSState.pwrstates.OFF;

    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        log.debug("Oriel ShutterDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            if (type == Channel.TYPE_SWITCH) {
                value = openshut;
            }
        } catch (Exception e) {
            log.debug("Error reading channel type " + type + ": " + e);
        }
        return (value);
    }

    @Command(name = "setState", description = "set Oriel Shutter device status")
    public void setState(int istate) {
        kstate = TSState.pwrstates.values()[istate];
    }

    @Command(name = "getState", description = "get Oriel Shutter device status")
    public int getState() {
        return (kstate.ordinal());
    }

}
