package org.lsst.ccs.subsystem.teststand;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.newportlamp.NewportLamp;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.teststand.data.TSConfig;

/**
 *
 * Device class for the NewportLamp driver
 *
 * @author homer
 */
public class NewportLampDevice extends Device implements LampDevice {

    private NewportLamp lampDev;
    private String lampType = "undef";
    TSConfig cfg = new TSConfig();
    private double[] runlampcurr = new double[cfg.MAXSTATES];

    private int itype = Ascii.CONN_TYPE_SERIAL, port;
    private String host;
    
    boolean isConnected = false;
    boolean failedToInitialize = false;

    public NewportLampDevice(String host, int port) throws DriverException {
        this(Ascii.CONN_TYPE_SERIAL, host, port);
    }

    public NewportLampDevice(int itype, String host, int baud) throws DriverException {
        this.itype = itype;
        this.host = host;
        this.port = port;

        try {
            lampDev = new NewportLamp();
        } catch (DriverException f) {
            System.out.println("Failed to create NewportLamp device!");
        }
    }

    /**
     ***************************************************************************
     **
     ** isLampPowerOn - Is the NewportLamp Device on? *
     * **************************************************************************
     */
    @Command(name = "isLampPowerOn", description = "returns whether the NewportLamp device is active")
    public boolean isLampPowerOn() {
        boolean state = (1 == 0);
        try {
            state = lampDev.getState();
        } catch (DriverException e) {
            log.error("NewportLamp device failed to close!!!");
        }
        return (state);
    }

    /**
     ***************************************************************************
     **
     ** enables/disable the power to the lamp *
     * **************************************************************************
     */
    @Command(name = "setLampPowerEnable", description = "enables/disables the power to the lamp from the NewportLamp device")
    public void setLampPowerEnable(
            @Argument(name = "enable", description = "enables/disables lamp power") boolean on
    ) {

        try {
            log.debug("Setting lamp power state to "+on);
            lampDev.setState(on);
        } catch (DriverException e) {
            throw new RuntimeException("NewportLamp device failed to set lamp power enable/disable!!!",e);
        }
    }

    /**
     ***************************************************************************
     *
     * returns the current read from the NewportLamp device
     * 
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getLampCurrent", description = "returns the #amps read from the NewportLamp device")
    public double getLampCurrent() {
        double current = 0.0;

        try {
            if (isConnected) current = lampDev.getAmps();
        } catch (DriverException e) {
            log.error("NewportLamp device failed to return current reading!!!");
        }
        return (current);
    }

    /**
     ***************************************************************************
     **
     ** sets the current for the NewportLamp device *
     * **************************************************************************
     */
    @Command(name = "setLampCurrent", description = "(not implemented) sets the current for the NewportLamp device")
    public void setLampCurrent(double current) {
        log.error("Purposely not implemented");
        /*
        try {
            if (isConnected) lampDev.setAmps();
        } catch (DriverException e) {
            log.error("NewportLamp device failed to return current reading!!!");
        
        }
        */
        return;
    }

    /**
     ***************************************************************************
     *
     * returns the current read from the NewportLamp device
     * 
     * **************************************************************************
     */
    @Command(type = Command.CommandType.QUERY, name = "getLampPower", description = "returns the #amps read from the NewportLamp device")
    public double getLampPower() {
        double pwr = 0.0;

        try {
            if (isConnected) pwr = lampDev.getWatts();
        } catch (DriverException e) {
            log.error("NewportLamp device failed to return power reading!!!");
        }
        return (pwr);
    }

    /**
     ***************************************************************************
     **
     ** returns the current read from the NewportLamp device *
     * **************************************************************************
     */
    @Command(name = "setLampPower", description = "(not implimented) sets the power for the NewportLamp device")
    public void setLampPower(double current) {
            log.error("Purposely not implemented!!!");
  /*
        try {
            if (isConnected) lampDev.setAmps();
        } catch (DriverException e) {
            log.error("NewportLamp device failed to return current reading!!!");
        
        }
        */
        return;
    }

    /**
     ***************************************************************************
     **
     ** defines the lamp type *
     * **************************************************************************
     */
    @Command(name = "setType", description = "define lamp type")
    public void setType(String type) {
        this.lampType = type;
    }

    /**
     ***************************************************************************
     **
     ** returns the lamp type as defined by the subsys or user *
     * **************************************************************************
     */
    @Command(name = "getType", description = "returns the lamp type")
    public String getType() {
        return (lampType);
    }

    /**
     ***************************************************************************
     **
     ** returns the identity read from the NewportLamp device *
     * **************************************************************************
     */
    @Command(name = "getLampIdent", description = "returns the identity read from the NewportLamp device")
    public String getLampIdent() {
        String ident = "";

        try {
            ident = lampDev.getIdent();
        } catch (DriverException e) {
            log.error("NewportLamp device failed to return identity!!!");
        }
        return (ident);
    }

    /**
     ***************************************************************************
     **
     ** Closes the connection. *
     * **************************************************************************
     */
    @Override
    protected void close() {
        try {
            lampDev.close();
        } catch (DriverException e) {
            log.error("NewportLamp device failed to close!!!");
        }
    }

    /**
     ***************************************************************************
     **
     ** Initializes the connection. *
     * **************************************************************************
     */
    @Override
    protected void initialize() {
        if (lampDev != null) {
            try {
                lampDev.open(itype, host, port);
                log.info("State of the NewportLamp device is"
                        + (lampDev.getState() ? "ON" : "OFF"));
                log.info("NewportLamp device initialized.");
                isConnected = true;
                setOnline(true);
            } catch (DriverException e) {
                if (!failedToInitialize) {
                    log.error("NewportLamp device failed to initialize!!!");
                }
                failedToInitialize = true;
            }
        }
    }

    /**
     ***************************************************************************
     **
     ** Reads a channel. *
     * **************************************************************************
     */
    @Override
    protected double readChannel(int chan, int type) {
        double value = 0;
        log.debug("NewportLampDevice readChannel called! chan=" + chan + " type=" + type);
        try {
            if ( chan == 1 ) {
                value = getLampCurrent();
            } else {
                value = getLampPower();
            }
        } catch (Exception e) {
            log.error("Error reading channel type " + type + " channel "+chan, e);
        }
        return (value);
    }

    public void setRunlampcurr(double runlampcurr, int iwstate) {
        this.runlampcurr[iwstate] = runlampcurr;
    }

    public double getRunlampcurr(int iwstate) {
        return runlampcurr[iwstate];
    }
    
    @Command(name="setPowerPreset", description="Set the power preset", type=Command.CommandType.ACTION)
    public void setPowerPreset(@Argument(name="powerPreset",description="desired power") int powerPreset) throws DriverException {
        lampDev.setPowerPreset(powerPreset);
    }
    
    @Command(name="getStatus", description="Read Status", type=Command.CommandType.QUERY)
    public String getStatus() throws DriverException {
        return lampDev.getStatus();
    }
    
    @Command(name="getPowerPreset", description="Read Power Preset", type=Command.CommandType.QUERY)
    public int getPowerPreset() throws DriverException {
        return lampDev.getPowerPreset();
    }

    @Command(name="getVolts", description="Read Volts", type=Command.CommandType.QUERY)
    public double getVolts() throws DriverException {
        return lampDev.getVolts();
    }
        
    @Command(name="getLampHrs", description="Read Lamp Hrs", type=Command.CommandType.QUERY)
    public String getLampHrs() throws DriverException {
        return lampDev.readNewportLamp("LAMP HRS?");
    }
    
    @Command(name="genericRead", description="Generic read", type=Command.CommandType.QUERY)
    public String genericRead(@Argument(name="command",description="generic read command") String command) throws DriverException {
        return lampDev.readNewportLamp(command);
    }
    
    
    
}
