package org.lsst.ccs.subsystems.shutter.ui;

import javax.swing.SwingUtilities;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.Command;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.TrendingStatus;
import org.lsst.ccs.framework.Module;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import javax.swing.JFrame;
import javax.swing.JPanel;
import org.lsst.ccs.subsystems.shutter.common.MotorHistory;
import org.lsst.ccs.subsystems.shutter.common.MotorStatus;
import org.lsst.ccs.subsystems.shutter.status.DriverStatus;
import org.lsst.ccs.subsystems.shutter.status.ShutterStatus;
import org.lsst.ccs.state.State;

/**
 *
 * @author turri
 */
public class ShutterUIModule extends Subsystem {

    ShutterAssembly assembly;
    ShutterControlPanel control;
    PlotPanel plotPanel;
    ShutterStatus currentShutterStatus = null;

    public ShutterUIModule() {
        log.info("[ShutterUIModule] Initializing the Shutter UI module ");
        assembly = new ShutterAssembly();
        control = new ShutterControlPanel();
        plotPanel = new PlotPanel();

        setName("ShutterUIModule");
        setListenToStatus(true);
        setStatusBroadcastPeriod(0);
        start();
    }

    @Override
    public String getStatusBusSelector() {
        return "origin=ccs-shutter";
    }

    public static void main(String[] args) {

        ShutterUIModule t = new ShutterUIModule();
        t.openDisplay();
    }

    public void openDisplay() {
        JPanel panel = new JPanel(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        c.gridx = GridBagConstraints.REMAINDER;
        panel.add(plotPanel, c);
        panel.add(assembly, c);
        panel.add(control, c);

        JFrame frame = new JFrame("Shutter UI");
        frame.getRootPane().setDoubleBuffered(true);
        frame.setContentPane(panel);
        frame.setResizable(false);
        frame.pack();
        frame.setVisible(true);
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }

    private void updateShutterStatus(ShutterStatus newStatus) {
        if (currentShutterStatus == null || !newStatus.equals(currentShutterStatus)) {
            currentShutterStatus = newStatus;
            control.setExposureTime(currentShutterStatus.getExposureTimeSeconds());
            control.setMovementTime(currentShutterStatus.getMoveTimeSeconds());
            control.setEnabled(currentShutterStatus.getState().equals(State.Ready));
        }
    }

    private void updateDisplay(DriverStatus s) {
        plotPanel.clearPlot();
        plotPanel.setTimePeriod(s.getExposureTime() + s.getOpenTime());
        if (s.isFirstBladeInOpenPosition()) {
            assembly.getShutter(0).setShutterPosition(0);
            assembly.getShutter(1).setShutterPosition(1);
        } else {
            assembly.getShutter(1).setShutterPosition(0);
            assembly.getShutter(0).setShutterPosition(1);
        }
    }

    @Override
    public void onStatus(BusMessage s) {

        if (s instanceof ShutterStatus) {
            ShutterStatus newStatus = (ShutterStatus) s;
            updateShutterStatus(newStatus);
        } else if (s instanceof DriverStatus) {
            DriverStatus driverStatus = (DriverStatus) s;
            updateDisplay(driverStatus);
        } else if (s instanceof MotorStatus) {
            final MotorStatus status = (MotorStatus) s;
            // Update to GUI model must happen on swing thread
            String motorName = status.getName();
            int motor = -1;
            if (motorName.contains("motor0")) {
                motor = 0;
            } else if (motorName.contains("motor1")) {
                motor = 1;
            }
            final int motorIndex = motor;

            if (motorIndex != -1) {
                SwingUtilities.invokeLater(new Runnable() {

                    public void run() {
                        plotPanel.plotMotorHistory(status.getHistory(), motorIndex);
                        float position = assembly.getShutter(motorIndex).getShutterPosition();
                        assembly.getShutter(motorIndex).setShutterPosition(position == 0 ? 1 : 0);
                        assembly.repaint();
                    }
                });
            }
        }

    }

    @Override
    public void onCommand(Command cmd) {
        System.out.println(cmd + " cmd correl(" + cmd.getCorrelId() + ")");
    }

    @Override
    public void onReply(CommandReply cmd) {
        System.out.println(cmd + " reply correl(" + cmd.getCorrelId() + ")");
    }
}
