package org.lsst.ccs.subsystems.shutter.simulator;

import org.lsst.ccs.subsystems.shutter.interfaces.HallSensorListener;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorSimulator;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorPosition;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorListener;


import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.subsystems.shutter.common.HallTransitionImpl;
import org.lsst.ccs.subsystems.shutter.interfaces.HallTransition;

/**
 *
 * @author azemoon
 */
public class HallSensorSimulator {

    private List<HallTransition> history = new ArrayList<HallTransition>();
    private ArrayList<HallSensorListener> hallListeners = new ArrayList<HallSensorListener>();
    private final MotorSimulator simulator;
    private long transitionTime;
    private MotorPosition prevMotorPosition = null;
    private int transitionID;
    private boolean retracting = false;
    private SensorData sd;
    private double apperture = 800;
    private List<SensorData> sensorData = new ArrayList<SensorData>();
    private boolean stopped = false;

    public HallSensorSimulator(final MotorSimulator simulator) {
        System.out.println("HallSensorSimulator STARTED ************** ");
        this.simulator = simulator;
        double currentPosition = simulator.getInitialMotorPosition();
        System.out.println("HallSensorSimulator:  currentPosition = " + currentPosition);
        this.buildSensorData();
        for (transitionID = 0; transitionID < sensorData.size(); transitionID++) {
            if (currentPosition < sensorData.get(transitionID).getBladePosition()) {
                break;
            }
        }
        if (transitionID == sensorData.size()) {
            transitionID--;
        }
        simulator.addMotorListener(new MotorListener() {

            @Override
            public void positionChanged(MotorPosition p) {
                HallTransitionImpl ht;
                if ( stopped) {
                    history.clear();
                    stopped = false;
                }
                
                if (transitionID == sensorData.size() || transitionID < 0) {
                    return;
                }
                sd = sensorData.get(transitionID);
                if (prevMotorPosition == null) {
                    retracting = simulator.isRetracting();
                    if (retracting ^ (p.getPosition() > sd.getBladePosition())) {
                        transitionID = retracting ? transitionID - 1 : transitionID + 1;
                        // TODO: check if next two lines needed
                        if (transitionID < 0) {
                            transitionID = 0;
                        } else if (transitionID == sensorData.size()) {
                            transitionID--;
                        }
                    }
                       // Send initial BladeSet position and motor start time as the first 
                       // HallTransitionStatus. This does not correspond to a real Hall transition
                       // and  is a fix until there is a better solution.
                       ht = new HallTransitionImpl(p.getTime(), 0, false, retracting,
                               (float) p.getPosition(), (float)(p.getPosition() *  apperture));
                       fireHallTransition(ht);
                       history.add(ht);
                } else {
                    //System.out.println("HallSensorSimulator: Time " + p.getTime() + " position " + p.getPosition() );
                    while (retracting ^ (p.getPosition() > sd.getBladePosition())) {
                        transitionTime = getTransitionTime(p, sd);
                        ht = new HallTransitionImpl(transitionTime, sd.getSensorID(),
                                retracting ? !sd.isOpen() : sd.isOpen(),
                                retracting, (float) sd.getBladePosition(),
                                (float) (sd.getBladePosition() * apperture));
                        fireHallTransition(ht);
                        history.add(ht);
                        transitionID = retracting ? transitionID - 1 : transitionID + 1;
                        if (transitionID == sensorData.size() || transitionID < 0) {
                            break;
                        }
                        sd = sensorData.get(transitionID);
                    }
                }
                prevMotorPosition = p;
            }

            @Override
            public void motorStopped() {
                fireHallSensorTransitionReady(history);
                stopped = true;
                prevMotorPosition = null;
                if (transitionID < 0) {
                    transitionID = 0;
                } else if (transitionID == sensorData.size()) {
                    transitionID--;
                }
            }
        });
    }

    public MotorSimulator getMotorSimulator() {
        return this.simulator;
    }

    public void addHallSensorListener(HallSensorListener l) {
        hallListeners.add(l);
    }

    public void removeHallSensorListener(HallSensorListener l) {
        hallListeners.remove(l);
    }

    private void fireHallTransition(HallTransitionImpl h) {
        for (HallSensorListener l : new ArrayList<HallSensorListener>(hallListeners)) {
            l.hallSensorTransition(h);
        }
    }
    
    private void fireHallSensorTransitionReady(List<HallTransition> history) {
        for (HallSensorListener l : new ArrayList<HallSensorListener>(hallListeners)) {
            l.hallSensorDataReady(history);
        }
    }

    private static class SensorData {

        double bladePosition;
        int sensorID;
        boolean open;

        public SensorData(double bladePosition, int sensorID, boolean open) {
            this.bladePosition = bladePosition;
            this.sensorID = sensorID;
            this.open = open;
        }

        public double getBladePosition() {
            return bladePosition;
        }

        public int getSensorID() {
            return sensorID;
        }

        public boolean isOpen() {
            return open;
        }
    }
/**
 * Build up sensor data assuming that nMagnet magnets and nSensor sensors are arranged
 * so that equally spaced blade positions bladePosition correspond to on/off sensor
 * signals. This method associates a sensor id with each value of bladePosition. The
 * same data is used for both blade sets.
 */
    public void buildSensorData() {
        System.out.println("HallSensorSimulator building sensor data ************** ");
        boolean open = false;
        double offset = 10 / apperture;
        //double offset = 0;
        double transitionWidth = 5 / apperture;
        int nMagnet = 9;
        int nSensor = 4;
        int nTransition = nMagnet * nSensor;
        double transitionSpacing = (1 - 2 * offset - transitionWidth) / (nTransition - 1);
        double bladePosition = offset;

        for (int id = 1; id <= nSensor; id++) {
            for (int n = 1; n <= nMagnet; n++) {
                sensorData.add(new SensorData(bladePosition, id, open));
                System.out.println("BuildSensorData: position " + bladePosition + " id " + id + " open " + open);
                sensorData.add(new SensorData(bladePosition + transitionWidth, id, !open));
                System.out.println("BuildSensorData: position " + (bladePosition + transitionWidth) + " id " + id + " open " + !open);
                bladePosition += transitionSpacing;
            }
        }
    }

    long getTransitionTime(MotorPosition p, SensorData sd) {
        // assume linear interpolation
        long currentTime = p.getTime();
        long prevTime = prevMotorPosition.getTime();
        double prevPosition = prevMotorPosition.getPosition();
        double currentPosition = p.getPosition();
        double transitionPosition = sd.getBladePosition();
        return prevTime + (long) ((currentTime - prevTime) * (transitionPosition - prevPosition)
                / (currentPosition - prevPosition));
    }
}
