package org.lsst.ccs.subsystems.shutter.gui;

import java.awt.BorderLayout;
import org.lsst.ccs.subsystems.shutter.common.HallTransitionImpl;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JPanel;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JSplitPane;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorListener;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorPosition;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorSimulator;

/**
 *
 * @author azemoon
 */
public class ShutterUI extends JPanel {

    final static double MAXCLOSETIME = 2;

    private final ShutterAssembly assembly;
    private PlotPanel plotPanel;
    private List<MotorHistory> motorHistories = new ArrayList<MotorHistory>();
    private List<HallSensorHistory> hallSensorHistories = new ArrayList<HallSensorHistory>();
    private List<MotorReadOut> motorReadOuts = new ArrayList<MotorReadOut>();
    private boolean inMotion = false;
    private boolean finalize = true;
    private EventListenerList listeners = new EventListenerList();
    private List<MotorSimulator> simulators = new ArrayList<MotorSimulator>();

    public ShutterUI() {
        super(new BorderLayout());

        simulators.add(new MotorSimulator());
        simulators.add(new MotorSimulator());

        assembly = new ShutterAssembly();
        motorHistories.add(new MotorHistory());
        motorHistories.add(new MotorHistory());
        hallSensorHistories.add(new HallSensorHistory());
        hallSensorHistories.add(new HallSensorHistory());
        motorReadOuts.add(new MotorReadOut());
        motorReadOuts.add(new MotorReadOut());

        plotPanel = new PlotPanel(motorHistories, hallSensorHistories, motorReadOuts);

        JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        split.add(plotPanel);
        split.add(assembly);
        add(split, BorderLayout.CENTER);

        simulators.get(0).setInitialMotorPosition(assembly.getShutter(0).getShutterPosition());
        simulators.get(1).setInitialMotorPosition(assembly.getShutter(1).getShutterPosition());
    }

    List<MotorHistory> getHistories() {
        return motorHistories;
    }

    public boolean isInMotion() {
        return inMotion;
    }

    public void addChangeListener(ChangeListener l) {
        listeners.add(ChangeListener.class, l);
    }

    public void removeChangeListener(ChangeListener l) {
        listeners.remove(ChangeListener.class, l);
    }

    private void fireChangeListeners() {
        ChangeEvent e = new ChangeEvent(this);
        for (ChangeListener l : listeners.getListeners(ChangeListener.class)) {
            l.stateChanged(e);
        }
    }

    public void moveToPosition(final int index, float targetPosition, final float moveTimeSeconds, boolean initialize) {

        float currentPosition = assembly.getShutter(index).getShutterPosition();
        System.out.println("UI: currentPosition " + currentPosition + "targetPosition  " + targetPosition);
        boolean retractBladeSet = targetPosition < currentPosition;

        if (initialize) {
            //startTime = System.currentTimeMillis();
            hallSensorHistories.get(index).setStartPosition((double) currentPosition);
            for (int i = 0; i <= 1; i++) {
                motorHistories.get(i).reset();
                hallSensorHistories.get(i).reset();
                motorReadOuts.get(i).reset();
            }
        } else {
            hallSensorHistories.get(index).setStartTime(hallSensorHistories.get(1 - index).getStartTime());
            motorHistories.get(index).setStartTime(motorHistories.get(1 - index).getStartTime());
        }

        if (!retractBladeSet && initialize) {
            float safeTargetPosition = 1 - assembly.getShutter(1 - index).getShutterPosition();
            if (targetPosition > safeTargetPosition) {
                System.out.printf("Unsafe operation, end position modified \n");
                targetPosition = safeTargetPosition;
                System.out.println("UI: currentPosition " + currentPosition + "targetPosition  " + targetPosition);
            }
        }

        final MotorSimulator motor = simulators.get(index);
        motor.addMotorListener(new MotorListener() {

            @Override
            public void positionChanged(MotorPosition p) {
                //System.out.println("UI: Time " + p.getTime() + " position " + p.getPosition() );
                assembly.getShutter(index).setShutterPosition((float) p.getPosition());
                assembly.repaint();
                motorHistories.get(index).addData(p.getTime(), p.getAcceleration(), p.getVelocity(), p.getPosition());
            }

            @Override
            public void motorStopped() {
                if (finalize) {
                    inMotion = false;
                    fireChangeListeners();
                }
                motor.removeMotorListener(this);
            }
        });

        inMotion = true;
        fireChangeListeners();

        motor.simulateMovement(targetPosition, (long) moveTimeSeconds);
    }

    void takeImage(final int firstMotorIndex, final float moveTimeSeconds, final float exposureTimeSeconds) {
        plotPanel.setTimePeriod(moveTimeSeconds + exposureTimeSeconds);
        int exposureTime = (int) (1000 * exposureTimeSeconds);

        //float currentPosition = assembly.getShutter(0).getShutterPosition();
        //boolean retractBladeSet0 = currentPosition > 0.5;

        // Which motor starts first
        //final int firstMotorIndex = retractBladeSet0 ? 0 : 1;
        final int secondMotorIndex = 1 - firstMotorIndex;
        final float firstTargetPosition = 0;
        final float secondTargetPosition = 1;
        //System.out.println("UI: firstTargetPosition " + firstTargetPosition + " secondTargetPosition " + secondTargetPosition);

        inMotion = true;
        fireChangeListeners();
        Timer timer1 = new Timer(exposureTime, new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                finalize = true;
                moveToPosition(secondMotorIndex, secondTargetPosition, moveTimeSeconds, false);
            }
        });
        timer1.setRepeats(false);

        timer1.start();
        finalize = false;
        moveToPosition(firstMotorIndex, firstTargetPosition, moveTimeSeconds, true);
    }

    void moveToPosition(int bladeSetIndex, float targetPosition, float movementTime) {
        plotPanel.setTimePeriod(movementTime );
        moveToPosition(bladeSetIndex, targetPosition, movementTime, true);
    }

    void takeImage(final float moveTimeSeconds, final float exposureTimeSeconds) {
        //boolean docked = false;
        float position0 = assembly.getShutter(0).getShutterPosition();
        float position1 = assembly.getShutter(1).getShutterPosition();
        System.out.println("UI: " + position0 + " " + position1 + " *******");
        //final float targetPosition;
        if ( ( position0 == 0 && position1 == 1) || ( position0 == 1 && position1 == 0) ) {
            //targetPosition = 1 - position0;
            final int firstMotorIndex = (position0 < position1) ? 1 : 0;
            takeImage(firstMotorIndex, moveTimeSeconds, exposureTimeSeconds);
       }  else {
            //takeImage(1, 1,(float)0.1 );
            System.out.println(" ****** Shutter not ready for taking image.******");
            //docked = true;
       }
       //return docked;
    }

    public void closeShutter() {
        //System.out.println("ShutterSimulator: ***** " + position0 + " " + position1 + " *******");
        float position0 = assembly.getShutter(0).getShutterPosition();
        float position1 = assembly.getShutter(1).getShutterPosition();
        if ( ( position0 == 0 && position1 == 1) || ( position0 == 1 && position1 == 0) ) {
            System.out.println("Shutter already closed ***** ");
        }  else {
            final int firstBladeSetIndex = (position0 < position1) ? 0 : 1;
            final int secondBladeSetIndex = 1 - firstBladeSetIndex;
            final double moveTimeSeconds = MAXCLOSETIME;// * (1 - Math.max(position0, position1));
            if (position0 == 0 || position1 == 0) {
                moveToPosition(secondBladeSetIndex, 1, (float) moveTimeSeconds, true);
            } else {

                Thread t = new Thread(new Runnable(){

                    @Override
                    public void run() {
                        moveToPosition(secondBladeSetIndex, 1 , (float) moveTimeSeconds, false);
                    }
                });
                t.start();
                moveToPosition(firstBladeSetIndex, 0, (float) moveTimeSeconds, true);
            }
        }
   }

    void recordMotorPositions(int motorIndex, List<MotorPosition> h) {
        int otherMotorIndex = 1 - motorIndex;
        long t0 = motorReadOuts.get(otherMotorIndex).getStartTime() > 0 ? motorReadOuts.get(otherMotorIndex).getStartTime() : h.get(0).getTime();
         motorReadOuts.get(motorIndex).addData(h, t0);
        //for (MotorPosition p : h) {
            //System.out.println("UI : set " + motorIndex + " Time" + p.getTime() + " smeared position " + p.getPosition());
        //}
    }

    void recordHallSensorPositions(int sensorIndex, HallTransitionImpl h) {
        hallSensorHistories.get(sensorIndex).addData(h.getTransitionTime(), h.getPosition());
        //System.out.println("UI: set " + sensorIndex + " Time " + h.getTransitionTime() + " position " + h.getPosition());
        //System.out.println("UI: set " + sensorIndex + " ID " + h.getSensorId() + " State " + h.isOpen() + " retracting " + h.isReverse());
    }
}
