package org.lsst.ccs.subsystems.shutter.gui;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.Timer;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.EventListenerList;
import org.lsst.ccs.subsystems.shutter.interfaces.BladePosition;
import org.lsst.ccs.subsystems.shutter.interfaces.HallTransition;
import org.lsst.ccs.subsystems.shutter.simulator.EncoderReadOut;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorListener;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorPosition;
import org.lsst.ccs.subsystems.shutter.simulator.motor.MotorSimulator;
import org.lsst.ccs.subsystems.shutter.status.*;

/**
 *
 * @author azemoon
 */
public class ShutterDisplay extends JPanel {

    private final ShutterAssembly assembly;
    private PlotPanelNew plotPanel;
    private List<MotorHistory> motorHistories = new ArrayList<MotorHistory>();
    private List<HallSensorHistory> hallSensorHistories = new ArrayList<HallSensorHistory>();
    private List<EncoderReadOut> encoderReadOuts = new ArrayList<EncoderReadOut>();
    private boolean inMotion = false;
    private EventListenerList listeners = new EventListenerList();
    private List<MotorSimulator> simulators = new ArrayList<MotorSimulator>();

    public ShutterDisplay() {
        super(new BorderLayout());

        simulators.add(new MotorSimulator());
        simulators.add(new MotorSimulator());

        assembly = new ShutterAssembly();
        motorHistories.add(new MotorHistory());
        motorHistories.add(new MotorHistory());
        hallSensorHistories.add(new HallSensorHistory());
        hallSensorHistories.add(new HallSensorHistory());
        encoderReadOuts.add(new EncoderReadOut());
        encoderReadOuts.add(new EncoderReadOut());

        plotPanel = new PlotPanelNew(motorHistories, hallSensorHistories, encoderReadOuts);

        JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        split.add(plotPanel);
        split.add(assembly);
        add(split, BorderLayout.CENTER);

        simulators.get(0).setInitialMotorPosition(assembly.getShutter(0).getShutterPosition());
        simulators.get(1).setInitialMotorPosition(assembly.getShutter(1).getShutterPosition());
    }

    List<MotorHistory> getHistories() {
        return motorHistories;
    }

    public boolean isInMotion() {
        return inMotion;
    }

    public void addChangeListener(ChangeListener l) {
        listeners.add(ChangeListener.class, l);
    }

    public void removeChangeListener(ChangeListener l) {
        listeners.remove(ChangeListener.class, l);
    }

    private void fireChangeListeners() {
        ChangeEvent e = new ChangeEvent(this);
        for (ChangeListener l : listeners.getListeners(ChangeListener.class)) {
            l.stateChanged(e);
        }
    }

    public void moveToPosition( MoveBladeSetStatus s ) {
        moveToPosition( s.getIndex(), s.getTargetPosition(), (float) s.getMoveTime(), true, true);
    }

    public void moveToPosition(final int index, float targetPosition, final float moveTimeSeconds, final boolean initialize, final boolean finalize) {

        float currentPosition = assembly.getShutter(index).getShutterPosition();
        System.out.println("ShutterDisplay: currentPosition " + currentPosition + " targetPosition  " + targetPosition + " moveTimeSeconds " + moveTimeSeconds);
        boolean retractBladeSet = targetPosition < currentPosition;

        if (initialize) {
            //startTime = System.currentTimeMillis();
            hallSensorHistories.get(index).setStartPosition((double) currentPosition);
            for (int i = 0; i <= 1; i++) {
                motorHistories.get(i).reset();
                hallSensorHistories.get(i).reset();
                encoderReadOuts.get(i).reset();
            }
        } else {
            hallSensorHistories.get(index).setStartTime(hallSensorHistories.get(1 - index).getStartTime());
            motorHistories.get(index).setStartTime(motorHistories.get(1 - index).getStartTime());
        }

        if (!retractBladeSet && initialize) {
            float safeTargetPosition = 1 - assembly.getShutter(1 - index).getShutterPosition();
            if (targetPosition > safeTargetPosition) {
                System.out.printf("Unsafe operation, end position modified \n");
                targetPosition = safeTargetPosition;
                System.out.println("UI: currentPosition " + currentPosition + "targetPosition  " + targetPosition);
            }
        }

        final MotorSimulator motor = simulators.get(index);
        motor.addMotorListener(new MotorListener() {

            @Override
            public void positionChanged(MotorPosition p) {
                //System.out.println("UI: Time " + p.getTime() + " position " + p.getPosition() );
                assembly.getShutter(index).setShutterPosition((float) p.getPosition());
                assembly.repaint();
                motorHistories.get(index).addData(p.getTime(), p.getAcceleration(), p.getVelocity(), p.getPosition());
            }

            @Override
            public void motorStopped() {
                if (finalize) {
                    inMotion = false;
                    fireChangeListeners();
                }
                motor.removeMotorListener(this);
            }
        });

        inMotion = true;
        fireChangeListeners();

        motor.simulateMovement(targetPosition, (long) moveTimeSeconds);
    }

    void takeImage( TakeImageStatus s) {
    //void takeImage(final float targetPosition, final float moveTimeSeconds, final float exposureTimeSeconds) {
        final float moveTimeSeconds = (float) s.getMoveTime();
        final float exposureTimeSeconds = (float) s.getExposureTime();
        final int firstMotorIndex = s.getfirstBladeSetIndex();

        plotPanel.setTimePeriod( moveTimeSeconds + exposureTimeSeconds);
        int exposureTime = (int) (1000 * exposureTimeSeconds);

        // Which motor starts first
        final int secondMotorIndex = 1 - firstMotorIndex;
        final float firstTargetPosition = 0;
        final float secondTargetPosition = 1;
        //System.out.println("UI: firstTargetPosition " + firstTargetPosition + " secondTargetPosition " + secondTargetPosition);

        inMotion = true;
        fireChangeListeners();
        Timer timer1 = new Timer(exposureTime, new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
                moveToPosition(secondMotorIndex, secondTargetPosition, moveTimeSeconds, false, true);
            }
        });
        timer1.setRepeats(false);

        timer1.start();
        moveToPosition(firstMotorIndex, firstTargetPosition, moveTimeSeconds, true, false);
    }

    void closeShutter(CloseShutterStatus s) {
        final double moveTimeSeconds = s.getMoveTime();
        final int firstBladeSetIndex = s.getfirstBladeSetIndex();
        final int secondBladeSetIndex = 1 - firstBladeSetIndex;
        System.out.println("****** ShutterDisplay closeShutter: moveTimeSeconds " + moveTimeSeconds + " *********");
        Thread t = new Thread(new Runnable(){

               @Override
               public void run() {
                    moveToPosition(secondBladeSetIndex, 1 , (float) moveTimeSeconds, false, true);
               }
        });
        inMotion = true;
        fireChangeListeners();
        t.start();
        moveToPosition(firstBladeSetIndex, 0, (float) moveTimeSeconds, true, false);
        //t.start();
 
    }

    void recordMovement( MovementStatus s) {
        System.out.println("****** recordMovement*********");
        int index = s.getIndex();
        boolean movementComplete = s.isMovementComplete();
        if ( movementComplete ) {
            recordMotorPositions(index, s.getHistory().getMovementProfile());
            recordHallSensorPositions(index, s.getHistory().getHallTransitions());
        }
    }

    void recordHallTransition(HallTransitionStatus s) {
        System.out.println("**********recordHallTransition***********");
        int index = s.getIndex();
        recordHallSensorPositions(index, s.getTransition());
    }

    void recordMotorPositions(int motorIndex, List<BladePosition> h) {
        System.out.println("**********recordMotorPositions " + h.size() + " ************");
        int otherMotorIndex = 1 - motorIndex;
        long t0 = encoderReadOuts.get(otherMotorIndex).getStartTime() > 0 ? encoderReadOuts.get(otherMotorIndex).getStartTime() : h.get(0).getTime();
         encoderReadOuts.get(motorIndex).addData(h, t0);
        for (BladePosition p : h) {
            System.out.println("UI : set " + motorIndex + " Time" + p.getTime() + " smeared position " + p.getRelPosition());
        }
    }

    void recordHallSensorPositions(int sensorIndex, HallTransition h) {
        System.out.println("************recordHallSensorPositions************");
        hallSensorHistories.get(sensorIndex).addData(h.getTransitionTime(), h.getPosition());
        System.out.println("UI: set " + sensorIndex + " Time " + h.getTransitionTime() + " position " + h.getPosition());
        System.out.println("UI: set " + sensorIndex + " ID " + h.getSensorId() + " State " + h.isOpen() + " retracting " + h.isReverse());
    }

    void recordHallSensorPositions(int sensorIndex, List<HallTransition> hlist) {
        System.out.println("************recordHallSensorPositions " + hlist.size() + " ************");
        for (HallTransition h: hlist) {
            hallSensorHistories.get(sensorIndex).addData(h.getTransitionTime(), h.getPosition());
            System.out.println("UI: set " + sensorIndex + " Time " + h.getTransitionTime() + " position " + h.getPosition());
            System.out.println("UI: set " + sensorIndex + " ID " + h.getSensorId() + " State " + h.isOpen() + " retracting " + h.isReverse());
        }
    }

}
