package org.lsst.ccs.subsystems.shutter.driver;

import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Scanner;
import org.lsst.ccs.drivers.iocard.AccesDio;
import org.lsst.ccs.drivers.iocard.Helios;
import org.lsst.ccs.drivers.parker.AcrComm;
import org.lsst.ccs.drivers.parker.TestAcrComm;
import org.lsst.ccs.subsystems.shutter.common.BladePositionImpl;
import org.lsst.ccs.subsystems.shutter.interfaces.BladePosition;
import org.lsst.ccs.subsystems.shutter.interfaces.HallTransition;
import org.lsst.ccs.subsystems.shutter.interfaces.MovementHistory;
import org.lsst.ccs.subsystems.shutter.interfaces.MovementHistoryListener;
import org.lsst.ccs.utilities.sa.CmndProc;
import org.lsst.ccs.utilities.sa.ConsOut;
import org.lsst.ccs.utilities.sa.Output;

/**
 ***************************************************************************
 **
 **  Program to perform various tests on a shutter blade set
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class TestBladeSet implements CmndProc.Dispatch, MovementHistoryListener {

   /**
    ***************************************************************************
    **
    **  Inner class holding pulse parameters
    **
    ***************************************************************************
    */
    public static class Pulse {

        int    mCount;   // Number of pulses to generate
        int    count;    // Current number of pulses
        int    oLine;    // Output line to pulse
        int    iLine;    // Input line to monitor
        int    iState;   // Input line state
        float  width;    // Pulse width (seconds)

    }

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    /*
    **  Drive states
    */
    private final static int
        DRV_OFF  = 0,
        DRV_ON   = 1;

    /*
    **  Miscellaneous
    */
    private final static int
        TEST_PROG = 2;

    /*
    **  Command codes
    */
    private final static int
        CMD_CALIB       = 0,
        CMD_STEP        = 1,
        CMD_HOME        = 2,
        CMD_MOVE        = 3,
        CMD_MOVEP       = 4,
        CMD_MOVEH       = 5,
        CMD_MOVED       = 6,
        CMD_CALREAD     = 7,
        CMD_CALSHOW     = 8,
        CMD_PULSE       = 9,
        CMD_CALTEST     = 10,
        CMD_MOVEX       = 11,
        CMD_THRSHOW     = 12,
        CMD_TIMESYNC    = 13,
        CMD_HALLTEST    = 14,
        CMD_DRIVE       = 15,
        CMD_MOVETEST    = 16,
        CMD_ENCTEST     = 17,
        CMD_POSITION    = 18,
        CMD_CONFSHOW    = 19,
        CMD_TEMPSHOW    = 20,
        NUM_CMDS        = 21;

    /*
    **  Per-command help text
    */
    private final static String[] helpConfshow = {
        "Show the blade configuration data",
        "confshow",
    };

    private final static String[] helpDrive = {
        "Set drive state",
        "drive  <state>",
        "state  The state to set, either 'on' or 'off'",
    };

    private final static String[] helpCalib = {
        "Calibrate Hall switch positions",
        "calibrate <dist> <step> [<time>] [<file>]",
        "dist    The total distance to travel (mm)",
        "step    The step size (mm)",
        "time    The time for each step (sec) (default 0.1)",
        "file    The base name of the file to receive the calibration data",
    };

    private final static String[] helpCaltest = {
        "Calibrate Hall switch positions (new calibration test)",
        "caltest <type> <dist> [<time>] [<file>]",
        "type    The test type: 0 = new cal; 1 = cap test 1; 2 = cap test 2",
        "dist    The distance to move (mm)",
        "time    The time for the move (sec) (default 10)",
        "file    The base name of the file to receive the calibration data",
    };

    private final static String[] helpCalread = {
        "Read file of Hall switch positions",
        "calread [<file>]",
        "file    The name of the file containing the calibration data",
    };

    private final static String[] helpCalshow = {
        "Show the Hall switch position data",
        "calshow",
    };

    private final static String[] helpPosition = {
        "Position the shutter blade set at a fraction of its full extent",
        "position [<posn>] [<time>]",
        "posn    The position to move to (0 - 1); if absent, display it",
        "time    The time for the move (secs) (default 1.0)",
    };

    private final static String[] helpMove = {
        "Move the shutter blade set",
        "move [<dist>] [<time>] [<type>] [<mvfrac>]",
        "dist    The distance to move (mm); if absent, display position",
        "time    The time for the move (secs) (default 1.0)",
        "type    The type of move to make: 'trap' or 'scurve' (default)",
        "mvfrac  The fraction of the time at maximum velocity (default 0)",
    };

    private final static String[] helpMoved = {
        "Move the shutter blade set, recording various data values",
        "moved [<dist>] [<time>] [<nsamp>] [<type>] [<mvfrac>] [<name>]",
        "dist    The distance to move (mm); if absent, display position",
        "time    The time for the move (secs) (default 1.0)",
        "nsamp   The number of times to sample the position (default 100)",
        "type    The type of move to make: 'trap' or 'scurve' (default)",
        "mvfrac  The fraction of the time at maximum velocity (default 0)",
        "name    The root part of the name of the output data file",
    };

    private final static String[] helpMoveh = {
        "Move the shutter blade set, recording Hall switch times",
        "moveh [<dist>] [<time>] [<nsamp>] [<type>] [<mvfrac>] [<name>]",
        "dist    The distance to move (mm); if absent, display position",
        "time    The time for the move (secs) (default 1.0)",
        "nsamp   The number of times to sample the position (default 100)",
        "type    The type of move to make: 'trap' or 'scurve' (default)",
        "mvfrac  The fraction of the time at maximum velocity (default 0)",
        "name    The root part of the name of the output data files",
    };

    private final static String[] helpMovep = {
        "Move the shutter blade set, recording sampled positions",
        "movep [<dist>] [<time>] [<nsamp>] [<type>] [<mvfrac>] [<name>]",
        "dist    The distance to move (mm); if absent, display position",
        "time    The time for the move (secs) (default 1.0)",
        "nsamp   The number of times to sample the position (default 100)",
        "type    The type of move to make: 'trap' or 'scurve' (default)",
        "mvfrac  The fraction of the time at maximum velocity (default 0)",
        "name    The root part of the name of the output data file",
    };

    private final static String[] helpMovex = {
        "Move the shutter blade set, recording test values",
        "movex [<dist>] [<time>] [<nsamp>] [<type>] [<mvfrac>] [<name>]",
        "dist    The distance to move (mm); if absent, display position",
        "time    The time for the move (secs) (default 1.0)",
        "nsamp   The number of times to sample the position (default 100)",
        "type    The type of move to make: 'trap' or 'scurve' (default)",
        "mvfrac  The fraction of the time at maximum velocity (default 0)",
        "name    The root part of the name of the output data file",
    };

    private final static String[] helpMovetest = {
        "Move the shutter blade set repeatedly",
        "movetest <count> <dist> [<time>]",
        "count   The number of back-and-forth cycles to perform",
        "dist    The distance to move (mm)",
        "time    The time for each move (secs) (default 1.0)",
    };

    private final static String[] helpHalltest = {
        "Move the shutter blade set repeatedly, recording Hall switch times",
        "halltest <count> <dist> [<time>] [<name>]",
        "count   The number of back-and-forth cycles to perform",
        "dist    The distance to move (mm)",
        "time    The time for each move (secs) (default 1.0)",
        "name    The root part of the name of the output data file",
    };

    private final static String[] helpEnctest = {
        "Move the shutter blade set repeatedly, recording encoder profiles",
        "enctest <count> <dist> [<time>] [<nsamp>] [<name>]",
        "count   The number of back-and-forth cycles to perform",
        "dist    The distance to move (mm)",
        "time    The time for each move (secs) (default 1.0)",
        "nsamp   The number of times to sample the position (default 100)",
        "name    The root part of the name of the output data file",
    };

    private final static String[] helpStep = {
        "Step the motor, recording actual steps",
        "step <count> <step> [<time>]",
        "count  The number of steps to take",
        "step   The step size (encoder counts)",
        "time   The time for each step (sec) (default 0.1)",
    };

    private final static String[] helpHome = {
        "Find the home position",
        "home [<optns>] [<vel>]",
        "optns  Bit mask of options: 1 = neg. dirn.; 2 = pos. final (dflt 0)",
        "vel    Velocity (mm/sec) (default 10)",
    };

    private final static String[] helpPulse = {
        "Start or stop regular pulse on DIO line",
        "pulse <freq> [<line>] [<count>] [<fract>]",
        "freq   The pulse frequency; 0 = off",
        "line   The digital output line to use (default: ACR line)",
        "count  If >= 0, the maximum number of pulses to generate (default -1)",
        "fract  The fraction of time the pulse is on (default 0.5)",
    };

    private final static String[] helpTimesync = {
        "Time the synchronization between computer and motor controller",
        "timesync  [<capt>] [<count>]",
        "capt   If non-zero, use capture for synch'ing (default not)",
        "count  The number of pulses to generate (default 10)",
    };

    private final static String[] helpTempshow = {
        "Show the drive and ambient temperatures",
        "tempshow",
    };

    private static final String[] helpThrshow = {
        "Show thread information",
        "thrshow",
    };

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private final Output out;
    private final CmndProc.Lookup moveTypes, driveTypes;
    private final int dioOPort, dioOLine, dioIPort, dioILine;
    private final int bladeIndex;
    private final BladeSetDrvr blade;
    private final CmndProc proc;
    private final BladeSetConfigurationDrvr config;
    private final TestAcrComm tacr;
    private final AccesDio acd;
    private final Helios hel;
    private final AcrComm acr;
    private String moveName = "Move_", calibName = "Calib_", caltName = "Calt_",
                   hallName = "Hall_", encName = "Enc_";
    private MovementHistory mhData;


   /**
    ***************************************************************************
    **
    **  Main constructor
    **
    ***************************************************************************
    */
    public TestBladeSet(int index, Output iOut) throws IOException
    {
        out = (iOut != null) ? iOut : new ConsOut();

        CmndProc.Command cmnd = new CmndProc.Command(NUM_CMDS);
        cmnd.add("confshow",    CMD_CONFSHOW,    helpConfshow);
        cmnd.add("drive",       CMD_DRIVE,       helpDrive);
        cmnd.add("calibrate",   CMD_CALIB,       helpCalib);
        cmnd.add("caltest",     CMD_CALTEST,     helpCaltest);
        cmnd.add("calread",     CMD_CALREAD,     helpCalread);
        cmnd.add("calshow",     CMD_CALSHOW,     helpCalshow);
        cmnd.add("position",    CMD_POSITION,    helpPosition);
        cmnd.add("move",        CMD_MOVE,        helpMove);
        cmnd.add("moved",       CMD_MOVED,       helpMoved);
        cmnd.add("movep",       CMD_MOVEP,       helpMovep);
        cmnd.add("movex",       CMD_MOVEX,       helpMovex);
        cmnd.add("moveh",       CMD_MOVEH,       helpMoveh);
        cmnd.add("movetest",    CMD_MOVETEST,    helpMovetest);
        cmnd.add("halltest",    CMD_HALLTEST,    helpHalltest);
        cmnd.add("enctest",     CMD_ENCTEST,     helpEnctest);
        cmnd.add("step",        CMD_STEP,        helpStep);
        cmnd.add("home",        CMD_HOME,        helpHome);
        cmnd.add("pulse",       CMD_PULSE,       helpPulse);
        cmnd.add("tempshow",    CMD_TEMPSHOW,    helpTempshow);
        cmnd.add("thrshow",     CMD_THRSHOW,     helpThrshow);
        cmnd.add("timesync",    CMD_TIMESYNC,    helpTimesync);

        moveTypes = new CmndProc.Lookup(2);
        moveTypes.add("scurve",  BladeSetDrvr.MOV_TYP_SCURVE);
        moveTypes.add("trap",    BladeSetDrvr.MOV_TYP_TRAP);

        driveTypes = new CmndProc.Lookup(2);
        driveTypes.add("on",  DRV_ON);
        driveTypes.add("off", DRV_OFF);

        proc = new CmndProc();
        proc.add(this, cmnd);

        bladeIndex = index;
        blade = new BladeSetDrvr();
        blade.setIndex(bladeIndex);
        blade.addMovementHistoryListener(this);

        config = blade.getConfig();
        dioOPort = config.getDioOPort();
        dioOLine = config.getDioOLine();
        dioIPort = config.getDioIPort();
        dioILine = config.getDioILine();

        acd = blade.getAcces();
        hel = blade.getHelios();
        acr = blade.getComm();
        tacr = new TestAcrComm(acr, proc, true, out);
    }


   /**
    ***************************************************************************
    **
    **  Main program
    **
    **  Initialize and run test
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        int index = 0;
        TestBladeSet test;

        if (args.length != 0) {
            try {
                index = Integer.valueOf(args[0]);
            }
            catch (NumberFormatException e) {
                System.out.println("Blade set index is not an integer");
                System.exit(0);
            }
            if (index < 0 || index > 1) {
                System.out.println("Blade set index must be 0 or 1");
                System.exit(0);
            }
        }
        try {
            test = new TestBladeSet(index, null);
            test.tacr.run();
        }
        catch (AcrComm.Exception e) {
            System.out.println(e);
        }
        catch (IOException e) {
            System.out.println(e);
        }

        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Process a command string
    **
    ***************************************************************************
    */
    public void process(String command)
    {
        tacr.process(command);
    }


   /**
    ***************************************************************************
    **
    **  Dispatch local command for processing
    **
    ***************************************************************************
    */
    @Override
    public boolean dispatch(int code, Scanner scan)
    {
        int found;
        Object[] args = new Object[16];

        switch (code) {

        case CMD_CONFSHOW:
            if (CmndProc.scanArgs(scan, "", args) >= 0)
                showConfig();
            break;

        case CMD_DRIVE:
            if (CmndProc.scanArgs(scan, "S", args) >= 0) {
                int type = driveTypes.encode((String)args[0], true);
                if (type == DRV_ON) blade.setDriveOn();
                else if (type == DRV_OFF) blade.setDriveOff();
            }
            break;

        case CMD_CALIB:
            if ((found = CmndProc.scanArgs(scan, "FFfs", args)) >= 0) {
                float dist = (Float)args[0];
                float step = (Float)args[1];
                float time = (found & 0x04) != 0 ? (Float)args[2] : 0.1f;
                if ((found & 0x08) != 0) calibName = (String)args[3] + "_";
                String file = genFileName(calibName, ".dat");
                calibrate(dist, step, time, file);
            }
            break;

        case CMD_CALTEST:
            if ((found = CmndProc.scanArgs(scan, "IFfs", args)) >= 0) {
                if ((found & 0x08) != 0) caltName = (String)args[3] + "_";
                int type = (Integer)args[0];
                float dist = (Float)args[1];
                float time = (found & 0x04) != 0 ? (Float)args[2] : 10f;
                String file = genFileName(caltName + type + "_", ".dat");
                if (type == 0)
                    calTest(dist, time, file);
                else if (type == 1 || type == 3)
                    calTest1(type == 1, dist, time, file);
                else if (type == 2)
                    calTest2(dist, time, file);
                else if (type == 4)
                    calTest4(dist, time, file);
                else
                    out.println("Invalid caltest type");
            }
            break;

        case CMD_CALREAD:
            if ((found = CmndProc.scanArgs(scan, "s", args)) >= 0) {
                int count;
                if ((found & 0x01) != 0)
                    count = config.readCalibration((String)args[0]);
                else
                    count = config.readCalibration();
                if (count >= 0)
                    out.println(count + " lines read from calibration file");
            }
            break;

        case CMD_CALSHOW:
            if (CmndProc.scanArgs(scan, "", args) >= 0)
                showCalib();
            break;

        case CMD_POSITION:
            if ((found = CmndProc.scanArgs(scan, "ffs", args)) >= 0) {
                if ((found & 0x01) == 0) {
                    showRelPosition();
                    break;
                }
                float dist = (Float)args[0];
                float time = (found & 0x02) != 0 ? (Float)args[1]
                                                 : config.getMoveTime();
                if ((found & 0x04) != 0) moveName = (String)args[2] + "_";
                position(dist, time, moveName);
            }
            break;

        case CMD_MOVE:
            if ((found = CmndProc.scanArgs(scan, "ffsf", args)) >= 0) {
                if ((found & 0x01) == 0) {
                    showPosition();
                    break;
                }
                String sType = (found & 0x04) != 0 ? (String)args[2] : "scurve";
                int type = moveTypes.encode(sType, true);
                if (type < 0) break;
                float dist = (Float)args[0];
                float time = (found & 0x02) != 0 ? (Float)args[1]
                                                 : config.getMoveTime();
                float mvFrac = (found & 0x08) != 0 ? (Float)args[3] : 0f;
                move(type, dist, time, mvFrac);
            }
            break;

        case CMD_MOVED:
        case CMD_MOVEH:
        case CMD_MOVEP:
        case CMD_MOVEX:
            if ((found = CmndProc.scanArgs(scan, "ffisfs", args)) >= 0) {
                if ((found & 0x01) == 0) {
                    showPosition();
                    break;
                }
                float dist = (Float)args[0];
                float time = (found & 0x02) != 0 ? (Float)args[1]
                                                 : config.getMoveTime();
                int nSamp = (found & 0x04) != 0 ? (Integer)args[2]
                                                : config.getNumSamp();
                String sType = (found & 0x08) != 0 ? (String)args[3] : "scurve";
                int type = moveTypes.encode(sType, true);
                if (type < 0) break;
                float mvFrac = (found & 0x10) != 0 ? (Float)args[4] : 0f;
                if ((found & 0x20) != 0) moveName = (String)args[5] + "_";
                if (code == CMD_MOVED)
                    moveD(type, dist, time, mvFrac, nSamp, moveName);
                else if (code == CMD_MOVEH)
                    moveH(type, dist, time, mvFrac, nSamp, moveName);
                else if (code == CMD_MOVEP)
                    moveP(type, dist, time, mvFrac, nSamp, moveName);
                else
                    moveX(type, dist, time, mvFrac, nSamp, moveName);
            }
            break;

        case CMD_MOVETEST:
            if ((found = CmndProc.scanArgs(scan, "IFf", args)) >= 0) {
                int count = (Integer)args[0];
                float dist = (Float)args[1];
                float time = (found & 0x04) != 0 ? (Float)args[2]
                                                 : config.getMoveTime();
                moveTest(count, dist, time);
            }
            break;

        case CMD_HALLTEST:
            if ((found = CmndProc.scanArgs(scan, "IFfs", args)) >= 0) {
                int count = (Integer)args[0];
                float dist = (Float)args[1];
                float time = (found & 0x04) != 0 ? (Float)args[2]
                                                 : config.getMoveTime();
                if ((found & 0x08) != 0) hallName = (String)args[3] + "_";
                hallTest(count, dist, time, hallName);
            }
            break;

        case CMD_ENCTEST:
            if ((found = CmndProc.scanArgs(scan, "IFfis", args)) >= 0) {
                int count = (Integer)args[0];
                float dist = (Float)args[1];
                float time = (found & 0x04) != 0 ? (Float)args[2]
                                                 : config.getMoveTime();
                int nSamp = (found & 0x08) != 0 ? (Integer)args[3]
                                                : config.getNumSamp();
                if ((found & 0x10) != 0) encName = (String)args[4] + "_";
                encTest(count, dist, time, nSamp, encName);
            }
            break;

        case CMD_STEP:
            if ((found = CmndProc.scanArgs(scan, "IIf", args)) >= 0) {
                int count = (Integer)args[0];
                int step = (Integer)args[1];
                float time = (found & 0x04) != 0 ? (Float)args[2] : 0.1f;
                step(count, step, time);
            }
            break;

        case CMD_HOME:
            if ((found = CmndProc.scanArgs(scan, "if", args)) >= 0) {
                int optns = (found & 0x01) != 0 ? (Integer)args[0] : 0;
                float vel = (found & 0x02) != 0 ? (Float)args[1] : 10f;
                blade.home(optns, vel);
            }
            break;

        case CMD_PULSE:
            if ((found = CmndProc.scanArgs(scan, "Fiif", args)) >= 0) {
                float freq = (Float)args[0];
                if (freq != 0F) {
                    Pulse pulse = new Pulse();
                    pulse.oLine
                      = (found & 0x02) != 0 ? (Integer)args[1] : dioOLine;
                    pulse.mCount
                      = (found & 0x04) != 0 ? (Integer)args[2] : -1;
                    float fract = (found & 0x08) != 0 ? (Float)args[3] : 0.5F;
                    if (freq < 2) freq = 2;
                    if (freq > 10000) freq = 10000;
                    if (fract > 1) fract = 1;
                    pulse.width = fract / freq;
                    if (pulse.width < 10e-6F) pulse.width = 10e-6F;
                    pulse.count = 0;
                    hel.cntrConfig(freq, false);
                    hel.cntrEnable(this, "genPulse", pulse);
                    hel.cntrStart();
                }
                else {
                    hel.cntrStop();
                    hel.cntrDisable();
                }
            }
            break;

        case CMD_TIMESYNC:
            if ((found = CmndProc.scanArgs(scan, "ii", args)) >= 0) {
                timeSync((found & 0x01) != 0 ? (Integer)args[0] != 0 : false,
                         (found & 0x02) != 0 ? (Integer)args[1] : 10);
            }
            break;

        case CMD_TEMPSHOW:
            if (CmndProc.scanArgs(scan, "", args) >= 0)
                showTemps();
            break;

        case CMD_THRSHOW:
            if (CmndProc.scanArgs(scan, "", args) >= 0) {
                Thread[] threads = new Thread[Thread.activeCount()];
                int nThread = Thread.enumerate(threads);
                out.println("Thread information:");
                for (int j = 0; j < nThread; j++)
                    out.println("  " + threads[j]);
            }
            break;

        default:
            out.println("Command not fully implemented");

        }

        return true;
    }


   /**
    ***************************************************************************
    **
    **  Show the blade configuration data
    **
    ***************************************************************************
    */
    private void showConfig()
    {
        out.format("Blade %s Configuration:\n", config.getIndex());
        out.format("  Controller node:             %s\n", config.getNode());
        out.format("  Encoder pulses/mm:           %s\n", config.getPpMm());
        out.format("  Blade home position (mm):    %s\n", config.getHome());
        out.format("  Blade open position (mm):    %s\n", config.getOpen());
        out.format("  Blade closed position (mm):  %s\n", config.getClosed());
        out.format("  Position sample count:       %s\n", config.getNumSamp());
        out.format("  Blade movement time (sec):   %s\n", config.getMoveTime());
        out.format("  DIO board base address:      0x%04x\n",
                   config.getDioAddr());
        out.format("  DIO board IRQ level:         %s\n", config.getDioLevel());
        out.format("  DIO board configuration:     0x%02x\n",
                   config.getDioConf());
        out.format("  Hall sensor DIO port:        %s\n", config.getDioHPort());
        out.format("  Motor output DIO port:       %s\n", config.getDioOPort());
        out.format("  Motor output DIO line:       %s\n", config.getDioOLine());
        out.format("  Motor input DIO port:        %s\n", config.getDioIPort());
        out.format("  Motor input DIO line:        %s\n", config.getDioILine());
        out.format("  AD board base address:       0x%04x\n",
                   config.getAdAddr());
        out.format("  AD board IRQ level:          %s\n", config.getAdLevel());
        out.format("  Temperature AD channel:      %s\n",
                   config.getAdTempChn());
    }


   /**
    ***************************************************************************
    **
    **  Calibrate the Hall switch positions
    **
    ***************************************************************************
    */
    private void calibrate(float dist, float step, float time, String file)
    {
        BladeSetCalibration calData = blade.calibrate(dist, step, time);
        showStop(calData.getStatus());
        BladeSetConfigurationDrvr.writeCalibration(calData, file);
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Calibrate the Hall switch positions (new test method)
    **
    ***************************************************************************
    */
    private void calTest(float dist, float time, String file)
    {
        BladeSetCalibration calData = blade.calibNew(dist, time);
        showStop(calData.getStatus());
        BladeSetConfigurationDrvr.writeCalibration(calData, file);
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Test the capture of encoder positions using pulses from CPU
    **
    ***************************************************************************
    */
    private void calTest1(boolean useCapt, float dist, float time, String file)
    {
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Set pulse parameters and initial line state
        */
        int mCount = 64;
        float freq = mCount / (2 * time);
        Pulse pulse = new Pulse();
        pulse.mCount = 2 * mCount;
        pulse.width = 0.5F / freq;
        pulse.oLine = dioOLine;
        acd.dioClrBit(dioOPort, pulse.oLine);

        /*
        **  Load the capture program and start it
        */
        acr.sendStr("prog" + TEST_PROG, 0);
        acr.sendStr("new prog" + TEST_PROG, 0);
        acr.sendStr("program", 0);
        acr.sendStr("clear", 0);
        acr.sendStr("dim lv(10)", 0);
        acr.sendStr("dim la(5)", 0);
        acr.sendStr("dim la0(100)", 0);
        acr.sendStr("dim la1(100)", 0);
        acr.sendStr("dim la2(100)", 0);
        acr.sendStr("dim la3(100)", 0);
        acr.sendStr("dim la4(100)", 0);
        acr.sendStr("lv1 = 0", 0);
        acr.sendStr("set " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("lv2 = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("for lv0 = 0 to " + (mCount - 1) + " step 1", 0);
        acr.sendStr("if (lv1 = 0)", 0);
        acr.sendStr("intcap axis0 " + AcrComm.ICM_RISE_3RD, 0);
        if (useCapt)
            acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0);
        else
            acr.sendStr("inh " + AcrComm.INPUT4_BIT, 0);
        acr.sendStr("else", 0);
        acr.sendStr("intcap axis0 " + AcrComm.ICM_FALL_3RD, 0);
        if (useCapt)
            acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0);
        else
            acr.sendStr("inh -" + AcrComm.INPUT4_BIT, 0);
        acr.sendStr("endif", 0);
        acr.sendStr("la0(lv0) = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("la3(lv0) = p" + AcrComm.ENC_POSN_PRM, 0);
        acr.sendStr("la1(lv0) = p" + AcrComm.INP_LINE_PRM, 0);
        acr.sendStr("la2(lv0) = p" + AcrComm.PRIM_AXIS_PRM, 0);
        acr.sendStr("la4(lv0) = p" + AcrComm.CAP_POSN_PRM, 0);
        acr.sendStr("lv1 = 1 - lv1", 0);
        acr.sendStr("next", 0);
        acr.sendStr("clr " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("endp", 0);

        /*
        **  Set up periodic digital output using counter interrupts
        */
        hel.cntrConfig(freq, false);
        hel.cntrEnable(this, "genPulse", pulse);

        /*
        **  Start the program and the counter
        */
        acr.sendStr("sys", 0);
        acr.sendStr("clr " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("run prog" + TEST_PROG, 0);
        acr.sendStr("inh " + AcrComm.CAL_SYNC_BIT, 0);
        hel.cntrStart();

        /*
        **  If distance is non-zero, perform the move
        **  Otherwise just wait the specified time
        */
        if (dist != 0f)
            blade.move(BladeSetDrvr.MOV_TYP_SCURVE, dist, time, 0F);
        else
            sleep(time);

        /*
        **  Wait for the program to complete
        */
        acr.sendStr("inh -" + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("intcap axis0 off", 0);
        hel.cntrStop();
        hel.cntrDisable();

        /*
        **  Output the captured encoder positions
        */
        int[] vals = new int[3];
        acr.getVars(TEST_PROG, 0, 3, vals);
        int count = vals[0];
        int[] times = new int[count], inpts = new int[count],
              flags = new int[count], posns = new int[count],
              capts = new int[count];
        acr.getVars(TEST_PROG, 0, 0, count, times);
        acr.getVars(TEST_PROG, 1, 0, count, inpts);
        acr.getVars(TEST_PROG, 2, 0, count, flags);
        acr.getVars(TEST_PROG, 3, 0, count, posns);
        acr.getVars(TEST_PROG, 4, 0, count, capts);
        out.format("Capture count = %s, pulse count = %s\n",
                   count, 2 * pulse.count);
        int sTime = vals[2], pTime = sTime;
        for (int j = 0; j < count; j++) {
            ofl.format("%5s %5s %1s %1s %8.2f %6s %8.2f %6s\n",
                       times[j] - sTime, times[j] - pTime,
                       (inpts[j] >> 4) & 1, (flags[j] >> 9) & 1,
                       config.position(posns[j]), posns[j],
                       config.position(capts[j]), capts[j]);
            pTime = times[j];
        }

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Test the capture of encoder positions using pulses from controller
    **
    ***************************************************************************
    */
    private void calTest2(float dist, float time, String file)
    {
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Set initial state of output line and expected transition
        */
        int outLine = AcrComm.OUTPUT33_BIT;
        acr.clearBit(outLine);
        int capMode = AcrComm.ICM_RISE_3RD;

        /*
        **  Set up interrupt parameters
        */
        Pulse pulse = new Pulse();
        pulse.mCount = 32;
        pulse.iLine = dioILine;
        pulse.oLine = dioOLine;
        pulse.width = time / (2 * pulse.mCount);
        pulse.iState = acd.dioInpBit(dioIPort, pulse.iLine);
        acd.dioClrBit(dioOPort, pulse.oLine);
        acd.attachInt(1 << dioIPort, this, "passDio", pulse);

        /*
        **  Load the pulse/capture program and start it
        */
        acr.sendStr("prog" + TEST_PROG, 0);
        acr.sendStr("new prog" + TEST_PROG, 0);
        acr.sendStr("program", 0);
        acr.sendStr("clear", 0);
        acr.sendStr("dim lv(10)", 0);
        acr.sendStr("dim la(2)", 0);
        acr.sendStr("dim la0(100)", 0);
        acr.sendStr("dim la1(100)", 0);
        acr.sendStr("lv1 = " + capMode, 0);
        acr.sendStr("set " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("lv2 = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("for lv0 = 0 to " + (2 * pulse.mCount - 1) + " step 1", 0);
        acr.sendStr("intcap axis0 (lv1)", 0);
        acr.sendStr("if (lv1 = " + capMode + ") then set " + outLine, 0);
        acr.sendStr("if (lv1 <> " + capMode + ") then clr " + outLine, 0);
        acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0);
        acr.sendStr("la1(lv0) = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("la0(lv0) = p" + AcrComm.CAP_POSN_PRM, 0);
        acr.sendStr("lv1 = " + (AcrComm.ICM_RISE_3RD + AcrComm.ICM_FALL_3RD)
                      + " - lv1", 0);
        acr.sendStr("dwl " + pulse.width, 0);
        acr.sendStr("next", 0);
        acr.sendStr("endp", 0);

        /*
        **  Start the program
        */
        acr.sendStr("sys", 0);
        acr.sendStr("clr " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("run prog" + TEST_PROG, 0);
        acr.sendStr("inh " + AcrComm.CAL_SYNC_BIT, 0);

        /*
        **  If distance is non-zero, perform the move
        **  Otherwise just wait the specified time
        */
        if (dist != 0f)
            blade.move(BladeSetDrvr.MOV_TYP_SCURVE, dist, time, 0F);
        else
            sleep(time);

        /*
        **  Halt the program and stop the capture
        */
        acr.sendStr("halt prog" + TEST_PROG, 0);
        acr.sendStr("intcap axis0 off", 0);
        acd.detachInt();

        /*
        **  Output the captured encoder positions
        */
        int[] vals = new int[3];
        acr.getVars(TEST_PROG, 0, 3, vals);
        int count = vals[0];
        int[] posns = new int[count], times = new int[count];
        acr.getVars(TEST_PROG, 0, 0, count, posns);
        acr.getVars(TEST_PROG, 1, 0, count, times);
        out.format("Capture count = %s, transition count = %s (%s)\n",
                   count, pulse.count,
                   count == pulse.count ? "ok" : "mismatch");
        int pTime = vals[2];
        for (int j = 0; j < count; j++) {
            ofl.format("%5s %5s %8.2f %6s\n", times[j] - vals[2],
                       times[j] - pTime, config.position(posns[j]), posns[j]);
            pTime = times[j];
        }

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Test the capture of encoder positions using pulses from controller
    **
    ***************************************************************************
    */
    private void calTest4(float dist, float time, String file)
    {
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Set initial state of output line and expected transition
        */
        int outLine = AcrComm.OUTPUT33_BIT;
        acr.clearBit(outLine);

        /*
        **  Set up interrupt parameters
        */
        Pulse pulse = new Pulse();
        pulse.mCount = 32;
        pulse.iLine = dioILine;
        pulse.oLine = dioOLine;
        pulse.width = time / (2 * pulse.mCount);
        pulse.iState = acd.dioInpBit(dioIPort, pulse.iLine);
        acd.dioClrBit(dioOPort, pulse.oLine);
        acd.attachInt(1 << dioIPort, this, "passDio", pulse);

        /*
        **  Set up other parameters
        */
        float delay = 0.009f, ohead = 0.005f;

        /*
        **  Load the pulse/capture program and start it
        */
        acr.sendStr("prog" + TEST_PROG, 0);
        acr.sendStr("new prog" + TEST_PROG, 0);
        acr.sendStr("program", 0);
        acr.sendStr("clear", 0);
        acr.sendStr("dim lv(10)", 0);
        acr.sendStr("dim la(2)", 0);
        acr.sendStr("dim la0(100)", 0);
        acr.sendStr("dim la1(100)", 0);
        acr.sendStr("lv1 = 0", 0);
        acr.sendStr("set " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("lv2 = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("for lv0 = 0 to " + (2 * pulse.mCount - 1) + " step 1", 0);
        acr.sendStr("if (lv1 = 0) then set " + outLine, 0);
        acr.sendStr("if (lv1 <> 0) then clr " + outLine, 0);
        acr.sendStr("dwl " + delay, 0);
        acr.sendStr("la1(lv0) = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("la0(lv0) = p4096", 0);
        acr.sendStr("lv1 = 1 - lv1", 0);
        acr.sendStr("dwl " + (pulse.width - delay - ohead), 0);
        acr.sendStr("next", 0);
        acr.sendStr("clr " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("endp", 0);

        /*
        **  Start the program
        */
        acr.sendStr("sys", 0);
        acr.sendStr("clr " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("run prog" + TEST_PROG, 0);
        acr.sendStr("inh " + AcrComm.CAL_SYNC_BIT, 0);

        /*
        **  If distance is non-zero, perform the move
        **  Otherwise just wait the specified time
        */
        if (dist != 0f)
            blade.move(BladeSetDrvr.MOV_TYP_SCURVE, dist, time, 0F);
        else
            sleep(time);

        /*
        **  Wait for the program to complete
        */
        acr.sendStr("inh -" + AcrComm.CAL_SYNC_BIT, 0);
        acd.detachInt();

        /*
        **  Output the recorded times and input line values
        */
        int[] vals = new int[3];
        acr.getVars(TEST_PROG, 0, 3, vals);
        int count = vals[0];
        int[] inpts = new int[count], times = new int[count];
        acr.getVars(TEST_PROG, 0, 0, count, inpts);
        acr.getVars(TEST_PROG, 1, 0, count, times);
        out.format("Capture count = %s, transition count = %s (%s)\n",
                   count, pulse.count,
                   count == pulse.count ? "ok" : "mismatch");
        int pTime = vals[2];
        for (int j = 0; j < count; j++) {
            ofl.format("%5s %5s %08x\n", times[j] - vals[2], times[j] - pTime,
                       inpts[j]);
            pTime = times[j];
        }

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Show the Hall sensor calibration data
    **
    ***************************************************************************
    */
    private void showCalib()
    {
        ArrayList<BladeSetCalibration.Item> cList
            = config.getCalibration().getData();

        out.println("Dirn   Swtch   Index   Action   Position");
        out.println("----   -----   -----   ------   --------");
        for (int j = 0; j < cList.size(); j++) {
            BladeSetCalibration.Item data = cList.get(j);
            out.format("%4s   %5s   %5s   %6s   %8.2f\n",
                       data.isReverse() ? "Back" : "Forw",
                       data.getSensor(), data.getIndex(),
                       data.isOpen() ? "Open " : "Close", data.getPosition());
        }
    }


   /**
    ***************************************************************************
    **
    **  Perform a position operation
    **
    ***************************************************************************
    */
    private void position(float relPosn, float time, String name)
    {
        long tStamp = System.currentTimeMillis();
        String fileH = genFileName(name, "_H.dat", tStamp);
        String fileP = genFileName(name, "_P.dat", tStamp);
        PrintStream oflH = null, oflP = null;

        /*
        **  Open the output files
        */
        try {
            oflH = new PrintStream(fileH);
            oflP = new PrintStream(fileP);
        }
        catch (IOException e) {
            out.println(e);
            if (oflH != null) oflH.close();
            if (oflP != null) oflP.close();
            return;
        }

        /*
        **  Perform the move, capturing position and Hall data
        */
        showMoveStart();
        blade.moveToPosition(relPosn, time);
        showMoveEnd();
        showStop(mhData.getStatus());

        /*
        **  Return if failure
        */
        if (mhData == null) {
            oflH.close();
            oflP.close();
            return;
        }

        /*
        **  Output the sampled position and Hall sensor values
        */
        writePosnData(mhData, oflP);
        writeHallData(mhData, oflH);

        /*
        **  Close the output files
        */
        oflH.close();
        oflP.close();
        out.println("Files " + fileH + " and " + fileP + " created");
    }


   /**
    ***************************************************************************
    **
    **  Receives the final movement history data
    **
    **  <p>
    **  This routine is called synchronously by the moveToPosition method.
    **
    ***************************************************************************
    */
    @Override
    public void movementHistoryFinalized(MovementHistory mh)
    {
        mhData = mh;
    }


   /**
    ***************************************************************************
    **
    **  Receives the updated movement history data
    **
    **  <p>
    **  This routine is not called.
    **
    ***************************************************************************
    */
    @Override
    public void movementHistoryUpdated(MovementHistory mh)
    {
    }


   /**
    ***************************************************************************
    **
    **  Perform a move
    **
    ***************************************************************************
    */
    private void move(int type, float dist, float time, float mvFract)
    {
        showMoveStart();
        int status = blade.move(type, dist, time, mvFract);
        showMoveEnd();
        showStop(status);
    }


   /**
    ***************************************************************************
    **
    **  Perform a move, recording various parameters periodically
    **
    ***************************************************************************
    */
    private void moveD(int type, float dist, float time, float mvFract,
                      int nSamp, String name)
    {
        String file = genFileName(name, "_D.dat");
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Set the parameters to be sampled and perform the move
        */
        int[] iParms = {AcrComm.SYS_CLOCK_PRM, AcrComm.ENC_POSN_PRM,
                        AcrComm.PRIM_AXIS_PRM};
        int[] fParms = {AcrComm.CMDD_CURR_PRM, AcrComm.CMDD_TORQ_PRM,
                        AcrComm.ACTL_TORQ_PRM, AcrComm.BUS_VOLT_PRM};
        showMoveStart();
        BladeSetDrvr.SampData data = blade.moveD(type, dist, time, mvFract,
                                                 nSamp, iParms, fParms);
        showMoveEnd();
        showStop(data.status);

        /*
        **  Return if error
        */
        if (data == null) {
            ofl.close();
            return;
        }

        /*
        **  Output the sampled values
        */
        for (int j = 0; j < data.sList.size(); j++) {
            BladeSetDrvr.Sample samp = data.sList.get(j);
            ofl.format("%12s %5s %8.2f %12s %12s %12s %12s %08x\n",
                       data.startTime + samp.time / 1000,
                       samp.time / 1000,
                       config.position(samp.iVals[1]), samp.fVals[0],
                       samp.fVals[1], samp.fVals[2], samp.fVals[3],
                       samp.iVals[2]);
            if ((samp.iVals[2] & 0x01000000) == 0) break;
        }

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Perform a move, recording both Hall switch transitions and periodic
    **  position values
    **
    ***************************************************************************
    */
    private void moveH(int type, float dist, float time, float mvFract,
                      int nSamp, String name)
    {
        long tStamp = System.currentTimeMillis();
        String fileH = genFileName(name, "_H.dat", tStamp);
        String fileP = genFileName(name, "_P.dat", tStamp);
        PrintStream oflH = null, oflP = null;

        /*
        **  Open the output files
        */
        try {
            oflH = new PrintStream(fileH);
            oflP = new PrintStream(fileP);
        }
        catch (IOException e) {
            out.println(e);
            if (oflH != null) oflH.close();
            if (oflP != null) oflP.close();
            return;
        }

        /*
        **  Perform the move, capturing position and Hall data
        */
        showMoveStart();
        MovementHistory data = blade.moveH(type, dist, time, mvFract, nSamp);
        showMoveEnd();
        showStop(data.getStatus());

        /*
        **  Return if failure
        */
        if (data == null) {
            oflH.close();
            oflP.close();
            return;
        }

        /*
        **  Output the sampled position and Hall sensor values
        */
        writePosnData(data, oflP);
        writeHallData(data, oflH);

        /*
        **  Close the output files
        */
        oflH.close();
        oflP.close();
        out.println("Files " + fileH + " and " + fileP + " created");
    }


   /**
    ***************************************************************************
    **
    **  Perform a move, recording the position periodically
    **
    ***************************************************************************
    */
    private void moveP(int type, float dist, float time, float mvFract,
                      int nSamp, String name)
    {
        String file = genFileName(name, "_P.dat");
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Perform the move, capturing position data
        */
        showMoveStart();
        MovementHistory data = blade.moveP(type, dist, time, mvFract, nSamp);
        showMoveEnd(data.getEndPosition(), data.getEndTime() / 1000);
        showStop(data.getStatus());

        /*
        **  Return if error
        */
        if (data == null) {
            ofl.close();
            return;
        }

        /*
        **  Output the sampled position values
        */
        writePosnData(data, ofl);

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Perform a move, recording test values periodically
    **
    ***************************************************************************
    */
    private void moveX(int type, float dist, float time, float mvFract,
                       int nSamp, String name)
    {
        String file = genFileName(name, "_X.dat");
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Set the parameters used by the move program and run it
        */
        int[] iParms = {AcrComm.ENC_POSN_PRM, AcrComm.INP_LINE_PRM},
              fParms = {};
        showMoveStart();
        BladeSetDrvr.SampData data = blade.moveD(type, dist, time, mvFract,
                                                 nSamp, iParms, fParms);
        showMoveEnd();
        showStop(data.status);

        /*
        **  Return if error
        */
        if (data == null) {
            ofl.close();
            return;
        }

        /*
        **  Output the sampled values
        */
        for (int j = 0; j < data.sList.size(); j++) {
            BladeSetDrvr.Sample samp = data.sList.get(j);
            ofl.format("%8.2f %08x\n",
                       config.position(samp.iVals[0]), samp.iVals[1]);
        }

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Repeatedly perform a back and forth move cycle
    **
    ***************************************************************************
    */
    private void moveTest(int count, float dist, float time)
    {
        final int type = BladeSetDrvr.MOV_TYP_SCURVE;
        final float mxvf = 0f;

        /*
        **  Make sure distance is positive, etc
        */
        if (dist < 0) dist = -dist;
        if (dist == 0) dist = 10;

        /*
        **  Loop over the cycle count, moving out and back
        */
        showMoveStart();
        int si, status = 0;
        for (si = 0; si < count; si++) {
            if ((status = blade.move(type, dist, time, mxvf)) != 0) break;
            if ((status = blade.move(type, -dist, time, mxvf)) != 0) break;
        }
        showMoveEnd();

        /*
        **  Output termination message
        */
        if (status != 0)
            out.format("%s after %s cycles\n", getStop(status), si);
        else
            out.format("%s cycles completed successfully\n", si);
    }


   /**
    ***************************************************************************
    **
    **  Repeatedly perform a back and forth move cycle, recording Hall switch
    **  transition times
    **
    ***************************************************************************
    */
    private void hallTest(int count, float dist, float time, String name)
    {
        /*
        **  Get calibration data
        */
        ArrayList<BladeSetCalibration.Item> cList
            = config.getCalibration().getData();

        /*
        **  Open the output file
        */
        String file = genFileName(name, ".dat");
        PrintStream ofl;
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Initialize storage for Hall times
        */
        if (count <= 0) count = 1;
        int[][] times = new int[cList.size()][count];

        /*
        **  Make sure distance is positive; move to home
        */
        if (dist < 0) dist = -dist;
        int type = BladeSetDrvr.MOV_TYP_SCURVE;
        float mxvf = 0f;
        int nSamp = config.getNumSamp();
        float posn = blade.getPosition();
        if (posn > 1f || posn < -1f)
            blade.move(type, -posn, 1f, mxvf);

        /*
        **  Loop over the cycle count
        */
        for (int si = 0; si < count; si++) {

            /*
            **  Home the shutter blade set, move out, move back
            */
            blade.move(type, -10f, 1f, mxvf);
            blade.home(0, 10f);
            MovementHistory data0 = blade.moveH(type, -dist, time, mxvf, nSamp);
            if (data0 == null) break;
            MovementHistory data1 = blade.moveH(type, dist, time, mxvf, nSamp);
            if (data1 == null) break;

            /*
            **  Copy switch times to array
            */
            int ci = 0;
            List<HallTransition> hList = data0.getHallTransitions();
            long sTime = data0.getStartTime();
            for (int j = 0; j < hList.size(); j++) {
                HallTransition hItem = (HallTransition)hList.get(j);
                BladeSetCalibration.Item cItem = cList.get(ci);
                boolean open = hItem.isOpen();
                int sensor = hItem.getSensorId();
                if (!cItem.isReverse() || cItem.getSensor() != sensor
                      || cItem.isOpen() != open) {
                    out.format("Inconsistent data: %s true %s %s %s %s, "
                                 + "index = %s, cycle = %s\n",
                               cItem.isReverse(), cItem.getSensor(), sensor,
                               cItem.isOpen(), open, ci, si);
                    continue;
                }
                times[ci++][si]
                  = (int)((hItem.getTransitionTime() - sTime + 50) / 100);
            }
            hList = data1.getHallTransitions();
            sTime = data1.getStartTime();
            for (int j = 0; j < hList.size(); j++) {
                HallTransition hItem = (HallTransition)hList.get(j);
                BladeSetCalibration.Item cItem = cList.get(ci);
                boolean open = hItem.isOpen();
                int sensor = hItem.getSensorId();
                if (cItem.isReverse() || cItem.getSensor() != sensor
                      || cItem.isOpen() != open) {
                    out.format("Inconsistent data: %s false %s %s %s %s, "
                                 + "index = %s, cycle = %s\n",
                               cItem.isReverse(), cItem.getSensor(), sensor,
                               cItem.isOpen(), open, ci, si);
                    continue;
                }
                times[ci++][si]
                  = (int)((hItem.getTransitionTime() - sTime + 50) / 100);
            }
        }


        /*
        **  Output the switch times
        */
        for (int ci = 0; ci < cList.size(); ci++) {
            BladeSetCalibration.Item cItem = cList.get(ci);
            ofl.format("%s %s %s %8.2f", cItem.getSensor(),
                       cItem.isOpen() ? "O" : "C",
                       cItem.isReverse() ? "-" : "+", cItem.getPosition());
            for (int si = 0; si < count; si++)
                ofl.format(" %5s", times[ci][si]);
            ofl.println();
        }

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Repeatedly perform a back and forth move cycle, recording position
    **  profiles
    **
    ***************************************************************************
    */
    private void encTest(int count, float dist, float time, int nSamp,
                         String name)
    {
        /*
        **  Open the output file
        */
        String file = genFileName(name, ".dat");
        PrintStream ofl;
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Initialize storage for position data
        */
        if (count <= 0) count = 1;
        int[][] times = new int[2 * nSamp][count];
        float[][] posns = new float[2 * nSamp][count];

        /*
        **  Make sure distance is positive; move to home
        */
        if (dist < 0) dist = -dist;
        int type = BladeSetDrvr.MOV_TYP_SCURVE;
        float mxvf = 0f;
        float posn = blade.getPosition();
        if (posn > 1f || posn < -1f)
            blade.move(type, -posn, 1f, mxvf);

        /*
        **  Loop over the cycle count
        */
        for (int si = 0; si < count; si++) {

            /*
            **  Move out and back
            */
            MovementHistory data0 = blade.moveP(type, -dist, time, mxvf, nSamp);
            if (data0 == null) break;
            MovementHistory data1 = blade.moveP(type, dist, time, mxvf, nSamp);
            if (data1 == null) break;

            /*
            **  Copy sample profile to arrays
            */
            int ci = 0;
            List<BladePosition> pList = data0.getMovementProfile();
            long sTime = data0.getStartTime();
            int size = pList.size();
            BladePosition pItem, pDummy = new BladePositionImpl(-1, -1f);
            for (int j = 0; j < nSamp; j++) {
                if (j >= size) pItem = pDummy;
                else pItem = pList.get(j);
                times[ci][si] = (int)((pItem.getTime() - sTime + 50) / 100);
                posns[ci++][si] = pItem.getPosition();
            }
            pList = data1.getMovementProfile();
            sTime = data1.getStartTime();
            size = pList.size();
            for (int j = 0; j < nSamp; j++) {
                if (j >= size) pItem = pDummy;
                else pItem = pList.get(j);
                times[ci][si] = (int)((pItem.getTime() - sTime + 50) / 100);
                posns[ci++][si] = pItem.getPosition();
            }
        }


        /*
        **  Output the position profile data
        */
        for (int ci = 0; ci < 2 * nSamp; ci++) {
            ofl.format("%s", ci < nSamp ? "-" : "+");
            for (int si = 0; si < count; si++)
                ofl.format(" %5s %8.2f", times[ci][si], posns[ci][si]);
            ofl.println();
        }

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }


   /**
    ***************************************************************************
    **
    **  Get step accuracy statistics
    **
    ***************************************************************************
    */
    private void step(int count, int step, float time)
    {
        /*
        **  Configure and enable the drive
        */
        float vel = Math.abs(2F * step / time);
        float acc = Math.abs(8F * step / (time * time));
        float jrk = acc * acc / vel;
        acr.setFloatParm(AcrComm.PPU_PRM, 1F);
        acr.setFloatParm(AcrComm.JOG_VEL_PRM, vel);
        acr.setFloatParm(AcrComm.JOG_ACC_PRM, acc);
        acr.setFloatParm(AcrComm.JOG_DEC_PRM, acc);
        acr.setFloatParm(AcrComm.JOG_JRK_PRM, jrk);
        acr.sendStr("axis0 slm (" + config.posSoftLimit() * config.ppu() + ","
                      + config.negSoftLimit() * config.ppu() + ")", 0);
        blade.enableDrive();

        /*
        **  Loop to do steps and read encoder positions
        */
        int nStep, minm = Integer.MAX_VALUE, maxm = Integer.MIN_VALUE,
            pPosn, cPosn, fPosn, sumSq = 0;
        pPosn = fPosn = acr.getIntParm(AcrComm.ENC_POSN_PRM);
        for (nStep = 0; nStep < count; nStep++) {
            acr.sendStr("axis0 jog inc " + step, 0);
            acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
            if (showStop() != null) break;
            cPosn = acr.getIntParm(AcrComm.ENC_POSN_PRM);
            int cStep = cPosn - pPosn;
            if (cStep > maxm) maxm = cStep;
            if (cStep < minm) minm = cStep;
            sumSq += cStep * cStep;
            pPosn = cPosn;
        }

        /*
        **  Calculate and display results
        */
        double avg = (double)(pPosn - fPosn) / (nStep == 0 ? 1 : nStep);
        double var = (double)sumSq / (nStep == 0 ? 1 : nStep) - avg * avg;
        out.format("Steps = %s, counts = %s, min = %s, max = %s, "
                     + "avg = %.2f, sigma = %.4f\n",
                   nStep, pPosn - fPosn, minm == Integer.MAX_VALUE ? 0 : minm,
                   maxm == Integer.MIN_VALUE ? 0 : maxm, avg, Math.sqrt(var));

        /*
        **  Reset PPU & soft limits, and disable the drive
        */
        acr.setFloatParm(AcrComm.PPU_PRM, config.ppu());
        acr.sendStr("axis0 slm (" + config.posSoftLimit() + ","
                      + config.negSoftLimit() + ")", 0);
        blade.disableDrive();
    }


   /**
    ***************************************************************************
    **
    **  Time the synchronization between computer and motor controller
    **
    ***************************************************************************
    */
    private void timeSync(boolean useCapt, int count)
    {
        /*
        **  Set initial states of ACR output & input lines
        */
        acr.clearBit(AcrComm.OUTPUT33_BIT);
        acd.dioClrBit(dioOPort, dioOLine);

        /*
        **  Set up routine to respond to ACR line transitions
        */
        BladeSetDrvr.Hall hall = new BladeSetDrvr.Hall(2 * count);
        hall.state = acd.dioInpBit(dioIPort, dioILine);
        acd.attachInt(1 << dioIPort, this, "readACR", hall);

        /*
        **  Load the synch program, start it, and wait for completion
        */
        if ((count & 1) != 0) count++;
        acr.sendStr("prog" + TEST_PROG, 0);
        acr.sendStr("new prog" + TEST_PROG, 0);
        acr.sendStr("program", 0);
        acr.sendStr("clear", 0);
        acr.sendStr("dim lv(10)", 0);
        acr.sendStr("dim la(1)", 0);
        acr.sendStr("dim la0(" + count + ")", 0);
        acr.sendStr("lv0 = 0", 0);
        acr.sendStr("set " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("while (lv0 < " + count + ")", 0);
        acr.sendStr("intcap axis0 " + AcrComm.ICM_RISE_3RD, 0);
        acr.sendStr("set " + AcrComm.OUTPUT33_BIT, 0);
        if (useCapt)
            acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0);
        else
            acr.sendStr("inh " + AcrComm.INPUT4_BIT, 0);
        acr.sendStr("la0(lv0) = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("lv0 = lv0 + 1", 0);
        acr.sendStr("intcap axis0 " + AcrComm.ICM_FALL_3RD, 0);
        acr.sendStr("clr " + AcrComm.OUTPUT33_BIT, 0);
        if (useCapt)
            acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0);
        else
            acr.sendStr("inh -" + AcrComm.INPUT4_BIT, 0);
        acr.sendStr("la0(lv0) = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("lv0 = lv0 + 1", 0);
        acr.sendStr("wend", 0);
        acr.sendStr("clr " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("endp", 0);

        acr.sendStr("sys", 0);
        acr.clearBit(AcrComm.CAL_SYNC_BIT);
        acr.sendStr("run prog" + TEST_PROG, 0);
        acr.sendStr("inh " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("inh -" + AcrComm.CAL_SYNC_BIT, 0);

        /*
        **  Disable DIO line transition interrupts
        */
        acd.detachInt();

        /*
        **  Output the captured encoder positions
        */
        int[] vals = new int[1];
        acr.getVars(TEST_PROG, 0, 1, vals);
        int cCount = vals[0];
        int[] times = new int[cCount];
        acr.getVars(TEST_PROG, 0, 0, vals[0], times);
        out.format("Counts: requested = %s, controller = %s, computer = %s\n",
                   count, cCount, hall.count);
        int pCTime = times[0];
        long pHTime = hall.data[0].time;
        count = (cCount > hall.count) ? cCount : hall.count;
        for (int j = 0; j < count; j++) {
            if (j < cCount) {
                out.format("%4s", times[j] - pCTime);
                pCTime = times[j];
            }
            else
                out.format("    ");
            if (j < hall.count) {
                BladeSetDrvr.HallItem item = hall.data[j];
                long hTime = item.time;
                out.format("  %4s  %s", (hTime - pHTime) / 1000000,
                           item.open ? "O" : "C");
                pHTime = hTime;
            }
            out.println();
        }
    }


   /**
    ***************************************************************************
    **
    **  Show temperatures
    **
    ***************************************************************************
    */
    private void showTemps()
    {
        out.format("Drive temperature: %.2f, Ambient temperature: %.2f\n",
                   blade.getDriveTemperature(), blade.getAmbientTemperature());
    }


   /**
    ***************************************************************************
    **
    **  Write a file of sampled position data
    **
    ***************************************************************************
    */
    private void writePosnData(MovementHistory data, PrintStream ofl)
    {
        long sTime = data.getStartTime() / 1000;
        List<BladePosition> pList = data.getMovementProfile();
        for (int j = 0; j < pList.size(); j++) {
            BladePosition pItem = pList.get(j);
            long pTime = pItem.getTime() / 1000;
            ofl.format("%12s %5s %8.2f %8.5f\n", pTime, pTime - sTime,
                       pItem.getPosition(), pItem.getRelPosition());
        }
    }


   /**
    ***************************************************************************
    **
    **  Write a file of Hall sensor data
    **
    ***************************************************************************
    */
    private void writeHallData(MovementHistory data, PrintStream ofl)
    {
        long sTime = data.getStartTime() / 1000;
        List<HallTransition> hList = data.getHallTransitions();
        for (int j = 0; j < hList.size(); j++) {
            HallTransition hItem = (HallTransition)hList.get(j);
            long hTime = (hItem.getTransitionTime() + 500) / 1000;
            ofl.format("%12s %5s %s %s %s %8.2f %8.5f\n",
                       hTime, hTime - sTime, hItem.getSensorId(),
                       hItem.isOpen() ? "O" : "C",
                       hItem.isReverse() ? "-" : "+",
                       hItem.getPosition(), hItem.getRelPosition());
        }
    }


   /**
    ***************************************************************************
    **
    **  Generate a file name containing blade index and time stamp
    **
    ***************************************************************************
    */
    private String genFileName(String head, String tail, long tStamp)
    {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(tStamp);
        return head + bladeIndex + "_"
                 + String.format("%tY%<tm%<td_%<tH%<tM%<tS", cal) + tail;
    }


    private String genFileName(String head, String tail)
    {
        return genFileName(head, tail, System.currentTimeMillis());
    }


   /**
    ***************************************************************************
    **
    **  Show the move start position and time
    **
    ***************************************************************************
    */
    private void showMoveStart(float posn, long time)
    {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(time);
        out.format("Started move: posn = %8.2f, time = %tF %<tT.%<tL\n",
                   posn, cal);
    }


    private void showMoveStart()
    {
        showMoveStart(blade.getPosition(), System.currentTimeMillis());
    }


   /**
    ***************************************************************************
    **
    **  Show the move end position and time
    **
    ***************************************************************************
    */
    private void showMoveEnd(float posn, long time)
    {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(time);
        out.format("Ended move  : posn = %8.2f, time = %tF %<tT.%<tL\n",
                   posn, cal);
    }


    private void showMoveEnd()
    {
        showMoveEnd(blade.getPosition(), System.currentTimeMillis());
    }


   /**
    ***************************************************************************
    **
    **  Show the current shutter blade set position
    **
    ***************************************************************************
    */
    private void showPosition()
    {
        int posn = acr.getIntParm(AcrComm.ENC_POSN_PRM);
        out.format("Position = %.2f mm (encoder = %s)\n",
                   config.position(posn), posn);
    }


   /**
    ***************************************************************************
    **
    **  Show the current shutter blade set relative position
    **
    ***************************************************************************
    */
    private void showRelPosition()
    {
        out.format("Relative position = %.5f\n", blade.getCurrentPosition());
    }


   /**
    ***************************************************************************
    **
    **  Show the reason for a premature stop, if any
    **
    ***************************************************************************
    */
    private String showStop(int status)
    {
        String stopped = getStop(status);
        if (stopped != null) out.println(stopped);

        return stopped;
    }

    private String showStop()
    {
        return showStop(blade.moveStatus());
    }


   /**
    ***************************************************************************
    **
    **  Get message text for a premature movement stop
    **
    ***************************************************************************
    */
    private String getStop(int status)
    {
        if (status == 0) return null;
        else return MovementStatus.message(status);
    }

    private String getStop()
    {
        return getStop(blade.moveStatus());
    }


   /**
    ***************************************************************************
    **
    **  Generate a pulse on a digital output line
    **
    ***************************************************************************
    */
    private void genPulse(int value, Object parm)
    {
        Pulse pulse = (Pulse)parm;
        if (pulse.count >= pulse.mCount) return;
        acd.dioSetBit(dioOPort, pulse.oLine);
        sleep(pulse.width);
        acd.dioClrBit(dioOPort, pulse.oLine);
        pulse.count++;
    }


   /**
    ***************************************************************************
    **
    **  Pass a transition from digital input to digital output
    **
    ***************************************************************************
    */
    private void passDio(int value, Object parm)
    {
        Pulse pulse = (Pulse)parm;
        int state = acd.dioInpBit(dioIPort, pulse.iLine);
        if (state == pulse.iState) return;
        pulse.iState = state;
        pulse.count++;
        if (state != 0) acd.dioClrBit(dioOPort, pulse.oLine);
        else acd.dioSetBit(dioOPort, pulse.oLine);
    }


   /**
    ***************************************************************************
    **
    **  Read the ACR line and respond to changes
    **
    **  This routine is called whenever the input line changes
    **
    ***************************************************************************
    */
    private void readACR(int value, Object parm)
    {
        BladeSetDrvr.Hall hall = (BladeSetDrvr.Hall)parm;
        int state = acd.dioInpBit(dioIPort, dioILine);
        if (state == hall.state) return;

        hall.state = state;
        boolean open = state != 0;
        if (!open) acd.dioSetBit(dioOPort, dioOLine);
        else acd.dioClrBit(dioOPort, dioOLine);

        if (hall.count >= hall.data.length) return;
        BladeSetDrvr.HallItem item = hall.data[hall.count++];
        item.time = System.nanoTime();
        item.open = open;
    }


   /**
    ***************************************************************************
    **
    **  Sleep for a time
    **
    ***************************************************************************
    */
    private void sleep(float secs)
    {
        long nsecs = (long)(1e9F * secs);
        try {
            Thread.sleep(nsecs / 1000000, (int)(nsecs % 1000000));
        }
        catch (InterruptedException e) {
        }
    }

}
